/*
 * The MIT License
 *
 * Copyright 2015 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.cli;

import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.cli.spi.CommandsPackage;
import xyz.cofe.collection.Pointer;
import xyz.cofe.fs.File;
import xyz.cofe.fs.FileSystems;
import xyz.cofe.text.Output;
import xyz.cofe.text.Text;
import xyz.cofe.text.parser.Token;
import xyz.cofe.text.template.BasicTemplate;

import static xyz.cofe.text.template.BasicTemplate.template;

/**
 * Интерпретирующая машина
 * @author nt.gocha@gmail.com
 */
public class CommandLineMachine {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(CommandLineMachine.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(CommandLineMachine.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(CommandLineMachine.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(CommandLineMachine.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(CommandLineMachine.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(CommandLineMachine.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(CommandLineMachine.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    public CommandLineMachine(){
        String dump_cli_var = System.getenv("DUMP_CLI");
        if( dump_cli_var!=null ){
            this.dump = dump_cli_var.equalsIgnoreCase("true");
        }

        loadCommands();
    }
    
    private final WeakHashMap setOutputCommands = new WeakHashMap();
    
    public void loadCommands(){
        registerCommands(getMemory(), getOutput(), this, this);
        for( CommandsPackage cp : ServiceLoader.load(CommandsPackage.class) ){
            if( cp==null )continue;
            loadCommands(cp);
            if( cp instanceof SetOutput ){
                setOutputCommands.put(cp, true);
            }
        }
    }

    public static void loadCommands( Memory memory, Output output, CommandLineMachine cli ){
        if( memory==null )throw new IllegalArgumentException( "memory==null" );
        for( CommandsPackage cp : ServiceLoader.load(CommandsPackage.class) ){
            if( cp==null )continue;
            registerCommands(memory, output, cli, cp);
        }
    }

    public static void loadCommands( Memory memory, Output output ){
        if( memory==null )throw new IllegalArgumentException( "memory==null" );
        for( CommandsPackage cp : ServiceLoader.load(CommandsPackage.class) ){
            if( cp==null )continue;
            registerCommands(memory, output, null, cp);
        }
    }

    public void loadCommands(Object commands){
        if( commands==null )return;
        registerCommands(getMemory(), getOutput(), this, commands);
    }

    public static void registerCommands( Memory memory, Output output, CommandLineMachine cli, Object commands){
        if( memory==null )throw new IllegalArgumentException( "memory==null" );
        if( commands instanceof SetOutput && output!=null ){
            ((SetOutput)commands).setOutput(output);
        }
        if( commands!=null ){
            memory.inspect(commands, cli);
        }
    }

    //<editor-fold defaultstate="collapsed" desc="output">
    protected Output output;

    public Output getOutput(){
        if( output!=null )return output;
        output = new Output(System.out, true);
        return output;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="memory">
    protected Memory memory;

    public Memory getMemory() {
        if( memory!=null )return memory;
        memory = new Memory();
        return memory;
    }

    public void setMemory(Memory memory) {
        this.memory = memory;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="dump">
    private boolean dump = false;

    public boolean isDump() {
        return dump;
    }

    public void setDump(boolean dump) {
        this.dump = dump;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="dumpOutput">
    private Output dumpOutput;

    public Output getDumpOutput() {
        if( dumpOutput!=null )return dumpOutput;
        dumpOutput = new Output(System.out);
        return dumpOutput;
    }

    public void setDumpOutput(Output dumpOutput) {
//        if( dumpOutput==null ){
//            this.dumpOutput = new PrintWriter(System.out);
//        }else{
//            this.dumpOutput = new PrintWriter(dumpOutput);
//        }
        this.dumpOutput = dumpOutput;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="createParser()">
    public Parser createParser(){
        return createParser(null);
    }
    
    public Parser createParser(Boolean trace){
        Parser parser = null;
        
        if( trace==null ){
            String traceEnvVar = System.getenv("TRACE_CLI");
            if( traceEnvVar!=null && traceEnvVar.equalsIgnoreCase("true") ){
                trace = true;
            }else{
                String tr = System.getProperty("xyz.cofe.cli.CommandLineMachine.trace");
                if( tr!=null && tr.equalsIgnoreCase("true") ){
                    trace = true;
                }
            }
        }
        
        if( trace!=null && trace ){
            TraceParser tparser = new TraceParser();
            parser = tparser;

            Output tout = null;

            String traceOut = System.getenv("TRACE_CLI_OUT");
            if( traceOut!=null ){
                File traceFile = FileSystems.get(traceOut);
                File tdir = traceFile.getParent();
                if( !tdir.isExists() )tdir.mkdirs();

                OutputStream outstrm = traceFile.openWrite();
                OutputStreamWriter outw = new OutputStreamWriter(outstrm);
                tout = new Output(outw, true);
            }else{
                tout = new Output(System.out, true);
            }

            tparser.getTraceOptions().setOutput(tout);

            String methods = System.getenv("TRACE_CLI_METHODS");
            if( methods!=null ){
                String[] methods_arr = methods.split(",");
                tparser.getTraceOptions().getMethods().clear();
                tparser.getTraceOptions().getMethods().addAll(Arrays.asList(methods_arr));
            }

            String exclude_methods = System.getenv("TRACE_CLI_METHODS_EXCLUDE");
            if( exclude_methods!=null ){
                String[] methods_arr = exclude_methods.split(",");
                for( String m : methods_arr ){
                    tparser.getTraceOptions().getMethods().remove(m);
                }
            }
        }
        
        if( parser==null ){
            parser = new Parser();
        }
        
        parser.setMemory(getMemory());
        
        return parser;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="createLexer()">
    public Lexer createLexer(){
        Lexer lexer = new Lexer();
        if( dump ){
            lexer.setDumpOutput(getDumpOutput());
        }
        return lexer;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="start(String[] args)">
    //<editor-fold defaultstate="collapsed" desc="start(String[])">
    public Object start(String[] args){
        if( args==null )throw new IllegalArgumentException( "args==null" );
        return start(createLexer().parse(args));
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="start(String)">
    public Object start(String script){
        if( script==null )throw new IllegalArgumentException( "script==null" );
        Lexer lexer = createLexer();
        lexer.setSkipWhitespace(true);
        
        Pointer ptr = lexer.parse(script);
        return start(ptr);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="start(Pointer)">
    public Object start(Pointer<Token> toks){
        if( toks==null )throw new IllegalArgumentException( "toks==null" );

        Parser parser = createParser();
        parser.setMemory(getMemory());

        Object res = null;

        Value v = parser.parse( toks );
        Throwable evalErr = null;

        if( v!=null ){

            try{
                res = v.eval();
            }catch(Throwable err){
                evalErr = err;
            }

            if( isDump() ){
                int i=-1;
//                for( Token tok : toks.getList() ){
//                    String arg = tok.getMatchedText();
//                    i++;
//                    getDumpOutput().println(
//                        Text.indent(arg,
//                                    template("[dump arg ${i:>7}] ")
//                                        .bind("i", i)
//                                        .align()
//                                        .eval()
//                        ));
//                    getDumpOutput().flush();
//                }

                StringWriter sw = new StringWriter();

                SourceDump sdump = new SourceDump();
                sdump.setMemory(parser.getMemory());
                sdump.output(sw);
                sdump.dump(v);

                getDumpOutput().println(
                    Text.indent(
                        sw.toString().trim(),
                        "[dump source tree] ") );
                getDumpOutput().flush();

                if( res!=null ){
                    getDumpOutput().println(
                        Text.indent(
                            new Declaration().getTypeName(res.getClass()),
                            "[dump result type] "
                        )
                    );

                    getDumpOutput().println(
                        Text.indent(
                            res.toString().trim(),
                            "[dump result     ] "
                        )
                    );
                    getDumpOutput().flush();
                }else{
                    getDumpOutput().println(
                        "[dump result     ] null");
                    getDumpOutput().flush();
                }
            }
        }

        if( evalErr!=null ){
            if( evalErr instanceof Error ){
                throw (Error)evalErr;
            }else{
                throw new Error(evalErr);
            }
        }
        return res;
    }
//</editor-fold>
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="shell()">
    public InteractiveShell shell(){
        InteractiveShell shell = new InteractiveShell(this);
        getOutput().reset(shell.getOutputWriter());
        if( dump ){
            getDumpOutput().reset(shell.createConsoleWriter("   lexer> "));
        }
        return shell;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getHelp">
    public String getHelp(){
//        inspect(getMemory(), this);
        return generateHelp();
    }

    private Iterable<Function> getCommands(){
        FunctionSetHelper fshlp = new FunctionSetHelper(getMemory());
        fshlp =
            new FunctionSetHelper(
                fshlp.filter(
                    fshlp.functions() ,
                    fshlp.not( fshlp.in( fshlp.operators().functions() ) )
                )
            );
        return fshlp.functions();
    }

    private Map<String,Set<Function>> getCommandsMap(){
        LinkedHashMap<String,Set<Function>> m = new LinkedHashMap<String, Set<Function>>();
        for( Function f : getCommands() ){
            String name = getMemory().getNameOf(f);
            if( name!=null ){
                Set<Function> sf = m.get(name);
                if( sf==null ){
                    sf = new LinkedHashSet<Function>();
                    m.put(name, sf);
                }
                sf.add(f);
            }
        }
        return m;
    }

    private Map<Class,Set<Function>> getOperatorsMap(){
        LinkedHashMap<Class,Set<Function>> m = new LinkedHashMap<Class, Set<Function>>();

        FunctionSetHelper fshlp = new FunctionSetHelper(getMemory());
        fshlp = fshlp.operators();

        for( Function fOP : fshlp.functions() ){
            Class[] opParams = fOP.getParameters();
            if( opParams.length>0 ){
                Set<Function> sf = m.get(opParams[0]);
                if( sf==null ){
                    sf = new LinkedHashSet<Function>();
                    m.put(opParams[0], sf);
                }
                sf.add(fOP);
            }
        }

        return m;
    }

    private String generateHelp(){
        String commands = generateHelpOfCommands();
        String operators = generateHelpOfOperators();

        String help =
            template(
                "Commands:\n"+
                    "${commands}\n"+
                    "\n"+
                    "Operators:\n"+
                    "${operators}"
            )
                .align()
//            .useJavaScript()
                .bind("commands", commands)
                .bind("operators", operators)
                .eval();

        return help;
    }

    private String generateHelpOfOperators(){
        String fundescjoin = "\n\n";
        StringBuilder sb = new StringBuilder();
        Declaration decl = new Declaration(new Memory());

        for( Map.Entry<Class,Set<Function>> me : getOperatorsMap().entrySet() ){
            Class cls = me.getKey();
            sb.append("  ");
            sb.append(decl.getTypeName(cls));
            sb.append("\n");

            int i = -1;
            for( Function op : me.getValue() ){
                if( op==null )continue;

                i++;
                String help = generateHelpOfOperator(op);
                if( help!=null ){
                    if( i>0 )sb.append("\n");
                    sb.append(help);
                }
            }

            sb.append(fundescjoin);
        }

        return sb.toString();
    }

    private String generateHelpOfOperator( Function operator ){
        Declaration declr = new Declaration(getMemory());
        LinkedHashMap help = new LinkedHashMap();
        StringBuilder stmpl = new StringBuilder();

        String name = getMemory().getNameOf(operator);
        name = Text.trimStart(name, getMemory().getOperatorPrefix());
        
        String shortDesc = null;
        String longDesc = null;
        String retHelp = null;

        help.put("name", name);
        help.put("declare", declr.getDeclareOf(operator));

        if( operator instanceof GetHelp ){
            GetHelp gh = (GetHelp)operator;

            shortDesc = gh.getShortHelp();
            help.put("short", shortDesc);

//            stmpl.append("  ${help.name}");
//            if( shortDesc!=null && shortDesc.length()>0 ){
//                stmpl.append(" -  ${help.short:50}");
//            }
//            stmpl.append("\n");

            stmpl.append("    ${help.declare}").append("\n");
            if( shortDesc!=null && shortDesc.length()>0 ){
                stmpl.append("      ${help.short:60}");
                stmpl.append("\n");
            }

            Class[] paramTypes = operator.getParameters();
            if( paramTypes!=null && paramTypes.length>0 ){
                StringBuilder sptmpl = new StringBuilder();
                for( int pi=0; pi<paramTypes.length; pi++ ){
                    String paramHelp = gh.getParameterHelp(pi);
                    if( paramHelp!=null && paramHelp.length()>0 ){
                        help.put("param"+pi+"name", declr.getParamNameOf(operator, pi));
                        help.put("param"+pi+"help", paramHelp);
                        help.put("param"+pi+"type", declr.getTypeName(paramTypes[pi]));

                        sptmpl.append("        ${help.param"+pi+"name}");
                        sptmpl.append(" : ${help.param"+pi+"type}");
                        sptmpl.append(" - ${help.param"+pi+"help:60}");
                        sptmpl.append("\n");
                    }
                }

                if( sptmpl.length()>0 ){
                    stmpl.append("      parameters:").append("\n");
                    stmpl.append(sptmpl);
                }
            }

            retHelp = gh.getReturnHelp();
            if( retHelp!=null && retHelp.length()>0 ){
                help.put("return", retHelp);
                stmpl.append("      return:").append("\n");
                stmpl.append("        ${help.return:70}").append("\n");
            }

            longDesc = gh.getLongHelp();
            if( longDesc!=null && longDesc.length()>0 ){
                help.put("long", longDesc);
                stmpl.append("      ${help.long:70}").append("\n");
            }

            String sample = gh.getSampleHelp();
            if( sample!=null && sample.length()>0 ){
                help.put("sample", sample);
                stmpl.append("      sample:").append("\n");
                stmpl.append("        ${help.sample:70}").append("\n");
            }
        }

        BasicTemplate.EasyTemplate templ = template(stmpl.toString());

        templ.bind("help", help);
        templ.align();
        return templ.eval();
    }

    private String generateHelpOfCommands(){
        String fundescjoin = "\n\n";
        StringBuilder sb = new StringBuilder();

        for( String name : getCommandsMap().keySet() ){
            for( Function fun : getCommandsMap().get(name) ){
                if( name==null || fun==null )continue;

                String funhelp = generateHelpOfCommand(name, fun);
                if( funhelp!=null ){
                    funhelp = funhelp.trim();
                }

                if( funhelp!=null ){
                    if( sb.length()>0 )sb.append(fundescjoin);
                    sb.append(funhelp);
                }
            }
        }

        return sb.toString();
    }

    private String generateHelpOfCommand( String name, Function fun ){
        LinkedHashMap help = new LinkedHashMap();
        StringBuilder stmpl = new StringBuilder();
        Declaration declaration = new Declaration();
        String decl = declaration.getDeclareOf(fun);

        help.put("name", name);
        help.put("declare", decl);

        String shortDesc = null;
        String longDesc = null;
        String retHelp = null;

        if( fun instanceof GetHelp ){
            GetHelp gh = (GetHelp)fun;

            shortDesc = gh.getShortHelp();
            longDesc = gh.getLongHelp();

            help.put("short", shortDesc);

            stmpl.append("  ${help.name}");
            if( shortDesc!=null && shortDesc.length()>0 ){
                stmpl.append(" -  ${help.short:50}");
            }
            stmpl.append("\n");

            stmpl.append("    ${help.name}${help.declare}").append("\n");

            Class[] paramTypes = fun.getParameters();
            if( paramTypes!=null && paramTypes.length>0 ){
                StringBuilder sptmpl = new StringBuilder();
                for( int pi=0; pi<paramTypes.length; pi++ ){
                    String paramHelp = gh.getParameterHelp(pi);
                    if( paramHelp!=null && paramHelp.length()>0 ){
                        help.put("param"+pi+"name", declaration.getParamNameOf(fun, pi));
                        help.put("param"+pi+"help", paramHelp);
                        help.put("param"+pi+"type", declaration.getTypeName(paramTypes[pi]));

                        sptmpl.append("      ${help.param"+pi+"name}");
                        sptmpl.append(" : ${help.param"+pi+"type}");
                        sptmpl.append(" - ${help.param"+pi+"help:60}");
                        sptmpl.append("\n");
                    }
                }

                if( sptmpl.length()>0 ){
                    stmpl.append("    parameters:").append("\n");
                    stmpl.append(sptmpl);
                }
            }

            retHelp = gh.getReturnHelp();
            if( retHelp!=null && retHelp.length()>0 ){
                help.put("return", retHelp);

                stmpl.append("    return:").append("\n");
                stmpl.append("      ${help.return:70}").append("\n");
            }

            if( longDesc!=null && longDesc.length()>0 ){
                help.put("long", longDesc);
                stmpl.append("    ${help.long:70}").append("\n");
            }

            String sample = gh.getSampleHelp();
            if( sample!=null && sample.length()>0 ){
                help.put("sample", sample);
                stmpl.append("    sample:").append("\n");
                stmpl.append("      ${help.sample:70}").append("\n");
            }
        }else{
            stmpl.append("  ${help.name}").append("\n");
            stmpl.append("    ${help.name}${help.declare}").append("\n");
        }

        BasicTemplate.EasyTemplate templ = template(stmpl.toString());

        templ.bind("help", help);
        templ.align();
        return templ.eval();
    }
//</editor-fold>
}
