/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.cli;


import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.Predicate;
import xyz.cofe.text.template.BasicTemplate;

/**
 *
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class FunctionBuilder {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(FunctionBuilder.class.getName());
    private static final Level logLevel = logger.getLevel();

    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    private CommandLineMachine cmdMachine;

    public CommandLineMachine getCmdMachine() {
        return cmdMachine;
    }

    public void setCmdMachine(CommandLineMachine cmdMachine) {
        this.cmdMachine = cmdMachine;
    }

    public static class MethodCall implements Function, IsOperator, GetHelp, GetMethod
    {
        private Object owner;
        private Method method;
        private boolean operator = false;

        private String shortHelp;
        private String longHelp;
        private String[] paramHelp;
        private String returnHelp;
        private String sampleHelp;

        public MethodCall( CommandLineMachine cli, Object owner, Method method ){
            if( owner==null )throw new IllegalArgumentException( "owner==null" );
            if( method==null )throw new IllegalArgumentException( "method==null" );
            this.owner = owner;
            this.method = method;

            Help h = method.getAnnotation(Help.class);
            if( h!=null ){
                this.shortHelp = h.shortDesc().length()>0 ? h.shortDesc() : null;
                if( h.template() && this.shortHelp!=null ){
                    BasicTemplate.EasyTemplate etmpl = BasicTemplate.template(this.shortHelp);
                    etmpl.bind("owner", owner);
                    etmpl.bind("cli", cli);
                    etmpl.align();
                    this.shortHelp = etmpl.eval();
                }

                this.longHelp = h.longDesc().length()>0 ? h.longDesc() : null;
                if( h.template() && this.longHelp!=null ){
                    BasicTemplate.EasyTemplate etmpl = BasicTemplate.template(this.longHelp);
                    etmpl.bind("owner", owner);
                    etmpl.bind("cli", cli);
                    etmpl.align();
                    this.longHelp = etmpl.eval();
                }

                this.sampleHelp = h.sample().length()>0 ? h.sample() : null;
                if( h.template() && this.sampleHelp!=null ){
                    BasicTemplate.EasyTemplate etmpl = BasicTemplate.template(this.sampleHelp);
                    etmpl.bind("owner", owner);
                    etmpl.bind("cli", cli);
                    etmpl.align();
                    this.sampleHelp = etmpl.eval();
                }

                this.returnHelp = h.returnDesc().length()>0 ? h.returnDesc() : null;
                if( h.template() && this.returnHelp!=null ){
                    BasicTemplate.EasyTemplate etmpl = BasicTemplate.template(this.returnHelp);
                    etmpl.bind("owner", owner);
                    etmpl.bind("cli", cli);
                    etmpl.align();
                    this.returnHelp = etmpl.eval();
                }


                Annotation[][] anns = method.getParameterAnnotations();
                paramHelp = new String[anns.length];
                for( int pi=0; pi<paramHelp.length; pi++ ){
                    for( Annotation ann : anns[pi] ){
                        if( ann instanceof Help ){
                            Help ph = (Help)ann;
                            String shorth = ph.shortDesc();
                            String longh = ph.longDesc();
                            String t = (longh!=null && longh.length()>0) ?
                                longh : shorth;

                            if( t!=null && t.length()>0 ){
                                if( h.template() ){
                                    BasicTemplate.EasyTemplate etmpl = BasicTemplate.template(t);
                                    etmpl.bind("owner", owner);
                                    etmpl.bind("cli", cli);
                                    etmpl.align();
                                    t = etmpl.eval();
                                }

                                paramHelp[pi] = t;
                            }
                        }
                    }
                }
            }
        }

        public MethodCall( MethodCall src ){
            if( src==null )throw new IllegalArgumentException( "src==null" );
            this.owner = src.owner;
            this.method = src.method;
            this.operator = src.operator;
            this.shortHelp = src.shortHelp;
            this.longHelp = src.longHelp;
            this.returnHelp = src.returnHelp;
            this.paramHelp = src.paramHelp;
            this.sampleHelp = src.sampleHelp;
        }

        //<editor-fold defaultstate="collapsed" desc="short help">
        @Override
        public String getShortHelp() {
            return shortHelp;
        }

        public void setShortHelp(String shortHelp) {
            this.shortHelp = shortHelp;
        }
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="long help">
        @Override
        public String getLongHelp() {
            return longHelp;
        }

        public void setLongHelp(String longHelp) {
            this.longHelp = longHelp;
        }
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="sample help">
        @Override
        public String getSampleHelp() {
            return sampleHelp;
        }

        public void setSampleHelp(String sampleHelp) {
            this.sampleHelp = sampleHelp;
        }
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="operator prop">
        @Override
        public boolean isOperator() {
            return operator;
        }

        public void setOperator(boolean operator) {
            this.operator = operator;
        }
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="owner">
        public Object getOwner() {
            return owner;
        }
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="method">
        @Override
        public Method getMethod() {
            return method;
        }
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="parameters type">
        @Override
        public Class[] getParameters() {
            return method.getParameterTypes();
        }

        public String[] getParamHelp() {
            return paramHelp;
        }

        public void setParamHelp(String[] paramHelp) {
            this.paramHelp = paramHelp;
        }

        @Override
        public String getParameterHelp(int parameterIndex) {
            if( parameterIndex<0 )return null;
            if( paramHelp==null )return null;
            if( parameterIndex>=paramHelp.length )return null;
            String s = paramHelp[parameterIndex];
            return s==null ? null : s;
        }
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="return type">
        @Override
        public Class getReturn() {
            return method.getReturnType();
        }

        //<editor-fold defaultstate="collapsed" desc="return help">
        @Override
        public String getReturnHelp() {
            return returnHelp;
        }

        public void setReturnHelp(String returnHelp) {
            this.returnHelp = returnHelp;
        }
        //</editor-fold>
        //</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="apply()">
        @Override
        public Object apply(Object... args) {
            try {
                Object res = method.invoke(owner, args);
                return res;
            } catch (IllegalAccessException ex) {
                Logger.getLogger(FunctionBuilder.class.getName()).log(Level.SEVERE, null, ex);
            } catch (IllegalArgumentException ex) {
                Logger.getLogger(FunctionBuilder.class.getName()).log(Level.SEVERE, null, ex);
            } catch (InvocationTargetException ex) {
                Logger.getLogger(FunctionBuilder.class.getName()).log(Level.SEVERE, null, ex);
            }

            return null;
        }
//</editor-fold>

        @Override
        public String toString() {
            return "mcall"+( method!=null ? " "+method : "");
        }
    }

    protected Func3<Function, CommandLineMachine , Object, Method> builder;

    public Func3<Function, CommandLineMachine, Object, Method> getBuilder() {
        if( builder!=null )return builder;
        builder = new Func3<Function, CommandLineMachine, Object, Method>() {
            @Override
            public Function apply(CommandLineMachine cli, Object arg1, Method arg2) {
                return new MethodCall(cli, arg1, arg2);
            }
        };
        return builder;
    }

    public void setBuilder(Func3<Function, CommandLineMachine, Object, Method> builder) {
        this.builder = builder;
    }

    public Function create( final Object owner, final Method method ){
        if( owner==null )throw new IllegalArgumentException( "owner==null" );
        if( method==null )throw new IllegalArgumentException( "method==null" );

        Function fun = getBuilder().apply(getCmdMachine(), owner, method);
        return fun;
    }

    public Function[] create( Object owner, Predicate<Method> mfilter ){
        if( owner==null )throw new IllegalArgumentException( "owner==null" );
        if( mfilter==null )throw new IllegalArgumentException( "mfilter==null" );

        Function[] res = new Function[]{};

        Method[] methods = owner.getClass().getMethods();
        for( Method m : methods ){
            if( mfilter.validate(m) ){
                Function f = create(owner, m);
                if( f==null )continue;

                res = Arrays.copyOf(res, res.length+1);
                res[res.length-1] = f;
            }
        }

        return res;
    }

    public Function[] create( Object owner, final Class<? extends java.lang.annotation.Annotation> ann ){
        if( owner==null )throw new IllegalArgumentException( "owner==null" );
        if( ann==null )throw new IllegalArgumentException( "ann==null" );

        Predicate<Method> p = new Predicate<Method>() {
            @Override
            public boolean validate(Method t) {
                Object oann = t.getAnnotation(ann);
                return oann != null;
            }
        };

        return create(owner, p);
    }
}
