/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.cli;


import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Iterators;
import xyz.cofe.collection.Predicate;

/**
 * 
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class FunctionSetHelper {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(FunctionSetHelper.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    private Iterable<Function> funitr;
    private FunctionSet fset;
    private Memory mem;
    
    public FunctionSetHelper(final Memory mem){
        if( mem==null )throw new IllegalArgumentException( "mem==null" );
        this.mem = mem;
        this.funitr = new Iterable<Function>() {
            @Override
            public Iterator<Function> iterator() {
                List<Function> l = new ArrayList<Function>();
                for( Object o : mem.values() ){
                    if( o instanceof Function ){
                        l.add((Function)o);
                    }else if( o instanceof FunctionSet ){
                        for( Function f : ((FunctionSet)o).getFunctions() ){
                            l.add(f);
                        }
                    }
                }
                return l.iterator();
            }
        };
    }
    
    public FunctionSetHelper(Iterable<Function> funitr){
        if( funitr==null )throw new IllegalArgumentException( "funitr==null" );
        this.funitr = funitr;
    }

    public FunctionSetHelper(Map<String,Set<Function>> map){
        if( map==null )throw new IllegalArgumentException( "map==null" );
        final Map<String,Set<Function>> mp = map;
        this.funitr = new Iterable<Function>() {
            @Override
            public Iterator<Function> iterator() {
                List<Function> lfun = new ArrayList<Function>();
                for( Map.Entry<String,Set<Function>> en : mp.entrySet() ){
                    lfun.addAll(en.getValue());
                }
                return lfun.iterator();
            }
        };
    }
    
    public FunctionSetHelper(FunctionSet fset){
        this.fset = fset;
    }
    
    public Function first(){
        for( Function f : functions() ){
            return f;
        }
        return null;
    }
    
    //<editor-fold defaultstate="collapsed" desc="print()">
    public FunctionSetHelper print(){
        OutputStreamWriter w = new OutputStreamWriter(System.out);
        print(w);
        try {
            w.flush();
        } catch (IOException ex) {
            Logger.getLogger(FunctionSetHelper.class.getName()).log(Level.SEVERE, null, ex);
        }
        
        return this;
    }
    
    public FunctionSetHelper print(Writer w){
        if( w==null )throw new IllegalArgumentException( "w==null" );
        
        SourceDump sdump = new SourceDump();
        int i=-1;
        for( Function f : functions() ){
            try {
                i++;
//                String decl = f==null ? "null" : sdump.getDeclareOf(f);
                
                w.write(Integer.toString(i));
                w.write(". ");
//                w.write(decl);
                w.write("\n");
            } catch (IOException ex) {
                Logger.getLogger(FunctionSetHelper.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        
        return this;
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="functions()">
    public Iterable<Function> functions(){
        if( fset!=null )return fset.getFunctions();
        if( funitr!=null )return funitr;
        return Iterators.empty();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="filter">
    public Iterable<Function> filter( Iterable<Function> src, Predicate<Function> f ){
        if( src==null )throw new IllegalArgumentException( "src==null" );
        if( f==null )throw new IllegalArgumentException( "f==null" );
        return Iterators.predicate(src, f);
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="argumentsCount">
    public Predicate<Function> argumentsCountFilter( final Predicate<Integer> countPred ){
        if( countPred==null )throw new IllegalArgumentException( "countPred==null" );
        return new Predicate<Function>() {
            @Override
            public boolean validate(Function t) {
                if( t==null )return false;
                return countPred.validate(t.getParameters().length);
            }
        };
    }
    
    public Predicate<Function> argumentsCountFilter( final int count ){
        return argumentsCountFilter(new Predicate<Integer>() {
            @Override
            public boolean validate(Integer t) {
                return count == t;
            }
        });
    }
    
    public FunctionSetHelper argumentsCount( int count ){
        return new FunctionSetHelper(
            filter(functions(), argumentsCountFilter(count))
        );
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="assignFromFilter()">
    public Predicate<Class> assignFromFilter( final Class type ){
        if( type==null )throw new IllegalArgumentException( "type==null" );
        return new Predicate<Class>() {
            @Override
            public boolean validate(Class t) {
                if( t==null )return false;
                return t.isAssignableFrom(type);
            }
        };
    }

    public Predicate<Class> assignFromFilter2( final Class type ){
        if( type==null )throw new IllegalArgumentException( "type==null" );
        return new Predicate<Class>() {
            @Override
            public boolean validate(Class t) {
                if( t==null )return false;
                return type.isAssignableFrom(t);
            }
        };
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="functionArguments()">
    public Predicate<Function> functionArgumentsFilter( final Predicate<Class> ... argTypes ){
        if( argTypes==null )throw new IllegalArgumentException( "argTypes==null" );
        for( int ai=0; ai<argTypes.length; ai++ ){
            if( argTypes[ai]==null )
                throw new IllegalArgumentException( "argTypes["+ai+"]==null" );
        }
        
        return new Predicate<Function>() {
            @Override
            public boolean validate(Function f) {
                if( f==null )return false;
                Class[] params = f.getParameters();
                if( params.length!=argTypes.length )return false;
                for( int i=0; i<params.length; i++ ){
                    boolean m = argTypes[i].validate(params[i]);
                    if( !m )return false;
                }
                return true;
            }
        };
    }
    
    public FunctionSetHelper functionArguments( final Predicate<Class> ... argTypes ){
        return new FunctionSetHelper(
            filter(functions(), functionArgumentsFilter(argTypes)));
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="functionArgument()">
    public Predicate<Function> functionArgumentFilter( final int argumentIndex, final Predicate<Class> argType ){
        if( argType==null )throw new IllegalArgumentException( "argType==null" );
        return new Predicate<Function>() {
            @Override
            public boolean validate(Function f) {
                if( f==null )return false;
                Class[] params = f.getParameters();
                if( argumentIndex>=params.length || argumentIndex<0 )return false;
                return argType.validate(params[argumentIndex]);
            }
        };
    }
    
    public FunctionSetHelper functionArgument( final int argumentIndex, final Predicate<Class> argType ){
        return new FunctionSetHelper(
            filter(
                functions(),
                functionArgumentFilter(argumentIndex, argType)
            )
        );
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="named()">
    public FunctionSetHelper named( final String name ){
        if( name==null )throw new IllegalArgumentException( "name==null" );
        if( mem==null )return new FunctionSetHelper( Iterators.<Function>empty() );
        
        Object o = mem.get(name);
        if( o instanceof Function ){
            FunctionSetHelper fs = new FunctionSetHelper( Iterators.<Function>single((Function)o));
            fs.mem = mem;
            return fs;
        }
        
        if( o instanceof FunctionSet ){
            FunctionSetHelper fs = new FunctionSetHelper( ((FunctionSet)o).getFunctions() );
            fs.mem = mem;
            return fs;
        }
        
        return new FunctionSetHelper( Iterators.<Function>empty() );
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="operators()">
    public Predicate<Function> operatorFilter(){
        return new Predicate<Function>() {
            @Override
            public boolean validate(Function f) {
                if( f==null )return false;
                if( f instanceof IsOperator ){
                    return ((IsOperator)f).isOperator();
                }
                return false;
            }
        };
    }
    
    public FunctionSetHelper operators(){
        return new FunctionSetHelper(
            filter(
                functions(),
                operatorFilter()
            )
        );
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="count()">
    public int count(){
        Iterable<Function>  itrf = functions();
        long c = Iterators.count(itrf);
        
        return new Long(c).intValue();
    }
//</editor-fold>
    
    public Predicate<Function> in( final Iterable<Function> src ){
        if( src==null )throw new IllegalArgumentException( "src==null" );
        return new Predicate<Function>() {
            @Override
            public boolean validate(Function f) {
                if( f==null )return false;
                return Iterators.in(f, src);
            }
        };
    }
    
    public Predicate<Function> not( final Predicate<Function> src ){
        if( src==null )throw new IllegalArgumentException( "src==null" );
        return new Predicate<Function>() {
            @Override
            public boolean validate(Function f) {
                if( f==null )return false;
                return ! src.validate(f);
            }
        };
    }
    
    public Predicate<Function> and( final Predicate<Function> ... src ){
        if( src==null )throw new IllegalArgumentException( "src==null" );
        return new Predicate<Function>() {
            @Override
            public boolean validate(Function f) {
                if( src==null )return false;
                if( src.length==0 )return false;
                for( Predicate<Function> p : src ){
                    if( !p.validate(f) )return false;
                }
                return true;
            }
        };
    }
    
    public Predicate<Function> or( final Predicate<Function> ... src ){
        if( src==null )throw new IllegalArgumentException( "src==null" );
        return new Predicate<Function>() {
            @Override
            public boolean validate(Function f) {
                if( src==null )return false;
                if( src.length==0 )return false;
                for( Predicate<Function> p : src ){
                    if( p.validate(f) )return true;
                }
                return false;
            }
        };
    }
}
