/*
 * The MIT License
 *
 * Copyright 2016 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.cli;


import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Stack;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.Pointer;
import xyz.cofe.text.EndLine;
import xyz.cofe.text.EndLineReWriter;
import xyz.cofe.text.Text;
import xyz.cofe.text.parser.Token;

/**
 *
 * @author nt.gocha@gmail.com
 */
public class InteractiveShell {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(InteractiveShell.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected final Lock lock;

    public InteractiveShell( CommandLineMachine cli ){
        if( cli==null )throw new IllegalArgumentException( "cli==null" );

        this.lock = new ReentrantLock();
        this.cli = cli;
    }

    //<editor-fold defaultstate="collapsed" desc="cli">
    protected CommandLineMachine cli;

    public CommandLineMachine getCli() {
        try{
            lock.lock();
            return cli;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="exit">
    protected Func1<Boolean,String> exit;

    public Func1<Boolean, String> getExit() {
        try{
            lock.lock();
            if( exit!=null ){
                return exit;
            }
            exit = new Func1<Boolean, String>() {
                @Override
                public Boolean apply( String inputLine ) {
                    if( inputLine==null )return true;
                    if( inputLine.trim().equalsIgnoreCase("exit") )return true;
                    return false;
                }
            };
            return exit;
        }finally{
            lock.unlock();
        }
    }

    public void setExit( Func1<Boolean, String> exit ) {
        try{
            lock.lock();
            this.exit = exit;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="lexer">
    protected Lexer lexer;

    public Lexer getLexer() {
        try{
            lock.lock();
            if( lexer!=null )return lexer;
            lexer = getCli().createLexer();
            return lexer;
        }finally{
            lock.unlock();
        }
    }

    public void setLexer( Lexer lexer ) {
        try{
            lock.lock();
            this.lexer = lexer;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="parser">
    protected Parser parser;

    public Parser getParser() {
        try{
            lock.lock();
            if( parser!=null )return parser;
            parser = cli.createParser();
            return parser;
        }finally{
            lock.unlock();
        }
    }

    public void setParser( Parser parser ) {
        try{
            lock.lock();
            this.parser = parser;
            this.openCloseMap = null;
            this.closeOpenMap = null;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="openCloseMap">
    protected Map<String,String> openCloseMap = new LinkedHashMap<String, String>();
    protected Map<String,String> getOpenCloseMap(){
        try{
            lock.lock();
            if( openCloseMap!=null )return openCloseMap;

            openCloseMap = new LinkedHashMap<String, String>();
            for( Map.Entry<String,String> en : getParser().getBrackets().entrySet() ){
                openCloseMap.put(en.getKey(), en.getValue());
            }

            return openCloseMap;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="closeOpenMap">
    protected Map<String,String> closeOpenMap = new LinkedHashMap<String, String>();
    protected Map<String,String> getCloseOpenMap(){
        try{
            lock.lock();
            if( closeOpenMap!=null )return closeOpenMap;

            closeOpenMap = new LinkedHashMap<String, String>();
            for( Map.Entry<String,String> en : getParser().getBrackets().entrySet() ){
                closeOpenMap.put(en.getValue(), en.getKey());
            }

            return closeOpenMap;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="printResult">
    protected Func1<Boolean,Object> printResult;
    
    public Func1<Boolean, Object> getPrintResult() {
        try{
            lock.lock();
            if( printResult==null ){
                printResult = new Func1<Boolean, Object>() {
                    @Override
                    public Boolean apply( Object arg ) {
                        return true;
                    }
                };
            }
            return printResult;
        }finally{
            lock.unlock();
        }
    }
    
    public void setPrintResult( Func1<Boolean, Object> printResult ) {
        try{
            lock.lock();
            this.printResult = printResult;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="running">
    protected boolean running = false;
    
    public boolean isRunning() {
        try{
            lock.lock();
            return running;
        }finally{
            lock.unlock();
        }
    }
    
    protected void setRunning( boolean running ) {
        try{
            lock.lock();
            this.running = running;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>
    
    protected Console console;

    public Console getConsole() {
        try {
            lock.lock();
            if( console!=null )return console;
            
            if( System.console()!=null ){
                try {
                    console = new JLineConsole();
                } catch (IOException ex) {
                    console = new BaseConsole();
                }
            }else{
                console = new BaseConsole();
            }
            
//            console = new BaseConsole();
            
            return console;
        }
        finally {
            lock.unlock();
        }
    }

    protected String getPromptInput(){      return "   input> "; }
    protected String getPromptContinue(){   return "continue> "; }
    protected String getPromptResult(){     return "  result> "; }
    protected String getPromptError(){      return "   error> "; }
    protected String getPromptOutput(){     return "  output> "; }
    protected String getPromptDumpTree(){   return "dumptree> "; }

    //<editor-fold defaultstate="collapsed" desc="outputWriter">
    private Writer output;
    public Writer getOutputWriter(){
        try {
            lock.lock();
            if( output!=null )return output;
            output = createConsoleWriter(getPromptOutput());
            return output;
        }
        finally {
            lock.unlock();
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="output() / readInputLine()">
    private String lastOutputLine;
    
    private String readInputLine(String prompt){
        try {
            lock.lock();
            
            if( lastOutputLine!=null && lastOutputLine.length()>0 ){
                getConsole().println();
            }
            
            if( prompt!=null ){
                getConsole().print(prompt);
            }
            
            String line = getConsole().readLine();
            return line;
        }
        finally {
            lock.unlock();
        }
    }
    
    private void output( String text ){
        try {
            lock.lock();
            if( text==null )return;
            String[] lines = Text.splitNewLines(text);
            int linei = -1;
            for( String line : lines ){
                linei++;
                if( linei>0 ){
                    getConsole().println();
                }
                getConsole().print(line);
                lastOutputLine = line;
            }
        }
        finally {
            lock.unlock();
        }
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="createConsoleWriter()">
    public Writer createConsoleWriter(final String indent){
        Writer w = new Writer() {
            @Override
            public void write(char[] cbuf, int off, int len) throws IOException {
                output( new String(cbuf,off,len) );
            }
            @Override public void flush() throws IOException {}
            @Override public void close() throws IOException {}
        };
        
        if( indent==null || indent.length()<1 ){
            return w;
        }
        
        EndLineReWriter rewr = new EndLineReWriter(w){
            protected boolean indentWrited = false;
            
            protected void writeIndent(boolean checkWrited) throws IOException{
                if( checkWrited ){
                    if( indentWrited )return;
                }
                this.writer.append(indent);
                indentWrited = true;
            }
            
            @Override
            protected void writeChar(char c0) throws IOException {
                writeIndent(true);
                super.writeChar(c0);
            }
            
            @Override
            public Writer append(char c) throws IOException {
                writeIndent(true);
                return super.append(c);
            }
            
            @Override
            public Writer append(CharSequence csq, int start, int end) throws IOException {
                if((end - start) > 0)writeIndent(true);
                return super.append(csq, start, end);
            }
            
            @Override
            public Writer append(CharSequence csq) throws IOException {
                if( csq.length()>0 )writeIndent(true);
                return super.append(csq);
            }
            
            @Override
            public void write(String str, int off, int len) throws IOException {
                if( len>0 )writeIndent(true);
                super.write(str, off, len);
            }
            
            @Override
            public void write(String str) throws IOException {
                if( str.length()>0 )writeIndent(true);
                super.write(str);
            }
            
            @Override
            public void write(char[] cbuf) throws IOException {
                if( cbuf.length>0 )writeIndent(true);
                super.write(cbuf);
            }
            
            @Override
            public void write(int c) throws IOException {
                writeIndent(true);
                super.write(c);
            }
            
            @Override
            protected void writeEndLine() throws IOException {
                this.writer.append(EndLine.Default.get());
                indentWrited = false;
            }
        };
        
        return rewr;
    }
//</editor-fold>
    
    public void start(){
        try{
            try{
                lock.lock();
                boolean runn = isRunning();
                if( runn )throw new IllegalStateException("start() already called and not stopped");
                setRunning(true);
            }finally{
                lock.unlock();
            }

            StringBuilder buff = new StringBuilder();

            String prompt = getPromptInput();

            while( true ){
                String line = readInputLine(prompt);
                if( line==null )break;

                if( getExit().apply(line) )break;
                line = line.trim();
                if( line.length()==0 )continue;

                buff.append(line).append(EndLine.Default.get());

                Pointer<Token> toks = getLexer().parse(buff.toString());
                Stack<String> waitCloseBR = new Stack<String>();

                int bracketBalance = 0;
                boolean succBracket = true;

                for( Token tok : toks.getList() ){
                    String txt = tok.getMatchedText();
                    
                    if( Text.in(txt, getOpenCloseMap().keySet()) ){
                        waitCloseBR.add(getOpenCloseMap().get(txt));
                        bracketBalance++;
                    }else if( Text.in(txt, getCloseOpenMap().keySet()) ){
                        if( waitCloseBR.isEmpty() ){
                            succBracket = false;
                            break;
                        }
                        
                        String wait = waitCloseBR.pop();
                        if( txt.equals(wait) ){
                            bracketBalance--;
                            succBracket = true;
                        }else{
                            succBracket = false;
                            break;
                        }
                    }
                }
                
                if( bracketBalance>0 ){
                    prompt = getPromptContinue();
                    continue;
                }
                
                Value v = null;
                
                v = getParser().parse(toks);
                prompt = getPromptInput();

                buff.setLength(0);

                if( v==null ){
                    output( getPromptError()+"can't parse");
                    continue;
                }
                
                if( cli!=null && cli.isDump() ){
                    StringWriter sw = new StringWriter();

                    SourceDump sdump = new SourceDump();
                    sdump.setMemory(getParser().getMemory());
                    sdump.output(sw);
                    sdump.dump(v);

                    output(Text.indent(sw.toString(),getPromptDumpTree()));
                }

                Object res = null;
                try{
                    res = v.eval();
                }catch( Throwable err ){
                    output(Text.indent(err.getMessage(),getPromptError()));
                    continue;
                }

                if( res!=null && getPrintResult().apply(res) ){
                    output(Text.indent(res.toString(),getPromptResult()));
                }
            }

        }finally{
            setRunning(false);
        }
    }
}
