/*
 * The MIT License
 *
 * Copyright 2016 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.cli;


import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Pointer;
import xyz.cofe.text.Output;
import xyz.cofe.text.parser.ListLexer;
import xyz.cofe.text.parser.Token;
import xyz.cofe.text.parser.lex.AnyCharParser;
import xyz.cofe.text.parser.lex.FieldAccessParser;
import xyz.cofe.text.parser.lex.KeywordsParser;
import xyz.cofe.text.parser.lex.NumberConstParser;
import xyz.cofe.text.parser.lex.TextConstParser;
import xyz.cofe.text.parser.lex.WhiteSpace;
import xyz.cofe.text.parser.lex.WhiteSpaceParser;

/**
 *
 * @author nt.gocha@gmail.com
 */
public class Lexer {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(Lexer.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected void dump( Output out, List<Token> tokens ){
        if( tokens==null )return;
        if( out==null )return;
        int ti = -1;
        for( Token t : tokens ){
            ti++;
            out.print( (ti+1)+"/"+(tokens.size())+": " );
            if( t==null ){
                out.println( "null" );
            }else{
                out.println( t.getMatchedText() );
            }
        }
    }

    protected Output dumpOutput;

    public Output getDumpOutput() {
        return dumpOutput;
    }

    public void setDumpOutput(Output dumpOutput) {
        this.dumpOutput = dumpOutput;
    }

    public Pointer<Token> parse( String[] args ){
        if( args==null )throw new IllegalArgumentException( "args==null" );
        List<Token> ltokens = new ArrayList<Token>();
        for( String a : args ){
            if( a==null )continue;
            Token t = new Token("arg", a, 0, a.length());
            ltokens.add(t);
        }
        if( dumpOutput!=null )dump(dumpOutput, ltokens);
        return new Pointer<Token>(ltokens);
    }

    private boolean skipWhitespace = true;

    public boolean isSkipWhitespace() {
        return skipWhitespace;
    }

    public void setSkipWhitespace( boolean skipWhitespace ) {
        this.skipWhitespace = skipWhitespace;
    }

    private boolean keepNewLine = true;

    public boolean isKeepNewLine() {
        return keepNewLine;
    }

    public void setKeepNewLine( boolean keepNewLine ) {
        this.keepNewLine = keepNewLine;
    }

    private static String[] specChars = new String[]{
            "{","}"
            ,"[","]"
            ,"(",")"
            ,"!"
            ,"+", "-", "*", "/"
            ,".", ";", ","
            , "@", "#", "$", "'", "\"", "%"
            , ":", "\\", "|", "^", "&", "=", "_"
            , "<", ">", "<=", ">=", "<>", "!=", "==", "=>", "=<", "->"
            , "`", "~", "--"
            , "true", "false", "null"
    };

    public Pointer<Token> parse( String source ){
        if( source==null )throw new IllegalArgumentException( "source==null" );

        ListLexer llex = new ListLexer();

        WhiteSpaceParser ws = new WhiteSpaceParser();
        llex.getParsers().add(ws);

        TextConstParser tconst = new TextConstParser();
        llex.getParsers().add(tconst);

//        IdentifierParser id = new IdentifierParser();
//        llex.getParsers().add(id);

        FieldAccessParser fa = new FieldAccessParser();
        llex.getParsers().add(fa);

        NumberConstParser numconst = new NumberConstParser();
        llex.getParsers().add(numconst);

        KeywordsParser kw = new KeywordsParser(false, specChars);
        llex.getParsers().add(kw);

        AnyCharParser anyc = new AnyCharParser();
        llex.getParsers().add(anyc);

        List<Token> ltoks = llex.parse(source);

        if( isSkipWhitespace() ){
            for( Object t : ltoks.toArray() ){
                if( t instanceof WhiteSpace ){
                    WhiteSpace wst = (WhiteSpace)t;
                    String mt = wst.getMatchedText();
                    boolean isNewLine =
                        mt.equals("\n") |
                        mt.equals("\r") |
                        mt.equals("\n\r") |
                        mt.equals("\r\n")
                        ;

                    if( !keepNewLine ){
                        ltoks.remove(t);
                    }else{
                        if( !isNewLine ){
                            ltoks.remove(t);
                        }
                    }
                }
            }
        }

        if( dumpOutput!=null )dump(dumpOutput, ltoks);

        return new Pointer<Token>(ltoks);
    }
}
