/*
 * The MIT License
 *
 * Copyright 2016 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.cli;


import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.StringWriter;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.Func4;
import xyz.cofe.collection.Func5;
import xyz.cofe.collection.Func6;
import xyz.cofe.collection.Func7;
import xyz.cofe.collection.Func8;
import xyz.cofe.collection.Func9;
import xyz.cofe.collection.Pointer;
import xyz.cofe.collection.map.BasicEventMap;
import xyz.cofe.collection.map.LockEventMap;
import xyz.cofe.text.Text;
import xyz.cofe.text.parser.Token;
import xyz.cofe.text.parser.lex.FieldAccess;
import xyz.cofe.text.parser.lex.FieldAccessParser;
import xyz.cofe.text.parser.lex.NumberConst;
import xyz.cofe.text.parser.lex.NumberConstParser;
import xyz.cofe.text.parser.lex.TextConst;
import xyz.cofe.text.parser.lex.TextConstParser;
import xyz.cofe.typeconv.ExtendedCastGraph;
import xyz.cofe.typeconv.TypeCastGraph;

/**
 *
 * @author nt.gocha@gmail.com
 */
public class Parser {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(Parser.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected final Lock lock;

    public Parser(){
        lock = new ReentrantLock();
    }

    protected void trace( String message ){
    }

    //<editor-fold defaultstate="collapsed" desc="clone()">
    public Parser(Parser source){
        if( source==null )throw new IllegalArgumentException( "source==null" );
        lock = new ReentrantLock();
        try{
            source.lock.lock();
            this.memory = source.memory;
            this.typeCastGraph = source.typeCastGraph;
            this.assignOpertator = source.assignOpertator;
            this.requireNextStatementDelimiter = source.requireNextStatementDelimiter;
            this.nextStatement = source.nextStatement;
            this.parseNumbers = source.parseNumbers;
            this.brackets = new LockEventMap<String, String>(new LinkedHashMap<String, String>(), lock);
            if( source.brackets!=null ){
                this.brackets.putAll(source.brackets);
            }
            this.lambdaFollowOp = source.lambdaFollowOp;
        }finally{
            source.lock.unlock();
        }
    }

    @Override
    public Parser clone(){
        return new Parser(this);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="memory">
    protected Memory memory;

    public Memory getMemory() {
        try{
            lock.lock();
            if( memory!=null )
                return memory;
            memory = new Memory(new LinkedHashMap<String, Object>(),lock);
            return memory;
        }finally{
            lock.unlock();
        }
    }

    public void setMemory(Memory memory) {
        try{
            lock.lock();
            this.memory = memory;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="typeCastGraph">
    protected TypeCastGraph typeCastGraph;

    public TypeCastGraph getTypeCastGraph() {
        try{
            lock.lock();
            if( typeCastGraph!=null ){
                return typeCastGraph;
            }
            typeCastGraph = new ExtendedCastGraph();
            return typeCastGraph;
        }finally{
            lock.unlock();
        }
    }

    public void setTypeCastGraph(TypeCastGraph typeCastGraph) {
        try{
            lock.lock();
            this.typeCastGraph = typeCastGraph;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="requireNextStatementDelimiter">
    protected boolean requireNextStatementDelimiter = true;

    public boolean isRequireNextStatementDelimiter() {
        try{
            lock.lock();
            return requireNextStatementDelimiter;
        }finally{
            lock.unlock();
        }
    }

    public void setRequireNextStatementDelimiter(boolean requireNextStatementDelimiter) {
        try{
            lock.lock();
            this.requireNextStatementDelimiter = requireNextStatementDelimiter;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="nextStatement">
    public boolean isNextStatement( String st ){
        try{
            lock.lock();
            if( st==null )return false;
            for( String s : getNextStatement() ){
                if( s==null )continue;
                if( s.equals(st) )return true;
            }
            return false;
        }finally{
            lock.unlock();
        }
    }

    protected String[] nextStatement;

    public String[] getNextStatement() {
        try{
            lock.lock();
            if( nextStatement==null ){
                nextStatement = new String[]{ ";", ",", "\n\r", "\r\n", "\n", "\r" };
            }
            return nextStatement;
        }finally{
            lock.unlock();
        }
    }

    public void setNextStatement(String[] nextStatement) {
        try{
            lock.lock();
            this.nextStatement = nextStatement;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="parseNumbers">
    protected boolean parseNumbers = true;

    public boolean isParseNumbers() {
        try{
            lock.lock();
            return parseNumbers;
        }finally{
            lock.unlock();
        }
    }

    public void setParseNumbers(boolean parseNumbers) {
        try{
            lock.lock();
            this.parseNumbers = parseNumbers;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="assignOpertator">
    protected String assignOpertator = "=";

    public String getAssignOpertator() {
        try{
            lock.lock();
            return assignOpertator;
        }finally{
            lock.unlock();
        }
    }

    public void setAssignOpertator( String assignOpertator ) {
        try{
            lock.lock();
            this.assignOpertator = assignOpertator;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="brackets">
    protected Map<String,String> brackets;
    public Map<String,String> getBrackets(){
        try{
            lock.lock();
            if( brackets!=null )return brackets;

            brackets =
                new LockEventMap<String, String>(
                    new LinkedHashMap<String, String>(),
                    lock
                );

            brackets.put("(", ")");
            brackets.put("{", "}");
            brackets.put("[", "]");
            return brackets;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    public Value parse( Pointer<Token> ptr ){
        if( ptr==null )throw new IllegalArgumentException( "ptr==null" );
        try{
            lock.lock();
            defineVars.clear();
            return statements(ptr);
        }finally{
            lock.unlock();
        }
    }

    //<editor-fold defaultstate="collapsed" desc="statements()">
    /**
     * <pre>
     * statements ::= operator [ nextStatement statements ]
     * </pre>
     * @param ptr Токены/Лексемы
     * @return Значение или null
     */
    protected Value statements( Pointer<Token> ptr ){
        Value vOP1 = operator(ptr, null);

        if( vOP1!=null ){
            while( true ){
                Token t = ptr.lookup(0);

                if( t==null ){
                    break;
                }

                String expandOperatorName = t.getMatchedText();

                if( isBracket(expandOperatorName) )break;
                if( isNextStatement(expandOperatorName) )break;

//                if( !getMemory().containsKey(expandOperatorName) )break;
//
//                Object memt = getMemory().get(expandOperatorName);
//                if( memt==null )break;
//
//                if( !(memt instanceof Function || memt instanceof FunctionSet) )
//                    break;

                Object operators = getMemory().getOperators(expandOperatorName);
                if( operators==null )break;

                int pbefore = ptr.getIndex();
                ptr.push();

                Value lOP = operator(ptr, vOP1);
                if( lOP!=null ){
                    int pafter = ptr.getIndex();
                    if( pafter > pbefore ){
                        vOP1 = lOP;
                        ptr.pop();
                    }else{
                        ptr.restore();
                        break;
                    }
                }else{
                    ptr.restore();
                    break;
                }
            }

            Token t = ptr.lookup(0);
            String s = t!=null ? t.getMatchedText() : null;
            if( s!=null ){
                if( isRequireNextStatementDelimiter() ){
                    if( isNextStatement(s) ){
                        ptr.push();
                        ptr.move(1);

                        Value vOP2 = statements(ptr);
//                        Value vOP2 = operator(ptr,null);
                        if( vOP2!=null ){
                            ptr.pop();

                            if( vOP2 instanceof Sequence ){
                                Sequence sv = new Sequence();
                                sv.getChildrenList().add(vOP1);
                                sv.getChildrenList().addAll(((Sequence)vOP2).getChildrenList());
                                return sv;
                            }

                            Sequence sv = new Sequence();
                            sv.appendChild(vOP1);
                            sv.appendChild(vOP2);
                            return sv;
                        }

                        ptr.restore();
                    }else{
                        return vOP1;
                    }
                }else{
                    ptr.push();

                    while( true ){
                        if( isNextStatement(s) ){
                            ptr.move(1);
                        }

                        t = ptr.lookup(0);
                        s = t!=null ? t.getMatchedText() : null;
                        if( s==null )return vOP1;

                        break;
                    }

                    Value vOP2 = statements(ptr);
                    if( vOP2!=null ){
                        ptr.pop();

                        Sequence sv = new Sequence();
                        sv.appendChild(vOP1);
                        sv.appendChild(vOP2);
                        return sv;
                    }

                    ptr.restore();
                }
            }

            return vOP1;
        }

        return null;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="lambdaFollowOp">
    protected String lambdaFollowOp = "=>";

    public String getLambdaFollowOp() {
        try{
            lock.lock();
            if( lambdaFollowOp==null )lambdaFollowOp = "=>";
            return lambdaFollowOp;
        }finally{
            lock.unlock();
        }
    }

    public void setLambdaFollowOp( String lambdaFollowOp ) {
        try{
            lock.lock();
            this.lambdaFollowOp = lambdaFollowOp;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>

    public static class CatchedLambda {
        public List<String> args = new ArrayList<String>();
        public List<Token> body = new ArrayList<Token>();
        public Class getFunType(){
            if( args==null )return null;
            if( args.size()==0 )return Func0.class;
            if( args.size()==1 )return Func1.class;
            if( args.size()==2 )return Func2.class;
            if( args.size()==3 )return Func3.class;
            if( args.size()==4 )return Func4.class;
            if( args.size()==5 )return Func5.class;
            if( args.size()==6 )return Func6.class;
            if( args.size()==7 )return Func7.class;
            if( args.size()==8 )return Func8.class;
            if( args.size()==9 )return Func9.class;
            return null;
        }
    }

    protected CatchedLambda catchLambda( Pointer<Token> ptr ){
        if( ptr==null )return null;
        ptr.push();

        List<Token> body = new ArrayList<Token>();
        List<String> args = new ArrayList<String>();

        // parse arg names
        while( true ){
            Token t = ptr.lookup(0);
            if( t==null ){
                break;
            }

            String s = t.getMatchedText();
            if( s.matches("^((_|\\w)(_|\\w|\\d)*)$") ){
                args.add(s);
                ptr.move(1);
            }else{
                break;
            }
        }

        if( args.size()<1 ){
            ptr.restore();
            return null;
        }

        // to body ptr: "=>"
        Token toBodyPtr = ptr.lookup(0);
        if( toBodyPtr==null ){
            ptr.restore();
            return null;
        }

        String sToBody = toBodyPtr.getMatchedText();
        if( !sToBody.equals(getLambdaFollowOp()) ){
            ptr.restore();
            return null;
        }
        ptr.move(1);

        // check bracket
        Token t = null;
        t = ptr.lookup(0);
        if( t==null ){
            ptr.restore();
            return null;
        }
        String s = t.getMatchedText();
        if( isBracket(s) ){
            if( !getBrackets().containsKey(s) ){
                ptr.restore();
                return null;
            }

            Stack<String> bracketStack = new Stack<String>();
            bracketStack.add(getBrackets().get(s));
            body.add(t);
            ptr.move(1);

            while( true ){
                t = ptr.lookup(0);
                if( t==null ){
                    if( !bracketStack.isEmpty() ){
                        ptr.restore();
                        return null;
                    }
                    break;
                }

                s = t.getMatchedText();
                if( isBracket(s) ){
                    boolean openBr = getBrackets().containsKey(s);
                    if( openBr ){
                        bracketStack.add(getBrackets().get(s));
                        body.add(t);
                        ptr.move(1);
                    }else{
                        if( bracketStack.isEmpty() ){
                            ptr.restore();
                            return null;
                        }
                        String cbr = bracketStack.pop();
                        if( !cbr.equals(s) ){
                            ptr.restore();
                            return null;
                        }
                        if( bracketStack.isEmpty() ){
                            body.add(t);
                            ptr.move(1);
                            break;
                        }else{
                            body.add(t);
                            ptr.move(1);
                        }
                    }
                }else{
                    body.add(t);
                    ptr.move(1);
                }
            }
        }else{
            body.add(t);
            ptr.move(1);

            while( true ){
                t = ptr.lookup(0);
                if( t==null ){
                    break;
                }
                s = t.getMatchedText();
                if( isNextStatement(s) ){
                    ptr.move(1);
                    break;
                }
                body.add(t);
                ptr.move(1);
            }
        }

        ptr.pop();

        CatchedLambda clambda = new CatchedLambda();
        clambda.args = args;
        clambda.body = body;

        return clambda;
    }

    protected LambdaCall createLambdaCall( CatchedLambda clambda ){
        LambdaCall lcall = new LambdaCall();
        lcall.setParser(this.clone());
        lcall.setLambda(clambda);

        Class c = clambda.getFunType();
        if( c==null )throw new Error( "can't get class for CatchedLambda" );

        lcall.setType(c);
        return lcall;
    }

    protected LambdaCall createLambdaCall( CatchedLambda clambda, Class lambdaType ){
        LambdaCall lcall = new LambdaCall();
        lcall.setParser(this.clone());
        lcall.setLambda(clambda);
        lcall.setType(lambdaType);
        return lcall;
    }

    protected boolean isFunc( Class c ){
        if( c==null )return false;
        if( c.equals(Func0.class) )return true;
        if( c.equals(Func1.class) )return true;
        if( c.equals(Func2.class) )return true;
        if( c.equals(Func3.class) )return true;
        if( c.equals(Func4.class) )return true;
        if( c.equals(Func5.class) )return true;
        if( c.equals(Func6.class) )return true;
        if( c.equals(Func7.class) )return true;
        if( c.equals(Func8.class) )return true;
        if( c.equals(Func9.class) )return true;
        return false;
    }

    //<editor-fold defaultstate="collapsed" desc="functionCall()">
    protected FunctionCall functionCall( Pointer<Token> ptr, Function fun ){
        Class[] funParamTypes = fun.getParameters();

        FunctionCall fcall = new FunctionCall();
        fcall.setFunction(fun);

        for( int parami=0; parami<funParamTypes.length; parami++ ){
            Class funParamType = funParamTypes[parami];

            if( isFunc(funParamType) )
            {
                int lambdaParamCountNeed = 1;
                if( funParamType.equals(Func0.class) )lambdaParamCountNeed = 0;
                if( funParamType.equals(Func1.class) )lambdaParamCountNeed = 1;
                if( funParamType.equals(Func2.class) )lambdaParamCountNeed = 2;
                if( funParamType.equals(Func3.class) )lambdaParamCountNeed = 3;
                if( funParamType.equals(Func4.class) )lambdaParamCountNeed = 4;
                if( funParamType.equals(Func5.class) )lambdaParamCountNeed = 5;
                if( funParamType.equals(Func6.class) )lambdaParamCountNeed = 6;
                if( funParamType.equals(Func7.class) )lambdaParamCountNeed = 7;
                if( funParamType.equals(Func8.class) )lambdaParamCountNeed = 8;
                if( funParamType.equals(Func9.class) )lambdaParamCountNeed = 9;

                CatchedLambda clambda = catchLambda(ptr);
                if( clambda!=null && clambda.args.size()==lambdaParamCountNeed ){
                    Value v = createLambdaCall(clambda,funParamType);
                    if( v!=null ){
                        fcall.appendChild(v);
                        continue;
                    }
                }
            }

            Value v = operator(ptr,null);
            if( v==null ){
                logFine(
                    "can't parse argument "+parami+"\n"+"parse return null"
                );
                return null;
            }

            Class vt = v instanceof GetType
                       ? ((GetType)v).getType() : null;

            if( vt==null ){
                logWarning(
                    "parsed argument (argument index="+parami+") not implement GetType"
                );
                return null;
            }

            boolean assign = funParamType.isAssignableFrom(vt);
            if( assign ){
                fcall.appendChild(v);
            }else{
                TypeCastGraph tcgraph = getTypeCastGraph();

                List<Class> starts = tcgraph.findStartNode(vt, false, true, true, false);
                if( starts==null || starts.size()<1 )return null;

                List castPaths = tcgraph.getCastPaths(vt, funParamType);
                if( castPaths==null || castPaths.size()<1 )return null;

                TypeCastValue tcval = new TypeCastValue();

                tcval.setFromType(vt);
                tcval.setType(funParamType);
                tcval.setTypeCastGraph(getTypeCastGraph());
                tcval.appendChild(v);
                tcval.setImplicit(true);

                fcall.appendChild(tcval);
            }
        }

        return fcall;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="operator()">
    protected class OperatorScope {
        public Value v1;
        public boolean v1evaled;
        public Memory mem;
        public String op;
        public FunctionSetHelper unaryOp;
        public FunctionSetHelper binaryOp;
        public Value v2;
    }

    protected Stack<OperatorScope> operatorScope = new Stack<OperatorScope>();

    protected Value operator(
        Pointer<Token> ptr,
        Value v1
    ){
        OperatorScope scope = new OperatorScope();
        operatorScope.push(scope);

        scope.v1 = v1;

        try{
            Memory mem = getMemory();
            scope.mem = mem;

            Token t = ptr.lookup(0);
            String op = t!=null ? t.getMatchedText() : null;
            scope.op = op;
            if( op==null ){
                return null;
            }

            boolean v1evaled = false;

            if( v1==null ){
                v1 = postfixVal(ptr);
                if( v1==null )return null;
                v1evaled = true;
                scope.v1 = v1;
                scope.v1evaled = true;
            }

            if( isBracket(op) )return v1;
            if( isNextStatement(op) )return v1;

            if( v1evaled ){
                Token t2 = ptr.lookup(0);
                String t2s = t2==null ? null : t2.getMatchedText();
                if( t2s!=null ){
                    if( isBracket(t2s) )return v1;
                    if( isNextStatement(t2s) )return v1;
                    op = t2s;
                }else{
                    return v1;
                }
            }

            Class v1t = v1.getType();
            if( v1t==null )return null;

//            FunctionSetHelper fshlp = new FunctionSetHelper(mem);
//            fshlp = fshlp.named(op);

            FunctionSet fset = mem.getOperators(op);
            if( fset==null ){
                fset = new FunctionSetImpl();
            }

            List<Function> unaryFuns = new ArrayList<Function>();
            List<Function> binaryFuns = new ArrayList<Function>();

            int namedFunIdx = -1;
//            for( Function f : fshlp.functions() ){
            for( Function f : fset.getFunctions() ){
                if( f instanceof IsOperator ){
                    if( !((IsOperator)f).isOperator() ){
                        continue;
                    }
                }

                namedFunIdx++;

                Class[] params = f.getParameters();
                if( params.length==1 ){
                    Class param1 = params[0];
                    if( param1.isAssignableFrom(v1t) ){
                        trace( "named unary fun "+namedFunIdx+": "+f );
                        unaryFuns.add(f);
                    }
                }else if( params.length==2 ){
                    Class param1 = params[0];
                    if( param1.isAssignableFrom(v1t) ){
                        trace( "named binay fun "+namedFunIdx+": "+f );
                        binaryFuns.add(f);
                    }
                }
            }

            int unaryFunsCount = unaryFuns.size();
            int binaryFunsCount = binaryFuns.size();
            trace( "unary count ="+unaryFunsCount +" binary count ="+binaryFunsCount );

            if( binaryFunsCount==0 && unaryFunsCount==0 ){
                return v1;
            }

            if( binaryFunsCount<1 ){
                if( unaryFunsCount==1 ){
                    ptr.move(1);

                    FunctionCall fUnaryCall = new FunctionCall();
                    fUnaryCall.appendChild(v1);

                    Function f = unaryFuns.get(0);
                    fUnaryCall.setFunction(f);

                    trace("return unary "+f);
                    return fUnaryCall;
                }
                return v1;
            }

            ptr.push();
            ptr.move(1);

            Function firstBinary = binaryFuns.get(0);
            Class[] firstParams = firstBinary.getParameters();
            boolean binaryWaitLambda =
                binaryFunsCount==1 &&
                firstBinary!=null &&
                firstParams!=null &&
                firstParams.length==2 &&
                isFunc(firstParams[1]);

            if( binaryWaitLambda ){
                CatchedLambda clambda = catchLambda(ptr);
                if( clambda!=null ){
                    LambdaCall lc = createLambdaCall(clambda);

                    FunctionCall fBinaryCall = new FunctionCall();
                    fBinaryCall.appendChild(v1);
                    fBinaryCall.appendChild(lc);
                    fBinaryCall.setFunction(firstBinary);
                    ptr.pop();

                    trace("return binary with lambda "+firstBinary);

                    return fBinaryCall;
                }
            }

            Value v2 = operator(ptr,null);
            if( v2!=null ){
                Class v2t = v2.getType();
                if( v2t==null ){
                    ptr.restore();
                    if( unaryFunsCount==1 ){
                        ptr.move(1);
                        FunctionCall fUnaryCall = new FunctionCall();
                        fUnaryCall.appendChild(v1);
                        Function f = unaryFuns.get(0);
                        fUnaryCall.setFunction(f);
                        trace("return unary "+f);
                        return fUnaryCall;
                    }
                    return v1;
                }

                List<Function> explicitFuns = new ArrayList<Function>();
                for( Function f : binaryFuns ){
                    Class[] params = f.getParameters();
                    if( params.length==2 && params[1].isAssignableFrom(v2t) ){
                        explicitFuns.add(f);
                    }
                }

                int explicitFunsCount = explicitFuns.size();

                if( explicitFunsCount<1 ){
                    if( binaryFunsCount==1 ){
                        Function opfun = binaryFuns.get(0);
                        Class[] paramTypes = opfun.getParameters();

                        TypeCastValue tcval = new TypeCastValue();
                        tcval.setTypeCastGraph(getTypeCastGraph());
                        tcval.setFromType(v2.getType());
                        tcval.setType(paramTypes[1]);
                        tcval.setImplicit(true);
                        tcval.appendChild(v2);

                        if( !tcval.canCast() ){
                            ptr.restore();
//                            if( unaryOp.count()==1 ){
                            if( unaryFunsCount==1 ){
                                ptr.move(1);
                                FunctionCall fUnaryCall = new FunctionCall();
                                fUnaryCall.appendChild(v1);
//                                Function f = unaryOp.first();
                                Function f = unaryFuns.get(0);
                                fUnaryCall.setFunction(f);
                                trace("return unary "+f);
                                return fUnaryCall;
                            }
                            return v1;
                        }

                        FunctionCall fBinaryCall = new FunctionCall();
                        fBinaryCall.appendChild(v1);
                        fBinaryCall.appendChild(tcval);
                        fBinaryCall.setFunction(opfun);
                        ptr.pop();
                        trace("return binary implicit (explicit < 0) "+opfun);
                        return fBinaryCall;
                    }
                }

                if( explicitFunsCount>1 ){
                    ptr.restore();
                    if( unaryFunsCount==1 ){
                        ptr.move(1);
                        FunctionCall fUnaryCall = new FunctionCall();
                        fUnaryCall.appendChild(v1);
                        Function f = unaryFuns.get(0);
                        fUnaryCall.setFunction(f);
                        trace("return unary (explicit > 0) "+f);
                        return fUnaryCall;
                    }
                    return v1;
                }

                Function opfun = explicitFuns.get(0);

                FunctionCall fBinaryCall = new FunctionCall();
                fBinaryCall.appendChild(v1);
                fBinaryCall.appendChild(v2);
                fBinaryCall.setFunction(opfun);
                trace("return binary explicit = 1 "+opfun);
                ptr.pop();
                return fBinaryCall;
            }

            ptr.restore();

            if( unaryFunsCount==1 ){
                ptr.move(1);
                FunctionCall fUnaryCall = new FunctionCall();
                fUnaryCall.appendChild(v1);
                Function f = unaryFuns.get(0);
                fUnaryCall.setFunction(f);
                trace("return unary (v2 not parsed) "+f);
                return fUnaryCall;
            }

            return v1;
        }
        finally{
            operatorScope.pop();
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="bracket()">
    protected boolean isBracket( String s ){
        for( Map.Entry<String,String> b : getBrackets().entrySet() ){
            String b1 = b.getKey();
            String b2 = b.getValue();
            if( s.equals(b1) )return true;
            if( s.equals(b2) )return true;
        }
        return false;
    }

    protected static class BracketValue {
        public String openBracket;
        public String closeBracket;
        public Value body;
    }

//    private boolean Br

    protected BracketValue bracket( Pointer<Token> ptr ){
        if( ptr==null )throw new IllegalArgumentException( "ptr==null" );

        Token t = ptr.lookup(0);
        String s = t!=null ? t.getMatchedText() : null;
        if( s==null )return null;

        String bend = getBrackets().get(s);
        String bstart = bend==null ? null : s;

        if( bend==null || bstart==null )return null;

        ptr.push();
        ptr.move(1);

        Value v = statements(ptr);
        if( v!=null ){
            Token t2 = ptr.lookup(0);
            String s2 = t2!=null ? t2.getMatchedText() : null;
            if( s2!=null && s2.equals(bend) ){
                ptr.move(1);
                ptr.pop();

                BracketValue bv = new BracketValue();
                bv.body = v;
                bv.openBracket = s;
                bv.closeBracket = s2;

                return bv;
            }
        }

        ptr.restore();
        //TODO message require right bracket
        return null;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="postfixVal()">
    protected Value postfixVal( Pointer<Token> ptr ){
        Value v = val(ptr);

        Token t = ptr.lookup(0);
        if( t==null )return v;

        String ts = t.getMatchedText();
//        Object tm = getMemory().get(ts);
        Object tm = getMemory().getOperators(ts);
        if( tm instanceof Function || tm instanceof FunctionSet ){
            Value vo = operator(ptr, v);
            if( vo != null ){
                return vo;
            }
        }

        return v;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="val()">
    protected Value val( Pointer<Token> ptr ){
//        CatchedLambda clambda = catchLambda(ptr);
//        if( clambda!=null ){
//            Class funType = clambda.getFunType();
//            if( funType!=null ){
//                LambdaCall lc = createLambdaCall(clambda, funType);
//                return lc;
//            }
//        }

        BracketValue bracketv = bracket(ptr);

        if( bracketv!=null && bracketv.body!=null ){
//            if( bracketv.openBracket!=null && bracketv.openBracket.equals("{") ){
//                Block block = new Block();
//                block.setBody(bracketv.body);
//                return block;
//            }

            return bracketv.body;
        }

        Value assignv = writeMemValue(ptr);
        if( assignv!=null )return assignv;

        Value callv = callValue(ptr);
        if( callv!=null )return callv;

        Value readv = readMemValue(ptr);
        if( readv!=null )return readv;

        Value numv = numberConst(ptr);
        if( numv!=null )return numv;

        Value strv = stringConst(ptr);
        if( strv!=null )return strv;

        return null;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="DefineVarPtr">
    protected static class DefineVarPtr{
        public int indexBegin;
        public int indexEnd;
        public FieldAccess fieldAccess;
//        public AssignVar defVar;
        public DefineVar defVar;
//        public Value dataCode;
//        public Class dataType;
//        public Memory memory;
    }

    protected final List<DefineVarPtr> defineVars = new ArrayList<DefineVarPtr>();
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="writeMemValue()">
    protected Value writeMemValue( Pointer<Token> ptr ){
        Token t = ptr.lookup(0);
        String varn = t!=null ? t.getMatchedText() : null;
        if( varn==null )return null;

        FieldAccessParser faParser = new FieldAccessParser();
        FieldAccess fa = faParser.parse(varn, 0);
        if( fa==null )return null;
        if( !fa.getMatchedText().equals(varn) )return null;

        int indexBegin = ptr.getIndex();

        t = ptr.lookup(1);
        String assignTokVal = t!=null ? t.getMatchedText() : null;
        if( assignTokVal==null )return null;
        if( assignOpertator==null )return null;
        if( !assignOpertator.equals(assignTokVal) )return null;

        ptr.push();
        ptr.move(2);

        final Value newv
//            = val(ptr);
            = operator(ptr,null);
        if( newv==null ){
            ptr.restore();
            return null;
        }

        ptr.pop();

//        final Memory mem = getMemory();
//        final String varname = varn;
        AssignVar assignv = new AssignVar();
        assignv.setData(newv);
        assignv.setField(fa);
//        assignv.setMemory(memory);
        assignv.setMemory(getMemory());

        int indexEnd = ptr.getIndex();

        DefineVarPtr aval = new DefineVarPtr();
        aval.defVar = assignv;
//        aval.memory = mem;
//        aval.dataCode = newv;
//        aval.dataType = newv.getType();
        aval.fieldAccess = fa;
        aval.indexBegin = indexBegin;
        aval.indexEnd = indexEnd;

        defineVars.add(aval);

        return assignv;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="readMemValue()">
    protected Value readMemValue( Pointer<Token> ptr ){
        Token t = ptr.lookup(0);
        String varname = t!=null ? t.getMatchedText() : null;
        if( varname==null )return null;

        FieldAccessParser faParser = new FieldAccessParser();
        FieldAccess fa = faParser.parse(varname, 0);
        if( fa==null )return null;

        for( int ai=defineVars.size()-1; ai>=0; ai-- ){
            DefineVarPtr defvarptr = defineVars.get(ai);
            if( defvarptr.fieldAccess==null )continue;
            if( !defvarptr.fieldAccess.toString().equals(fa.toString()) )continue;

            if( defvarptr.defVar==null )continue;

            ReadVar rval = new ReadVar();
            rval.setData(defvarptr.defVar);
            ptr.move(1);
            return rval;
        }

        boolean matchtxt = fa.getMatchedText().equals(varname);
        if( !matchtxt )return null;

        final List resolved = new ArrayList();
//        boolean resolvedRes =
        resolveFieldAccess(getMemory(), fa, new Func1<Object, Object>() {
            @Override
            public Object apply(Object arg) {
                resolved.add(arg);
                return null;
            }
        });

        Object v = resolved.isEmpty() ? null : resolved.get(0);
        if( v==null )return null;

        ConstValue cv = new ConstValue(v);
        ptr.move(1);
        return cv;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="resolveFieldAccess()">
    protected boolean resolveFieldAccess( Memory mem, FieldAccess fa, Func1<Object,Object> readedValue ){
        if( fa==null )return false;
        if( mem==null )return false;
        if( readedValue==null )return false;

        String[] path = fa.getAccessPath();
        if( path==null || path.length==0 )return false;

        Object root = mem.get(path[0]);
        if( path.length==1 ){
            readedValue.apply(root);
            return true;
        }

        String[] newpath = new String[path.length-1];
        for( int i=1; i<path.length; i++ ){
            newpath[i-1] = path[i];
        }

        return resolveFieldAccess( root, newpath, readedValue );
    }

    protected boolean resolveFieldAccess( Object root, String[] path, Func1<Object,Object> readedValue ){
        if( root==null )return false;
        if( path==null )return false;

        if( path.length==0 ){
            return false;
        }

        BeanInfo bi = null;
        try {
            bi = Introspector.getBeanInfo(root.getClass());
        } catch (IntrospectionException ex) {
            Logger.getLogger(Parser.class.getName()).log(Level.SEVERE, null, ex);
            return false;
        }

        Object newroot = null;
        boolean readed = false;
        try {
            for( PropertyDescriptor pd : bi.getPropertyDescriptors() ){
                if( !pd.getName().equals(path[0]) )continue;

                if( pd.getPropertyType()==null ){
                    return false;
                }

                Method m = pd.getReadMethod();
                if( m==null )return false;

                newroot = m.invoke(root);
                readed = true;

                break;
            }

            if( !readed ){
                for( Field f : root.getClass().getFields() ){
                    if( !f.getName().equals(path[0]) )continue;

                    newroot = f.get(root);
                    readed = true;
                    break;
                }
            }
        } catch (IllegalAccessException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return false;
        } catch (IllegalArgumentException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return false;
        } catch (InvocationTargetException ex) {
            Logger.getLogger(AssignVar.class.getName()).log(Level.SEVERE, null, ex);
            return false;
        }

        if( !readed && (root instanceof Map) ){
            Map m = (Map)root;
            if( m.containsKey(path[0]) ){
                newroot = m.get(path[0]);
                readed = true;
            }
        }

        if( path.length==1 && readed ){
            readedValue.apply(newroot);
            return true;
        }

        if( !readed ){
            return false;
        }

        String[] newpath = new String[path.length-1];
        for( int i=1; i<path.length; i++ ){
            newpath[i-1] = path[i];
        }

        return resolveFieldAccess(newroot, newpath, readedValue);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="callValue()">
    protected Value callValue( Pointer<Token> ptr ){
//        String s = ptr.lookup(0);
        Token t = ptr.lookup(0);
        String s = t!=null ? t.getMatchedText() : null;
        if( s==null )return null;

        if( s.startsWith("\"") )return null;
        if( s.startsWith("\'") )return null;
        if( s.matches("(?is)\\d+[\\d\\.]*") )return null;
        if( isBracket(s) )return null;

        Memory mem = getMemory();

//        Object mval = mem.get(s);
        FunctionSet fset = mem.getFunctions(s);

//        if( mval==null )return null;
//        if( mval==null ){
        if( fset==null ){
            if( s.contains(".") ){
                FieldAccessParser faParser = new FieldAccessParser();
                FieldAccess fa = faParser.parse(s, 0);
                if( fa!=null && fa.getMatchedText().equals(s) ){
                    final List resolved = new ArrayList<Object>();
                    Boolean readed = resolveFieldAccess(mem, fa, new Func1<Object, Object>() {
                        @Override
                        public Object apply(Object arg) {
                            resolved.add(arg);
                            return null;
                        }
                    });
                    if( readed && resolved!=null && resolved.size()>0 ){
//                        mval = resolved.get(0);
                        Object r = resolved.get(0);
                        if( r instanceof FunctionSet ){
                            fset = (FunctionSet)r;
                        }else if( r instanceof Function ){
                            FunctionSetImpl fsetimpl = new FunctionSetImpl();
                            fsetimpl.add((Function)r);
                            fset = fsetimpl;
                        }
                    }
//                    if( mval==null )return null;
                    if( fset==null )return null;
                }else{
                    return null;
                }
            }else{
                return null;
            }
        }

//        if( mval instanceof Function ){
        if( fset instanceof Function ){
            ptr.push();
            ptr.move(1);

//            Token tb = ptr.lookup(0);

//            String bracketBegin = null;
//            if( tb!=null && isBracket(tb.getMatchedText()) ){
////                vb = bracket(ptr);
//            }

//            Value v = functionCall(ptr, (Function)mval);
            Value v = functionCall(ptr, (Function)fset);
            if( v!=null ){
                ptr.pop();
                return v;
            }
            ptr.restore();
//        } else if( mval instanceof FunctionSet ){
        } else if( fset instanceof FunctionSet ){
            ptr.push();
            ptr.move(1);

            Comparator<FunctionCall> cfcall = new Comparator<FunctionCall>() {
                @Override
                public int compare(FunctionCall fc1, FunctionCall fc2) {
                    int countImpl_1 = fc1.evalImplicitCastCount();
                    int countImpl_2 = fc2.evalImplicitCastCount();
                    int cmpImpl = countImpl_1 < countImpl_2 ? -1 : ( countImpl_1 == countImpl_2 ? 0 : 1 );

                    int callArgCnt_1 = fc1.getChildren().length;
                    int callArgCnt_2 = fc2.getChildren().length;
                    int cmpArgCnt = callArgCnt_1==callArgCnt_2
                                    ? 0
                                    : (callArgCnt_1 < callArgCnt_2 ? -1 : 1);

                    cmpArgCnt = -cmpArgCnt;

//                    if( cmpArgCnt!=0 )return cmpArgCnt;
//                    return cmpImpl;

                    if( cmpImpl!=0 )return cmpImpl;
                    return cmpArgCnt;
                }
            };

            List<FunctionCall> tfcall = new ArrayList<FunctionCall>();
            Map<FunctionCall,Integer> nextPtr = new LinkedHashMap<FunctionCall, Integer>();

            FunctionCall fcall = null;
//            for( Function f : ((FunctionSet)mval).getFunctions() ){
            for( Function f : ((FunctionSet)fset).getFunctions() ){
                ptr.push();

                fcall = functionCall(ptr, f);
                if( fcall!=null ){
                    tfcall.add(fcall);
                    nextPtr.put(fcall, ptr.getIndex());
                    ptr.restore();
                }else{
                    ptr.restore();
                }
            }

            if( tfcall.size()>1 ){
                Collections.sort(tfcall, cfcall);
            }

            int fcall2i = -1;
            for( FunctionCall fcall2 : tfcall ){
                fcall2i++;

                SourceDump sdump = new SourceDump();
                StringWriter sw = new StringWriter();
                sdump.output(sw);
                sdump.dump(fcall2);

                trace( "call variant "+fcall2i+": \n"+Text.indent(sw.toString(), "  ") );
            }

            Integer nextPTR = null;
            if( !tfcall.isEmpty() ){
                fcall = tfcall.get(0);
                nextPTR = nextPtr.get(fcall);
            }

            if( fcall!=null ){
                ptr.pop();
                if( nextPTR!=null )ptr.setIndex(nextPTR);
                return fcall;
            }else{
                ptr.restore();
            }
        }

//        return new ConstValue(mval);
        return null;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="numberConst()">
    protected Value numberConst( Pointer<Token> ptr ){
//        String s = ptr.lookup(0);
        Token t = ptr.lookup(0);
        String s = t!=null ? t.getMatchedText() : null;
        if( s==null )return null;

        if( isBracket(s) )return null;

        if( !s.matches("(?is)^-?\\d+(\\.\\d+)?$") ){
            return null;
        }

        NumberConstParser numConstPrs = new NumberConstParser(10, s.contains("."));
        NumberConst nconst = numConstPrs.parse(s, 0);
        if( nconst==null )return null;

        ptr.move(1);

        Number num = nconst.getNumber();
        if( num instanceof Long ){
            num = ((Long)num).intValue();
        }

        ConstValue cval = new ConstValue(num);
        return cval;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="stringConst()">
    protected Value stringConst( Pointer<Token> ptr ){
//        String s = ptr.lookup(0);
        Token t = ptr.lookup(0);
        String s = t!=null ? t.getMatchedText() : null;
        if( s==null )return null;

        if( isBracket(s) )return null;

        boolean decoded = false;
        if( s.startsWith("\"") || s.startsWith("\'") ){
            TextConstParser parser = new TextConstParser();
            TextConst tconst = parser.parse(s, 0);
            if( tconst!=null ){
                s = tconst.getDecodedText();
                decoded = true;
            }
        }

        ConstString cval = new ConstString(s,ptr.getIndex(),decoded);
        ptr.move(1);
        return cval;
    }
//</editor-fold>
}
