/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.cli;


import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.text.EndLine;
import xyz.cofe.text.EndLineReWriter;
import xyz.cofe.text.Output;
import xyz.cofe.text.Text;
import xyz.cofe.text.parser.Token;
import xyz.cofe.text.parser.lex.FieldAccess;

/**
 *
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class SourceDump {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(SourceDump.class.getName());
    private static final Level logLevel = logger.getLevel();

    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    private final Output out;
    private final EndLineReWriter endlReWriter;
    private final Output outDest;
    private int level = 0;

    public SourceDump(){
        outDest = new Output( new OutputStreamWriter(System.out) );
        endlReWriter = new EndLineReWriter(outDest){
            protected boolean indentWrited = false;

            protected void writeIndent(boolean checkWrited) throws IOException{
                if( checkWrited ){
                    if( indentWrited )return;
                    this.writer.append(getIndent());
                    indentWrited = true;
                }else{
                    this.writer.append(getIndent());
                    indentWrited = true;
                }
            }

            @Override
            protected void writeEndLine() throws IOException {
                if( this.writer!=null ){
                    this.writer.append(EndLine.Default.get());
                    indentWrited = false;
                }
            }

            @Override
            protected void writeChar(char c0) throws IOException {
                writeIndent(true);
                super.writeChar(c0);
            }

            @Override
            public Writer append(char c) throws IOException {
                writeIndent(true);
                return super.append(c);
            }

            @Override
            public Writer append(CharSequence csq, int start, int end) throws IOException {
                writeIndent(true);
                return super.append(csq, start, end);
            }

            @Override
            public Writer append(CharSequence csq) throws IOException {
                writeIndent(true);
                return super.append(csq);
            }

            @Override
            public void write(String str, int off, int len) throws IOException {
                writeIndent(true);
                super.write(str, off, len);
            }

            @Override
            public void write(String str) throws IOException {
                writeIndent(true);
                super.write(str);
            }

            @Override
            public void write(char[] cbuf) throws IOException {
                writeIndent(true);
                super.write(cbuf);
            }

            @Override
            public void write(int c) throws IOException {
                writeIndent(true);
                super.write(c);
            }
        };
        out = new Output(endlReWriter);
    }

    protected synchronized String getIndent(){
        if( level<=0 )return "";
        return Text.repeat("|---", level);
    }

    public synchronized void output( Writer writer ){
        if( writer==null )throw new IllegalArgumentException( "writer==null" );
//        out = new IndentStackWriter(writer);
//        out = new Output(writer);
        outDest.reset(writer);
    }

    public synchronized void dump( Value val ){
        if( val instanceof AssignVar ){
            dump( (AssignVar)val );
        }else if( val instanceof LambdaCall ){
            dump( (LambdaCall)val );
        }else if( val instanceof Block ){
            dump( (Block)val );
        }else if( val instanceof ReadVar ){
            dump( (ReadVar)val );
        }else if( val instanceof ConstString ){
            dump( (ConstString)val );
        }else if( val instanceof ConstValue ){
            dump( (ConstValue)val );
        }else if( val instanceof Sequence ){
            dump( (Sequence)val );
        }else if( val instanceof TypeCastValue ){
            dump( (TypeCastValue)val );
        }else if( val instanceof FunctionCall ){
            dump( (FunctionCall)val );
        }else{
            if( val!=null ){
                Value[] childrn = val.getChildren();
                out.println(val.toString());
                for( Value vc : childrn ){
                    level++;
                    dump(vc);
                    level--;
                }
            }else{
                out.println("null");
            }
        }

        out.flush();
    }

    //<editor-fold defaultstate="collapsed" desc="memory">
    private Memory memory;

    public Memory getMemory() {
        if( memory!=null )
            return memory;
        memory = new Memory();
        return memory;
    }

    public void setMemory(Memory memory) {
        this.memory = memory;
    }
//</editor-fold>

    protected synchronized void dump( LambdaCall lc ){
        if( lc==null )throw new IllegalArgumentException( "lc==null" );
        out.println("lambda call:");
        if( lc.lambda!=null ){
            if( lc.lambda.args!=null ){
                int argi = -1;
                out.println("arguments:");
                for( String arg : lc.lambda.args ){
                    argi++;
                    out.print("  ");
                    out.print(argi);
                    out.print(". ");
                    out.println(arg);
                }
            }
            out.println("body:");
            if( lc.lambda.body!=null ){
                List<String> toks = new ArrayList<String>();
                for( Token t : lc.lambda.body ){
                    if( t!=null )toks.add( t.getMatchedText() );
                }
                out.println( "  "+Text.join(toks, " ") );
            }
        }
    }
    
    //<editor-fold defaultstate="collapsed" desc="dump ReadVar">
    protected synchronized void dump( ReadVar rval ){
        if( rval==null )throw new IllegalArgumentException( "rval==null" );
        
//        AssignVar avar = rval.getData();
        DefineVar defvar = rval.getData();
        if( defvar==null ){
            out.println("read variable ...");
            return;
        }
        
//        FieldAccess fa = avar==null ? null : avar.getField();
//        String varn = fa==null ? null : fa.toString();
//        out.println( "read variable "+varn );
        
        String varn = defvar.getVarName();
        Class vart = defvar.getVarType();
        out.println( "read variable "+varn+(vart==null ? "" : " : "+vart.getName()) );
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="dump Block">
    protected synchronized void dump( Block bval ){
        if( bval==null )throw new IllegalArgumentException( "bval==null" );
        
        out.println( "block" );
        
        Value[] childrn = bval.getChildren();
        for( Value vc : childrn ){
            level++;
            dump(vc);
            level--;
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="dump AssignVar">
    protected synchronized void dump( AssignVar avar ){
        if( avar==null )throw new IllegalArgumentException( "avar==null" );
        
        FieldAccess fa = avar==null ? null : avar.getField();
        String varn = fa==null ? null : fa.toString();
        
        out.println( "assign "+varn );
        
        Value[] childrn = avar.getChildren();
        for( Value vc : childrn ){
            level++;
            dump(vc);
            level--;
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="dump Sequence">
    protected synchronized void dump( Sequence sval ){
        if( sval==null )throw new IllegalArgumentException( "sval==null" );
        out.println( "sequence" );
        
        Value[] childrn = sval.getChildren();
        for( Value vc : childrn ){
            level++;
            dump(vc);
            level--;
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="dump ConstString">
    protected synchronized void dump( ConstString constval ){
        if( constval==null )throw new IllegalArgumentException( "cval==null" );
        
        Object cval = constval.getConst();
        
        if( cval==null ){
            out.println("null");
            return;
        }
        
        if( cval instanceof String ){
            out.println(
                Text.encodeStringConstant((String)cval) + " : String"
            );
            return;
        }
        out.println( cval );
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="dump ConstValue">
    protected synchronized void dump( ConstValue constval ){
        if( constval==null )throw new IllegalArgumentException( "cval==null" );
        
        Object cval = constval.getConst();
        
        if( cval==null ){
            out.println("null");
            return;
        }
        
        if( cval instanceof String ){
            out.println(
                Text.encodeStringConstant((String)cval) + " : String"
            );
            return;
        }
        if( cval instanceof Boolean ){
            if( (Boolean)cval ){
                out.println("true : Boolean");
            }else{
                out.println("false : Boolean");
            }
            return;
        }
        out.println( cval );
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="dump TypeCastValue">
    protected synchronized void dump( TypeCastValue tc ){
        if( tc==null )throw new IllegalArgumentException( "tc==null" );
        
        Value val = tc.getValue();
        if( val==null ){
            out.println("!cast(null)");
            return;
        }
        
        Class ftype = tc.getFromType();
        if( ftype==null && val instanceof ConstValue ){
            ftype = ((ConstValue)val).getType();
        }
        
        Class ttype = tc.getType();
        if( ttype==null ){
            out.println("!cast(target=null)");
            return;
        }
        
        if( ttype==Void.class ){
            out.println("cast as void");
            level++;
            dump(val);
            level--;
            return;
        }
        
        String ttname = new Declaration().getTypeName(ttype);
        
        out.println( "cast as "+ttname+(tc.isImplicit() ? " implicit" : "") );
        level++;
        dump(val);
        level--;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="dump FunctionCall">
    protected synchronized void dump( FunctionCall fcall ){
        if( fcall==null )throw new IllegalArgumentException( "fcall==null" );
        
        Function fun = fcall.getFunction();
        String funNameR = getMemory().getNameOf(fun);
        String funName = funNameR!=null ? funNameR : "undefined";
        
        StringBuilder fundecl = new StringBuilder();
        if( fun!=null ){
            fundecl.append("(");
            Class[] paramTypes = fun.getParameters();
            if( paramTypes.length>0 ){
                fundecl.append(" ");
                for( int pi=0; pi<paramTypes.length; pi++ ){
                    if( pi>0 )fundecl.append(", ");
                    String argName = "arg"+pi;
                    String argTypeName = new Declaration().getTypeName(paramTypes[pi]);
                    fundecl.append(argName).append(":").append(argTypeName);
                }
                fundecl.append(" ");
            }
            fundecl.append(")");
            
            Class retType = fun.getReturn();
            fundecl.append(":").append(new Declaration().getTypeName(retType));
        }
        
        out.println("call function "+funName+fundecl.toString());
        
        // implicit count
        level++;
        int c = fcall.evalImplicitCastCount();
        if( c>0 ){
            out.println("implicit cast count = "+c);
        }
        level--;
        
        // args:
        int pi = -1;
        for( Value cval : fcall.getChildren() ){
            pi++;
            level++;
            out.print("arg ");
            out.print(pi);
            out.print(": ");
            dump(cval);
            level--;
        }
    }
//</editor-fold>
}
