/*
 * The MIT License
 *
 * Copyright 2016 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.cli;


import java.io.StringWriter;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.Stack;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Pointer;
import xyz.cofe.text.Output;
import xyz.cofe.text.Text;
import xyz.cofe.text.parser.Token;

/**
 *
 * @author nt.gocha@gmail.com
 */
public class TraceParser
extends Parser
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(TraceParser.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    public TraceParser() {
    }

    //<editor-fold defaultstate="collapsed" desc="clone()">
    public TraceParser( TraceParser source ) {
        super(source);
        if( source!=null ){
            this.options = source.getTraceOptions().clone();
        }
    }

    @Override
    public TraceParser clone() {
        return new TraceParser(this);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="options">
    public class Options {
        public Options(){
        }

        //<editor-fold defaultstate="collapsed" desc="clone">
        public Options(Options source){
            if( source==null )throw new IllegalArgumentException( "source==null" );
            this.output = source.output;
            this.closeOutput = source.closeOutput;
            this.ptrLookupCount = source.ptrLookupCount;
            if( source.methods!=null ){
                this.methods = new LinkedHashSet<String>();
                this.methods.addAll(source.methods);
            }
        }

        @Override
        public Options clone(){
            return new Options(this);
        }
//</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="output">
        protected Output output = null;

        public Output getOutput() {
            try{
                lock.lock();
                return output;
            }finally{
                lock.unlock();
            }
        }

        public void setOutput( Output output ) {
            try{
                lock.lock();
                this.output = output;
            }finally{
                lock.unlock();
            }
        }
//</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="closeOutput">
        private boolean closeOutput = false;

        public boolean isCloseOutput() {
            return closeOutput;
        }

        public void setCloseOutput(boolean closeOutput) {
            this.closeOutput = closeOutput;
        }
//</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="ptrLookupCount">
        protected int ptrLookupCount = 3;

        public int getPtrLookupCount() {
            try{
                lock.lock();
                return ptrLookupCount;
            }finally{
                lock.unlock();
            }
        }

        public void setPtrLookupCount( int ptrLookupCount ) {
            try{
                lock.lock();
                this.ptrLookupCount = ptrLookupCount;
            }finally{
                lock.unlock();
            }
        }
//</editor-fold>

        //<editor-fold defaultstate="collapsed" desc="methods">
        protected Set<String> methods;

        public Set<String> getMethods() {
            if( methods==null ){
                methods = new LinkedHashSet<String>();
                methods.add("parse");
                methods.add("bracket");
                methods.add("callValue");
                methods.add("functionCall");
                methods.add("stringConst");
                methods.add("numberConst");
                methods.add("readMemValue");
                methods.add("writeMemValue");
                methods.add("val");
                methods.add("postfixVal");
                methods.add("operator");
                methods.add("statements");
            }
            return methods;
        }
//</editor-fold>
    }

    //<editor-fold defaultstate="collapsed" desc="options">
    private Options options;

    public Options getTraceOptions() {
        try{
            lock.lock();
            if( options==null )options = new Options();
            return options;
        }finally{
            lock.unlock();
        }
    }
//</editor-fold>
//</editor-fold>

    protected Stack<String> callStack = new Stack<String>();

    //<editor-fold defaultstate="collapsed" desc="traceCallStack( out )">
    protected void traceCallStack( Output out ){
        if( out==null )return;
        if( callStack==null )return;
        if( callStack.isEmpty() )return;
        if( callStack.size()==1 ){
            out.print(callStack.get(0));
            return;
        }
        out.print( Text.repeat("|---",callStack.size()-1) );
        out.print( callStack.get(callStack.size()-1) );
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="tracePointer( out, ptr )">
    protected void tracePointer( Output out, Pointer<Token> ptr ){
        if( out==null )return;
        if( ptr==null )return;

        Options opts = getTraceOptions();
        if( opts==null )return;

        int count = opts.getPtrLookupCount();
        if( count<1 )return;

        for( int c=0; c<count; c++ ){
            Token t = ptr.lookup(c);
            if( t==null )break;

            out.print("[");
            out.print(ptr.getIndex()+c);
            out.print(":");
            if( t!=null ){
                String s = t.getMatchedText();
                if( s.length()>30 ){
                    out.print(Text.encodeStringConstant(s.substring(0, 30))+"...");
                }else{
                    out.print(Text.encodeStringConstant(s));
                }
            }
            out.print("]");
        }
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="traceMethodEnter( method )">
    protected void traceMethodEnter( String method ){
        if( method!=null )
            callStack.push(method);

        Options opts = getTraceOptions();
        Output out = opts!=null ? opts.getOutput() : null;
        if( out==null )return;

        if( method!=null && opts.getMethods().contains(method) ){
            traceCallStack(out);
            out.println();
        }
        
        out.flush();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="traceMethodEnter( method, ptr )">
    protected void traceMethodEnter( String method, Pointer<Token> ptr ){
        if( method!=null )
            callStack.push(method);

        Options opts = getTraceOptions();
        Output out = opts!=null ? opts.getOutput() : null;
        if( out==null )return;

        if( method!=null && opts.getMethods().contains(method) ){
            traceCallStack(out);

            out.print(": ");
            tracePointer(out, ptr);

            out.println();
        }
        
        out.flush();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="traceMethodEnter( method, ptr, fun )">
    protected void traceMethodEnter( String method, Pointer<Token> ptr,Function fun ){
        if( method!=null )
            callStack.push(method);

        Options opts = getTraceOptions();
        Output out = opts!=null ? opts.getOutput() : null;
        if( out==null )return;

        if( method!=null && opts.getMethods().contains(method) ){
            traceCallStack(out);

            out.print(": ");
            tracePointer(out, ptr);
            
            out.print(" fun=");
            out.print(fun);

            out.println();
        }
        
        out.flush();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="traceMethodEnter( method, ptr, value )">
    protected void traceMethodEnter( String method, Pointer<Token> ptr,Value v ){
        if( method!=null )
            callStack.push(method);

        Options opts = getTraceOptions();
        Output out = opts!=null ? opts.getOutput() : null;
        if( out==null )return;

        if( method!=null && opts.getMethods().contains(method) ){
            traceCallStack(out);

            out.print(">> ");
            tracePointer(out, ptr);

            out.println();
        }
        
        out.flush();
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="traceMethodExit()">
    protected void traceMethodExit( Object result ){
        String method = null;
        if( !callStack.isEmpty() ){
            method = callStack.pop();
        }

        Options opts = getTraceOptions();
        Output out = opts!=null ? opts.getOutput() : null;
        if( out==null )return;

        if( method!=null && opts.getMethods().contains(method) ){
            traceCallStack(out);

            if( result!=null ){
                out.print("= ");
                if( result instanceof Value ){
                    Value v = ((Value)result);

                    Class type = v.getClass();
                    String typeName = type.getName();

                    if( typeName.startsWith("xyz.cofe.cli.") ){
                        typeName = typeName.substring("xyz.cofe.cli.".length());
                    }

                    Class valResType = v.getType();

                    out.print(
                        typeName + ":" + valResType.getSimpleName()
                    );

                    if( v instanceof ConstValue ){
                        Object constval = ((ConstValue)v).getConst();
                        if( constval==null ){
                            out.print(" (null)");
                        }else{
                            if( constval instanceof String ){
                                String strConstVal = (String)constval;
                                if( strConstVal.length()>30 ){
                                    out.print( " ("+
                                               Text.encodeStringConstant(strConstVal.substring(0,30))+"..."
                                               +")"
                                    );
                                }else{
                                    out.print( " ("+
                                               Text.encodeStringConstant(strConstVal)
                                               +")"
                                    );
                                }
                            }else{
                                String str = constval.toString();
                                str.replaceAll("\\r\\n|\\n\\r|\\r|\\n|\\t", " ");
                                if( str.length()>30 ){
                                    str = str.substring(0, 30)+"...";
                                }
                                out.print(" ("+str+")");
                            }
                        }
                    }
                }else{
                    out.print( result.toString() );
                }
            }else{
                out.print("= null");
            }

            out.println();
        }
        
        out.flush();
    }
//</editor-fold>

    @Override
    protected void trace(String message) {
        if( message!=null ){
            Options opts = getTraceOptions();
            Output out = opts!=null ? opts.getOutput() : null;
            if( out==null )return;
            
            StringWriter sw = new StringWriter();
            Output o = new Output(sw);
            traceCallStack(o);
            o.flush();
            
            String l = Text.indent(message, sw.toString()+" ");
            out.println( l );
        }
        super.trace(message);
    }

    //<editor-fold defaultstate="collapsed" desc="trace - parser">
    @Override
    public Value parse( Pointer<Token> ptr ) {
        traceMethodEnter("parse", ptr);
        
        trace("tokens count="+ptr.getList().size());
        int ti = -1;
        for( Token t : ptr.getList() ){
            ti++;
            trace(""+(ti+1)+": "+Text.encodeStringConstant(t.getMatchedText()));
        }
        
        Value val = super.parse(ptr);
        traceMethodExit(val);

        Options opts = getTraceOptions();
        Output out = opts!=null ? opts.getOutput() : null;
        if( out!=null ){
            out.flush();
        }

        return val;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - bracket">
    @Override
    protected BracketValue bracket( Pointer<Token> ptr ) {
        traceMethodEnter("bracket", ptr);
        BracketValue val = super.bracket(ptr);
        traceMethodExit(val);
        return val;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - callValue">
    @Override
    protected Value callValue( Pointer<Token> ptr ) {
        traceMethodEnter("call", ptr);
        Value val = super.callValue(ptr);
        traceMethodExit(val);
        return val;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - functionCall">
    @Override
    protected FunctionCall functionCall( Pointer<Token> ptr, Function fun ) {
        traceMethodEnter("functionCall", ptr, fun);
        FunctionCall res = super.functionCall(ptr, fun);
        traceMethodExit(res);
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - stringConst">
    @Override
    protected Value stringConst( Pointer<Token> ptr ) {
        traceMethodEnter("stringConst", ptr);
        Value res = super.stringConst(ptr);
        traceMethodExit(res);
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - numberConst">
    @Override
    protected Value numberConst( Pointer<Token> ptr ) {
        traceMethodEnter("numberConst", ptr);
        Value res = super.numberConst(ptr);
        traceMethodExit(res);
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - readMemValue">
    @Override
    protected Value readMemValue( Pointer<Token> ptr ) {
        traceMethodEnter("readMemValue", ptr);
        Value res = super.readMemValue(ptr);
        traceMethodExit(res);
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - writeMemValue">
    @Override
    protected Value writeMemValue( Pointer<Token> ptr ) {
        traceMethodEnter("writeMemValue", ptr);
        Value res = super.writeMemValue(ptr);
        traceMethodExit(res);
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - val">
    @Override
    protected Value val( Pointer<Token> ptr ) {
        traceMethodEnter("val", ptr);
        Value res = super.val(ptr);
        traceMethodExit(res);
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - postfixVal">
    @Override
    protected Value postfixVal( Pointer<Token> ptr ) {
        traceMethodEnter("postfixVal", ptr);
        Value res = super.postfixVal(ptr);
        traceMethodExit(res);
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - operator">
    @Override
    protected Value operator( Pointer<Token> ptr, Value v1 ) {
        traceMethodEnter("operator", ptr, v1);
        Value res = super.operator(ptr, v1);
        traceMethodExit(res);
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="trace - statements">
    @Override
    protected Value statements( Pointer<Token> ptr ) {
        traceMethodEnter("statements", ptr);
        Value res = super.statements(ptr);
        traceMethodExit(res);
        return res;
    }
    //</editor-fold>
}
