/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection;

import java.math.BigInteger;

/**
 * Набор общих предикатов
 * @author gocha
 */
public class Predicates
{
    /**
     * Предикат возвращает true, если НЕ нулевая ссылка
     * @param <T> Тип объекта
     * @return предикат
     */
    public static <T> Predicate<T> isNotNull()
    {
        return new Predicate<T>() {
            @Override
            public boolean validate(T value) {
                return value!=null;
            }
        };
    }

    /**
     * Предикат возвращает true, если нулевая ссылка
     * @param <T> Тип объекта
     * @return предикат
     */
    public static <T> Predicate<T> isNull()
    {
        return new Predicate<T>() {
            @Override
            public boolean validate(T value) {
                return value==null;
            }
        };
    }

    /**
     * Предикат возвращает true, если объект находиться в указанной последовательности
     * @param <T> Тип объекта
     * @param src Последовательность
     * @return предикат
     */
    public static <T> Predicate<T> in(Iterable src)
    {
        if (src == null) {
            throw new IllegalArgumentException("src == null");
        }
        final Iterable fsrc = src;
        return new Predicate<T>()
        {
            @Override
            public boolean validate(T value)
            {
                return Iterators.in(value, fsrc);
            }
        };
    }

    /**
     * Предикат возвращает true, если объект равен указанному объекту
     * @param <T> Тип объекта
     * @param value Образец
     * @return предикат
     */
    public static <T> Predicate<T> isEquals(Object value)
    {
        final Object fvalue = value;
        return new Predicate<T>()
        {
            @Override
            public boolean validate(T value)
            {
                if( fvalue==null && value==null )return true;
                if( fvalue!=null && value==null )return false;
                if( fvalue==null && value!=null )return false;
                return fvalue.equals(value);
            }
        };
    }

    /**
     * Предикат возвращает true, если предикат A(объект) и предикат B(объект) возвращают true
     * @param <T> Тип объекта
     * @param a предикаты (A,B,C,...)
     * @return предикат
     */
    public static <T> Predicate<T> and(Predicate<T> ... a)
    {
        if (a == null) {
            throw new IllegalArgumentException("a == null");
        }
        final Predicate[] fa = a;
        return new Predicate<T>()
        {
            @Override
            public boolean validate(T value) {
                if( fa==null )return false;
                int co = 0;
                for( Predicate<T> p : fa ){
                    if( p==null )continue;
                    co++;
                    if( !p.validate(value) ){
                        return false;
                    }
                }
                return co > 0;
            }
        };
    }

    /**
     * Предикат возвращает true, если предикат A(объект) и предикат B(объект) возвращают true
     * @param <T> Тип объекта
     * @param a предикаты (A,B,C,...)
     * @return предикат
     */
    public static <T> Predicate<T> or(Predicate<T> ... a)
    {
        if (a == null) {
            throw new IllegalArgumentException("a == null");
        }
        final Predicate[] fa = a;
        return new Predicate<T>()
        {
            @Override
            public boolean validate(T value) {
                if( fa==null )return false;
                for( Predicate<T> p : fa ){
                    if( p==null )continue;
                    if( p.validate(value) ){
                        return true;
                    }
                }
                return false;
            }
        };
    }

    /**
     * Предикат возвращает true, если предикат A(объект) возвращает false
     * @param <T> Тип объекта
     * @param a предикат A(объект)
     * @return предикат
     */
    public static <T> Predicate<T> not(Predicate<T> a)
    {
        if (a == null) {
            throw new IllegalArgumentException("a == null");
        }
        final Predicate fa = a;
        return new Predicate<T>() {
            @Override
            public boolean validate(T value) {
                return !fa.validate(value);
            }
        };
    }

    /**
     * Предикат возвращает true, если предикат A(объект) (НЕ ИЛИ) предикат B(объект) возвращают true
     * @param <T> Тип объекта
     * @param a предикат A(объект)
     * @param b предикат B(объект)
     * @return предикат
     */
    public static <T> Predicate<T> xor(Predicate<T> a,Predicate<T> b)
    {
        if (a == null) {
            throw new IllegalArgumentException("a == null");
        }
        if (b == null) {
            throw new IllegalArgumentException("b == null");
        }
        final Predicate fa = a;
        final Predicate fb = b;
        return new Predicate<T>()
        {
            @Override
            public boolean validate(T value)
            {
                boolean va = fa.validate(value);
                boolean vb = fb.validate(value);
                return !(va==vb);
            }
        };
    }

    //<editor-fold defaultstate="collapsed" desc="compare numbers">
    public static Predicate<Number> moreOrEqualsThen( final Number compareValue ){
        return new Predicate<Number>() {
            @Override
            public boolean validate(Number value) {
                return !lessThen(compareValue).validate(value);
            }
        };
    }
    
    public static Predicate<Number> lessOrEqualsThen( final Number compareValue ){
        return new Predicate<Number>() {
            @Override
            public boolean validate(Number value) {
                return !moreThen(compareValue).validate(value);
            }
        };
    }
    
    public static Predicate<Number> moreThen( final Number compareValue ){
        return new Predicate<Number>() {
            @Override
            public boolean validate(Number value) {
                if( value==null || compareValue==null )return false;
                if( value instanceof Byte ){
                    if( compareValue instanceof Byte ){
                        return value.byteValue() > compareValue.byteValue();
                    }else if( compareValue instanceof Short ){
                        return value.shortValue() > compareValue.shortValue();
                    }else if( compareValue instanceof Integer ){
                        return value.intValue() > compareValue.intValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue() > compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.byteValue() > compareValue.byteValue();
                    }else if( compareValue instanceof Float ){
                        return value.byteValue() > compareValue.byteValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) > 0;
                    }
                }else if( value instanceof Short ){
                    if( compareValue instanceof Byte ){
                        return value.shortValue() > compareValue.shortValue();
                    }else if( compareValue instanceof Short ){
                        return value.shortValue() > compareValue.shortValue();
                    }else if( compareValue instanceof Integer ){
                        return value.intValue()> compareValue.intValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue()> compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.shortValue()> compareValue.shortValue();
                    }else if( compareValue instanceof Float ){
                        return value.shortValue()> compareValue.shortValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) > 0;
                    }
                }else if( value instanceof Integer ){
                    if( compareValue instanceof Byte ){
                        return value.intValue()> compareValue.intValue();
                    }else if( compareValue instanceof Short ){
                        return value.intValue() > compareValue.intValue();
                    }else if( compareValue instanceof Integer ){
                        return value.intValue()> compareValue.intValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue()> compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.intValue()> compareValue.intValue();
                    }else if( compareValue instanceof Float ){
                        return value.intValue()> compareValue.intValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) > 0;
                    }
                }else if( value instanceof Long ){
                    if( compareValue instanceof Byte ){
                        return value.longValue()> compareValue.longValue();
                    }else if( compareValue instanceof Short ){
                        return value.longValue() > compareValue.longValue();
                    }else if( compareValue instanceof Integer ){
                        return value.longValue()> compareValue.longValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue()> compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.longValue()> compareValue.longValue();
                    }else if( compareValue instanceof Float ){
                        return value.longValue()> compareValue.longValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) > 0;
                    }
                }else if( value instanceof Double ){
                    if( compareValue instanceof Byte ){
                        return value.doubleValue()> compareValue.doubleValue();
                    }else if( compareValue instanceof Short ){
                        return value.doubleValue() > compareValue.doubleValue();
                    }else if( compareValue instanceof Integer ){
                        return value.doubleValue()> compareValue.doubleValue();
                    }else if( compareValue instanceof Long ){
                        return value.doubleValue()> compareValue.doubleValue();
                    }else if( compareValue instanceof Double ){
                        return value.doubleValue()> compareValue.doubleValue();
                    }else if( compareValue instanceof Float ){
                        return value.doubleValue()> compareValue.doubleValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) > 0;
                    }
                }else if( value instanceof Float ){
                    if( compareValue instanceof Byte ){
                        return value.floatValue() > compareValue.floatValue();
                    }else if( compareValue instanceof Short ){
                        return value.floatValue() > compareValue.floatValue();
                    }else if( compareValue instanceof Integer ){
                        return value.floatValue() > compareValue.floatValue();
                    }else if( compareValue instanceof Long ){
                        return value.floatValue() > compareValue.floatValue();
                    }else if( compareValue instanceof Double ){
                        return value.doubleValue()> compareValue.doubleValue();
                    }else if( compareValue instanceof Float ){
                        return value.floatValue()> compareValue.floatValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) > 0;
                    }
                }
                return false;
            }
        };
    }
    
    public static Predicate<Number> lessThen( final Number compareValue ){
        return new Predicate<Number>() {
            @Override
            public boolean validate(Number value) {
                if( value==null || compareValue==null )return false;
                if( value instanceof Byte ){
                    if( compareValue instanceof Byte ){
                        return value.byteValue() < compareValue.byteValue();
                    }else if( compareValue instanceof Short ){
                        return value.shortValue() < compareValue.shortValue();
                    }else if( compareValue instanceof Integer ){
                        return value.intValue() < compareValue.intValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue() < compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.byteValue() < compareValue.byteValue();
                    }else if( compareValue instanceof Float ){
                        return value.byteValue() < compareValue.byteValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) < 0;
                    }
                }else if( value instanceof Short ){
                    if( compareValue instanceof Byte ){
                        return value.shortValue() < compareValue.shortValue();
                    }else if( compareValue instanceof Short ){
                        return value.shortValue() < compareValue.shortValue();
                    }else if( compareValue instanceof Integer ){
                        return value.intValue()< compareValue.intValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue()< compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.shortValue()< compareValue.shortValue();
                    }else if( compareValue instanceof Float ){
                        return value.shortValue()< compareValue.shortValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) < 0;
                    }
                }else if( value instanceof Integer ){
                    if( compareValue instanceof Byte ){
                        return value.intValue()< compareValue.intValue();
                    }else if( compareValue instanceof Short ){
                        return value.intValue() < compareValue.intValue();
                    }else if( compareValue instanceof Integer ){
                        return value.intValue()< compareValue.intValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue()< compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.intValue()< compareValue.intValue();
                    }else if( compareValue instanceof Float ){
                        return value.intValue()< compareValue.intValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) < 0;
                    }
                }else if( value instanceof Long ){
                    if( compareValue instanceof Byte ){
                        return value.longValue()< compareValue.longValue();
                    }else if( compareValue instanceof Short ){
                        return value.longValue() < compareValue.longValue();
                    }else if( compareValue instanceof Integer ){
                        return value.longValue()< compareValue.longValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue()< compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.longValue()< compareValue.longValue();
                    }else if( compareValue instanceof Float ){
                        return value.longValue()< compareValue.longValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) < 0;
                    }
                }else if( value instanceof Double ){
                    if( compareValue instanceof Byte ){
                        return value.doubleValue()< compareValue.doubleValue();
                    }else if( compareValue instanceof Short ){
                        return value.doubleValue() < compareValue.doubleValue();
                    }else if( compareValue instanceof Integer ){
                        return value.doubleValue()< compareValue.doubleValue();
                    }else if( compareValue instanceof Long ){
                        return value.doubleValue()< compareValue.doubleValue();
                    }else if( compareValue instanceof Double ){
                        return value.doubleValue()< compareValue.doubleValue();
                    }else if( compareValue instanceof Float ){
                        return value.doubleValue()< compareValue.doubleValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) < 0;
                    }
                }else if( value instanceof Float ){
                    if( compareValue instanceof Byte ){
                        return value.floatValue() < compareValue.floatValue();
                    }else if( compareValue instanceof Short ){
                        return value.floatValue() < compareValue.floatValue();
                    }else if( compareValue instanceof Integer ){
                        return value.floatValue() < compareValue.floatValue();
                    }else if( compareValue instanceof Long ){
                        return value.floatValue() < compareValue.floatValue();
                    }else if( compareValue instanceof Double ){
                        return value.doubleValue()< compareValue.doubleValue();
                    }else if( compareValue instanceof Float ){
                        return value.floatValue()< compareValue.floatValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) < 0;
                    }
                }
                return false;
            }
        };
    }
    
    public static Predicate<Number> notEquals( final Number compareValue ){
        return new Predicate<Number>() {
            @Override
            public boolean validate(Number value) {
                return !Predicates.equals(compareValue).validate(value);
            }
        };
    }
    
    public static Predicate<Number> equals( final Number compareValue ){
        return new Predicate<Number>() {
            @Override
            public boolean validate(Number value) {
                if( value==null || compareValue==null )return false;
                if( value instanceof Byte ){
                    if( compareValue instanceof Byte ){
                        return value.byteValue() == compareValue.byteValue();
                    }else if( compareValue instanceof Short ){
                        return value.shortValue() == compareValue.shortValue();
                    }else if( compareValue instanceof Integer ){
                        return value.intValue() == compareValue.intValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue() == compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.byteValue() == compareValue.byteValue();
                    }else if( compareValue instanceof Float ){
                        return value.byteValue() == compareValue.byteValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) == 0;
                    }
                }else if( value instanceof Short ){
                    if( compareValue instanceof Byte ){
                        return value.shortValue()== compareValue.shortValue();
                    }else if( compareValue instanceof Short ){
                        return value.shortValue() == compareValue.shortValue();
                    }else if( compareValue instanceof Integer ){
                        return value.intValue()== compareValue.intValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue()== compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.shortValue()== compareValue.shortValue();
                    }else if( compareValue instanceof Float ){
                        return value.shortValue()== compareValue.shortValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) == 0;
                    }
                }else if( value instanceof Integer ){
                    if( compareValue instanceof Byte ){
                        return value.intValue()== compareValue.intValue();
                    }else if( compareValue instanceof Short ){
                        return value.intValue() == compareValue.intValue();
                    }else if( compareValue instanceof Integer ){
                        return value.intValue()== compareValue.intValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue()== compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.intValue()== compareValue.intValue();
                    }else if( compareValue instanceof Float ){
                        return value.intValue()== compareValue.intValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) == 0;
                    }
                }else if( value instanceof Long ){
                    if( compareValue instanceof Byte ){
                        return value.longValue()== compareValue.longValue();
                    }else if( compareValue instanceof Short ){
                        return value.longValue() == compareValue.longValue();
                    }else if( compareValue instanceof Integer ){
                        return value.longValue()== compareValue.longValue();
                    }else if( compareValue instanceof Long ){
                        return value.longValue()== compareValue.longValue();
                    }else if( compareValue instanceof Double ){
                        return value.longValue()== compareValue.longValue();
                    }else if( compareValue instanceof Float ){
                        return value.longValue()== compareValue.longValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) == 0;
                    }
                }else if( value instanceof Double ){
                    if( compareValue instanceof Byte ){
                        return value.doubleValue()== compareValue.doubleValue();
                    }else if( compareValue instanceof Short ){
                        return value.doubleValue() == compareValue.doubleValue();
                    }else if( compareValue instanceof Integer ){
                        return value.doubleValue()== compareValue.doubleValue();
                    }else if( compareValue instanceof Long ){
                        return value.doubleValue()== compareValue.doubleValue();
                    }else if( compareValue instanceof Double ){
                        return value.doubleValue()== compareValue.doubleValue();
                    }else if( compareValue instanceof Float ){
                        return value.doubleValue()== compareValue.doubleValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) == 0;
                    }
                }else if( value instanceof Float ){
                    if( compareValue instanceof Byte ){
                        return value.floatValue() == compareValue.floatValue();
                    }else if( compareValue instanceof Short ){
                        return value.floatValue() == compareValue.floatValue();
                    }else if( compareValue instanceof Integer ){
                        return value.floatValue() == compareValue.floatValue();
                    }else if( compareValue instanceof Long ){
                        return value.floatValue() == compareValue.floatValue();
                    }else if( compareValue instanceof Double ){
                        return value.doubleValue()== compareValue.doubleValue();
                    }else if( compareValue instanceof Float ){
                        return value.floatValue()== compareValue.floatValue();
                    }else if( compareValue instanceof BigInteger ){
                        return BigInteger.valueOf(value.longValue()).compareTo((BigInteger)compareValue) == 0;
                    }
                }
                return false;
            }
        };
    }
    //</editor-fold>
}
