/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.iterators;

import java.util.Iterator;
import xyz.cofe.collection.NodesExtracter;

/**
 * <b>Обход дерева в глубь. С верху вниз.</b> <p>
 * Итератор по деверу объектов заданному через интерфес NodesExtracter&lt;T,T&gt;.
 * <p>
 * <b>Порядок обхода:</b> <p>
 * Древо: 
 * <table border="1" cellspacing="0" summary="Последовательность обхода">
 * <tr>
 * <td colspan="8" align="center">(1) Корень</td>
 * </tr>
 * <tr>
 * <td colspan="4" align="center">(2) Узел</td>
 * <td colspan="4" align="center">(3) Узел</td>
 * </tr>
 * <tr>
 * <td colspan="2" align="center">(4) Узел</td>
 * <td colspan="2" align="center">(5) Узел</td>
 * <td colspan="2" align="center">(6) Узел</td>
 * <td colspan="2" align="center">(7) Узел</td>
 * </tr>
 * <tr>
 * <td>(8) Лист</td><td>(9) Лист</td><td>(10) Лист</td><td>(11) Лист</td>
 * <td>(12) Лист</td><td>(13) Лист</td><td>(14) Лист</td><td>(15) Лист</td>
 * </tr>
 * </table>
 *  <p>
 * Последовательность обхода будет такая:
 * <b>
 * (1), (2), (4), (8), (9), (5), (10), (11), (3), (6), (12), (13), (7), (14), (15)
 * </b>
 * @author gocha
 * @param <T> Тип узла деревра
 * @see xyz.cofe.collection.NodesExtracter
 */
public class SimpleTreeIterator<T> implements Iterator<T>
{
    private NodesExtracter extracter = null;
    private T src = null;
    private boolean readSelf = false;
    private Iterator<T> children = null;
    private Iterator<T> deep = null;
    private T current = null;

    /**
     * Конструктор итератора
     * @param src Исходный объект
     * @param extract Извлечение дочерних объектов
     */
    public SimpleTreeIterator(T src,NodesExtracter<T,T> extract)
    {
        if (src == null) {
            throw new IllegalArgumentException("src == null");
        }
        if (extract == null) {
            throw new IllegalArgumentException("extract == null");
        }

        this.src = src;
        this.extracter = extract;
        go();
    }

    private void go()
    {
        current = findNext();
    }

    private T findNext()
    {
        // Порядок выполнения
        // 1) Если не прочитан сам узел, то возвращается сам узел
        if(!readSelf)
        {
            readSelf=true;

            Iterable<T> i = extracter.extract(src);
            if( i!=null )children = i.iterator();

            return src;
        }

        // 3) Читается дочерний дочернего
        if(deep!=null && deep.hasNext())
        {
            return deep.next();
        }

        // 2) Читается дочерний узел
        if(children!=null && children.hasNext())
        {
            while( true )
            {
                if( !children.hasNext() )break;
                T child = children.next();
                
                if( child!=null )
                {
                    deep = new SimpleTreeIterator(child, extracter);
                    return deep.next();
                }else{
                    deep = null;
                    continue;
                }
            }
        }

        return null;
    }

    @Override
    public boolean hasNext()
    {
        return current!=null;
    }

    @Override
    public T next()
    {
        if( current!=null )
        {
            T result = current;
            current = findNext();
            return result;
        }
        return null;
    }

    @Override
    public void remove()
    {
//        throw new UnsupportedOperationException("Not supported yet.");
    }
}
