/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.iterators;

import java.util.Iterator;

/**
 * Вычитает из исходного множества объекты заданые вторым множством
 * @author gocha
 * @param <T> Тип объектов в множестве
 */
public class SubIterator<T> implements Iterator<T>
{
	private Iterable<T> sub = null;
	private Iterator<T> src = null;
	private CompareEqu<T> cmp = null;
	private T current = null;

    /**
     * Конструктор последовательности
     * @param src Исходное множество
     * @param sub Вычитаемое множество
     * @param cmp Операция сравнения равенства
     */
	public SubIterator(Iterable<T> src,Iterable<T> sub, CompareEqu<T> cmp)
	{
		if (src == null)
		{
			throw new IllegalArgumentException("src == null");
		}
		if (sub == null)
		{
			throw new IllegalArgumentException("sub == null");
		}
		if (cmp == null)
		{
			throw new IllegalArgumentException("cmp == null");
		}

        reinit(src, sub, cmp);
	}
    
    /**
     * Повторная инициализация 
     * @param src Исходное множество
     * @param sub Вычитаемое множество
     * @param cmp Операция сравнения равенства
     */
    public void reinit(Iterable<T> src,Iterable<T> sub, CompareEqu<T> cmp)
    {
		if (src == null)
		{
			throw new IllegalArgumentException("src == null");
		}
		if (sub == null)
		{
			throw new IllegalArgumentException("sub == null");
		}
		if (cmp == null)
		{
			throw new IllegalArgumentException("cmp == null");
		}
        
		this.src = src.iterator();
		this.sub = sub;
		this.cmp = cmp;

		current = findNext();
    }

	private boolean checkIn(T v,Iterable<T> sub)
	{
		for( T c : sub )
		{
			if( cmp.isEqu(v, c) )return true;
		}
		return false;
	}

	private T findNext()
	{
		if( src==null )return null;
		while( src.hasNext() )
		{
			T v = src.next();
			if( v==null )continue;
			if( checkIn(v, sub) )continue;
			return v;
		}
		return null;
	}

	@Override
	public boolean hasNext()
	{
		return current != null;
	}

	@Override
	public T next()
	{
		T res = current;
		current = findNext();
		return res;
	}

	@Override
	public void remove()
	{
	}
}
