/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.list;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.LockMethod;

/**
 * Класс создающий уведомления об изменении списка. <br>
 * Во всех случаях будет переда позиция изменяемого объекта, 
 * это даст излишнюю нагрузку на производительность.<br>
 * Сообщение генериурются синхроно постфактум.<br>
 * Генерирует следующие сообщения:<br>
 * 
 * <table border='1' summary="Генерируемые события">
 * <tr>
 * <td>Метод</td>
 * <td>Сообщение</td>
 * </tr>
 * 
 * <tr>
 * <td>add(E e)</td>
 * <td>AddedPositionalItemEvent&lt;E, EventList&lt;E&gt;&gt;</td>
 * </tr>
 * 
 * <tr>
 * <td>add(int index, E e)</td>
 * <td>AddedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;</td>
 * </tr>
 * 
 * <tr>
 * <td>
 * addAll(Collection&lt;? extends E&gt; c)
 * </td>
 * <td>AddedPositionalItemEvent&lt;E, EventList&lt;E&gt;&gt;</td>
 * </tr>
 * 
 * <tr>
 * <td>
 * addAll(int index, Collection&lt;? extends E&gt; c)
 * </td>
 * <td>AddedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;</td>
 * </tr>
 * 
 * <tr>
 * <td>
 * clear() 
 * </td>
 * <td>
 * RemovedPositionalEvent&lt;E, EventList&lt;E&gt;&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * remove(Object o) 
 * </td>
 * <td>
 * RemovedPositionalEvent&lt;E, EventList&lt;E&gt;&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * remove(int index) 
 * </td>
 * <td>
 * RemovedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * removeAll(Collection&lt;?&gt; c) 
 * </td>
 * <td>
 * RemovedPositionalEvent&lt;E, EventList&lt;E&gt;&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * retainAll(Collection&lt;?&gt; c)
 * </td>
 * <td>
 * RemovedPositionalEvent&lt;E, EventList&lt;E&gt;&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * set(int index, E element) 
 * </td>
 * <td>
 * UpdatedPositionalItemEvent&lt;E, EventList&lt;E&gt;,Integer&gt;
 * </td>
 * </tr>
 * 
 * 
 * </table>
 * @author Камнев Георгий Павлович
 * @param <E> Тип элемента коллеции
 */
public class IndexEventList<E> extends BasicEventList<E>
{
    public IndexEventList()
    {
        this(new ArrayList<E>());
    }

	/**
     * Конструктор.
     *
     * @param list
     *            Обвараичваемый список
     * @throws NullPointerException
     *             если list==null
     */
    public IndexEventList(List<E> list) throws NullPointerException
    {
        super(list);
    }

    //<editor-fold defaultstate="collapsed" desc="add(element):boolean">
    /**
     * Добавление объекта в список
     * @param e Объект
     * @return Факт добавления
     */
    @Override
    public boolean add(final E e)
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return add0(e);
            }
        };
        
        boolean res = (boolean)(Boolean)
            lockRun(fn, new LockMethod("add", true));
        
        fireQueueEvents();
        
        return res;
    }
    
    @Override
    protected boolean add0(E e){
        int size = getWrappedList().size();
        AddingPositionalEvent<E> inserting = new AddingPositionalEvent<E>(e,this,size);
        addEventToQueue(inserting);
        
        fireQueueEvents();
        
        boolean res = getWrappedList().add(e);
        
        if(res){
            int pos = getWrappedList().size()-1;
            AddedPositionalEvent<E> inserted = new AddedPositionalEvent<E>(e,this,pos);
            addEventToQueue(inserted);
        }
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="remove(object)">
    /**
     * Удаление объекта из списка
     * @param o Объект
     * @return Факт удаления
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean remove(final Object o)
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return remove0(o);
            }
        };
        
        boolean res = (boolean)(Boolean)
            lockRun(fn, new LockMethod("remove", true));

        fireQueueEvents();
        
        return res;
    }
    
    @Override
    protected boolean remove0(Object o){
//        @SuppressWarnings("element-type-mismatch")
//        boolean res = super.remove(o);
        
        int index = getWrappedList().indexOf(o);
        if( index>=0 ){
            addEventToQueue( new RemovingPositionalEvent(o, this, index ) );
            fireQueueEvents();
        }
        
        @SuppressWarnings("element-type-mismatch")
        boolean res = getWrappedList().remove(o);

        if (res){
            addEventToQueue(new RemovedPositionalEvent(o, this, index));
        }
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="remove(index):E">
    /**
     * Удаление объекта из списка. <br>
     * Для удаленного элемента генерирует сообщение <b>RemovedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;</b>
     * @param index Индекс элемента
     * @return Удаленный элемент
     * @see xyz.cofe.collection.RemovedPositionalItemEvent
     */
    @Override
    public E remove(final int index)
    {
        // Func0 fn = (Func0) () -> remove0(index); // java 8
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                remove0(index);
                return null;
            }
        };
        
        E item = (E)
            lockRun(fn, new LockMethod("remove", true));
        
        fireQueueEvents();
        return item;
    }
    
    @Override
    protected E remove0(int index){
        addEventToQueue(new RemovingPositionalEvent<E>(get(index), this, index));
        fireQueueEvents();
        
        E item = getWrappedList().remove(index);
        
        addEventToQueue(new RemovedPositionalEvent<E>(item, this, index));
        return item;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="removeAll(elements)">
    /**
     * Удаляет группу указанных объектов из списка. <br>
     * Вызывает метод remove(e)<br>
     * Генерирует <b>RemovedItemEvent&lt;E, EventList&lt;E&gt;&gt;</b> для удаленных объектов.
     * @param c Группа объектов
     * @return Факт удаления
     * @see #remove(java.lang.Object)
     * @see xyz.cofe.collection.RemovedItemEvent
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean removeAll(final Collection<?> c)
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return removeAll0(c);
            }
        };
        
        boolean modified = (boolean)(Boolean)
            lockRun(fn, new LockMethod("removeAll", true));
        
        fireQueueEvents();
        
        return modified;
    }
    
    @Override
    protected boolean removeAll0(final Collection<?> c){
        if( c==null )return false;
        
        int idx = 0;
        for( Object e : this.toArray() ){
            if( c.contains(e) ){
                addEventToQueue( new RemovingPositionalEvent(e, this, idx) );
                fireQueueEvents();
                
                getWrappedList().remove(idx);

                addEventToQueue( new RemovedPositionalEvent(e, this, idx) );
                fireQueueEvents();
                
                continue;
            }
            
            idx++;
        }
        
//        int idx = -1;
//        TreeSet<Integer> removeIndexes = new TreeSet<>();
//        
//        for( Object e : this ){
//            idx++;
//            if( c.contains(e) ){
//                addEventToQueue( new RemovingPositionalEvent(e, this, idx ) );
//                removeIndexes.add(idx);
//            }
//        }
//        
//        Iterator<Integer> ridxitr = removeIndexes.descendingIterator();
//        while( ridxitr.hasNext() ){
//            Integer ridx  = ridxitr.next();
//            
//            Object o = getWrappedList().get(ridx);
//
//            @SuppressWarnings("element-type-mismatch")
//            boolean res = getWrappedList().remove(ridx);
//
//            if (res){
//                addEventToQueue(new RemovedPositionalEvent(o, this, ridx));
//            }
//        }
        
        return true;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="retainAll(elements)">
    /**
     * Удаляет объекты из списка, не входящиие в указанную группу объектов. <br>
     * <br>
     * Сравниваются исходная последовательность элементов, и та что осталась после удаления объектов.<br>
     * Те что были удалены для них генерирует сообщения <b>RemovedPositionalItemEvent&lt;E, EventList&lt;E&gt;&gt;</b>
     * в обратном порядке следования исходного списка элементов.
     * @param c Группа объектов
     * @return Факт удаления
     * @see xyz.cofe.collection.RemovedPositionalItemEvent
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean retainAll(final Collection<?> c)
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return retainAll0(c);
            }
        };
        
        boolean modified = (boolean)(Boolean)
            lockRun(fn, new LockMethod("removeAll", true));
        
        fireQueueEvents();
        
        return modified;
    }
    
    @Override
    protected boolean retainAll0(final Collection<?> c){
        if( c==null )return false;
        
        int idx = -1;
        TreeSet<Integer> removeIndexes = new TreeSet<Integer>();
        
        for( Object e : this ){
            idx++;
            if( !c.contains(e) ){
//                addEventToQueue( new RemovingPositionalEvent(e, this, idx ) );
                removeIndexes.add(idx);
            }
        }
        
        Iterator<Integer> ridxitr = removeIndexes.descendingIterator();
        while( ridxitr.hasNext() ){
            int ridx  = ridxitr.next();
            
            //Object o = getWrappedList().get(ridx);

            remove(ridx);

//            if (res){
//                addEventToQueue(new RemovedPositionalEvent(o, this, ridx));
//            }
        }
        
        return true;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="iterator()">
    @Override
    public Iterator<E> iterator() {
        //Func0 fn = (Func0) () -> iterator0(); // java 8
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return iterator0();
            }
        };
        
        Iterator<E> itr = (Iterator<E>)lockRun(fn, new LockMethod("iterator", false));
        
        fireQueueEvents();
        
        return itr;
    }

    @Override
    protected Iterator<E> iterator0(){
        final IndexEventList<E> f_elist = this;
        final Iterator<E> f_itr = getWrappedList().iterator();        
        return new IndexIterator<E>(f_elist, f_itr);
    }
    //</editor-fold>
}
