/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy <nt.gocha@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.collection.map;

import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.LockMethod;

/**
 * Event map с поддержкой блокировки Lock
 * @author nt.gocha@gmail.com
 */
public class SyncEventMap <Key,Value> extends BasicEventMap<Key,Value>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(SyncEventMap.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(SyncEventMap.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(SyncEventMap.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(SyncEventMap.class.getName(), method, result);
    }
    //</editor-fold>

    @SuppressWarnings("LeakingThisInConstructor")
    public SyncEventMap() {
        syncWrite = this;
        this.syncRead = this.syncWrite;
    }

    @SuppressWarnings("LeakingThisInConstructor")
    public SyncEventMap(Map<Key, Value> wrappedMap) {
        super(wrappedMap);
        syncWrite = this;
        this.syncRead = this.syncWrite;
    }

    public SyncEventMap(Map<Key, Value> wrappedMap, Object lock) {
        super(wrappedMap);
        this.syncWrite = lock==null ? this : lock;
        this.syncRead = this.syncWrite;
    }

    public SyncEventMap(Map<Key, Value> wrappedMap, Object syncRead, Object syncWrite) {
        super(wrappedMap);
        this.syncWrite = syncWrite==null ? this : syncWrite;
        this.syncRead = syncRead==null ? this.syncWrite : syncRead;
    }
    
    protected final Object syncWrite;
    protected final Object syncRead;

    //<editor-fold defaultstate="collapsed" desc="getLock()">
    /**
     * Возвращает блокировку на методы Map
     * @return блокировка на методы map или null
     */
    public Object getSyncWrite(){
        return syncWrite;
    }

    /**
     * Возвращает блокировку на методы Map
     * @return блокировка на методы map или null
     */
    public Object getSyncRead(){
        return syncRead;
    }
    //</editor-fold>
    
    @Override
    protected Object lockRun(Func0 run, LockMethod method) {
        if( run==null )throw new IllegalArgumentException("run==null");
        
        if( syncWrite==syncRead ){
            return lockRun(run);
        }else{
            if( method!=null ){
                boolean isRead = method.getWrite()==null || !method.getWrite();
                boolean isWrite = !isRead;
                if( isWrite ){
                    // write Lock
                    return lockRun(run);
                }else{
                    // read Lock
                    synchronized( syncRead ){
                        return run.apply();
                    }
                }
            }else{
                return lockRun(run);
            }
        }
    }

    @Override
    protected Object lockRun(Func0 run) {
        synchronized(syncWrite){
            return run.apply();
        }
    }
}
