/*
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.set;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.LockMethod;
import xyz.cofe.common.GetListenersHelper;
import xyz.cofe.common.ListenersHelper;
import xyz.cofe.common.Reciver;

/**
 * Множество с поддержкой уведомлений
 * @author gocha
 * @param <E> Тип элемента коллекции
 */
public class BasicEventSet<E> 
    extends SetWrapper<E> 
    implements EventSet<E>, GetListenersHelper
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(BasicEventSet.class.getName());
    
    private static Level logLevel(){ return logger.getLevel(); }
    private static boolean isLogSevere(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.SEVERE.intValue();
    }
    private static boolean isLogWarning(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.WARNING.intValue();
    }
    private static boolean isLogInfo(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.INFO.intValue();
    }
    private static boolean isLogFine(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.FINE.intValue();
    }
    private static boolean isLogFiner(){
        Level ll = logLevel();
        return ll == null
            ? false
            : ll.intValue() <= Level.FINER.intValue();
    }
    private static boolean isLogFinest(){
        Level ll = logLevel();
        return ll == null
            ? false
            : ll.intValue() <= Level.FINEST.intValue();
    }
    
    private static void logEntering(String method,Object ... args){
        logger.entering(BasicEventSet.class.getName(), method, args);
    }
    private static void logExiting(String method,Object result){
        logger.exiting(BasicEventSet.class.getName(), method, result);
    }
    
    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }
    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }
    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }    
    //</editor-fold>

    /**
     * Конструктор множества
     */
    public BasicEventSet()
    {
        super(new LinkedHashSet<E>());
//        lock = new ReentrantLock();
    }

    /**
     * Конструктор множества
     * @param set Исходное множество
     */
    public BasicEventSet(Set<E> set)
    {
        super(set);
//        lock = new ReentrantLock();
    }

    //<editor-fold defaultstate="collapsed" desc="listeners">
    private ListenersHelper<EventSetListener<E>,EventSetArgs<E>>
        listeners = new ListenersHelper<EventSetListener<E>, EventSetArgs<E>>(
            new Func2<Object, EventSetListener<E>, EventSetArgs<E>>(){
                @Override
                public Object apply(EventSetListener<E> listener, EventSetArgs<E> e)
                {
                    listener.eventSet(e);
                    return null;
                }
            }
        );

    @Override
    public ListenersHelper getListenersHelper() {
        return listeners;
    }
    
    /* (non-Javadoc) @see EventSet */
    @Override
    public Collection<EventSetListener<E>> getEventSetListeners(){
        return listeners.getListeners();
    }

    /* (non-Javadoc) @see EventSet */
    @Override
    public Closeable onAdded(final Reciver<E> listener){
        if( listener==null )throw new IllegalArgumentException( "listener==null" );
        final Closeable ch = addEventSetListener(new EventSetAdapter<E>(){
            @Override
            protected void inserted(E item) {
                listener.recive(item);
            }
        });
        return ch;
    }

    /* (non-Javadoc) @see EventSet */
    @Override
    public Closeable onRemoved(final Reciver<E> listener){
        if( listener==null )throw new IllegalArgumentException( "listener==null" );
        final Closeable ch = addEventSetListener(new EventSetAdapter<E>(){
            @Override
            protected void deleted(E item) {
                listener.recive(item);
            }
        });
        return ch;
    }
    
    /* (non-Javadoc) @see EventSet */
    @Override
    public Closeable addEventSetListener(EventSetListener<E> listener)
    {
        return listeners.addListener(listener);
    }

    @Override
    public Closeable addEventSetListener(EventSetListener<E> listener,boolean weak)
    {
        return listeners.addListener(listener,weak);
    }

    /* (non-Javadoc) @see EventSet */
    @Override
    public void removeEventSetListener(EventSetListener<E> listener)
    {
        listeners.removeListener(listener);
    }

    @Override
    public boolean containsEventSetListener(EventSetListener<E> listener)
    {
        return listeners.hasListener(listener);
    }

    protected void fireEventSet(EventSetArgs<E> args)
    {
        listeners.fireEvent(args);
    }

    @Override
    public Closeable onChanged(final Func3<Object, ? super E, ? super E, ? super E> fn, boolean weak)
    {
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return addEventSetListener(new EventSetListener<E>()
        {
            @Override
            public void eventSet(EventSetArgs<E> args)
            {
                if( args instanceof EventSetArgsInserted ){
                    EventSetArgsInserted<E> ev = (EventSetArgsInserted)args;
                    E added = ev.getAddedItem();
                    if( added==null ){
                        logWarning("EventSetArgsInserted added null element", added);
                    }
                    if( fn!=null ){
                        fn.apply(null, null, added);
                    }
                }else if( args instanceof EventSetArgsDeleted ){
                    EventSetArgsDeleted<E> ev = (EventSetArgsDeleted)args;
                    E deleted = ev.getRemovedItem();
                    if( deleted==null ){
                        logWarning("EventSetArgsDeleted removed null element", deleted);
                    }
                    if( fn!=null ){
                        fn.apply(null, deleted, null);
                    }
                }
            }
        }, weak);
    }

    @Override
    public Closeable onChanged(Func3<Object, ? super E, ? super E, ? super E> fn)
    {
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return onChanged(fn, false);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="eventQueue - Очередь сообщений">
    /**
     * Очередь сообщений
     */
    protected final Queue<EventSetArgs<E>> eventQueue = new LinkedList<EventSetArgs<E>>();
    
    //<editor-fold defaultstate="collapsed" desc="addEventToQueue(e)">
    /**
     * Добавляет уведомления в очередь с учетом блокировки
     * @param event уведомления
     * @return добавленное уведомление
     */
    protected EventSetArgs<E> addEventToQueue(EventSetArgs<E> event)
    {
        synchronized(eventQueue){
            addEventToQueue0(event);
        }
        return event;
    }
    
    /**
     * Добавляет уведомления в очередь без учета блокировки
     * @param job уведомления
     */
    private void addEventToQueue0(EventSetArgs<E> event){
        if( event!=null )eventQueue.add(event);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="fireQueueEvents()">
    /**
     * Выполняет уведомления из очереди с учетем блокировки
     */
    protected void fireQueueEvents(){
        synchronized(eventQueue){
            fireQueueEvents0();
        }
    }
    
    /**
     * Выполняет уведомления из очереди без учета блокировки
     */
    private void fireQueueEvents0()
    {
        while(true)
        {
            EventSetArgs<E> event = eventQueue.poll();
            if( event==null )break;
            listeners.fireEvent(event);
        }
    }
    //</editor-fold>
    //</editor-fold>    

    protected volatile boolean cancelForEach = true;
    protected volatile boolean cancelForAll = false;
    
    //<editor-fold defaultstate="collapsed" desc="createInserting(e)">
    /**
     * Создает уведомление о добавлении
     * @param e Добавляемый объект
     * @return уведомление
     */
    protected EventSetArgs<E> createInserting(E e){
        return new EventSetArgs<E>(this, EventSetAction.Inserting, e);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="createInserted(e)">
    /**
     * Создает уведомление о добавлении
     * @param e Добавляемый объект
     * @return уведомление
     */
    protected EventSetArgs<E> createInserted(E e){
        return new EventSetArgsInserted<E>(this,e);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="createDeleting(e)">
    /**
     * Создает уведомление о удалении
     * @param e Удаляемый объект
     * @return уведомление
     */
    protected EventSetArgs<E> createDeleting(E e){
        return new EventSetArgs<E>(this, EventSetAction.Deleting, e);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="createDeleted(e)">
    /**
     * Создает уведомление о удалении
     * @param e Удаляемый объект
     * @return уведомление
     */
    protected EventSetArgs<E> createDeleted(E e){
        return new EventSetArgsDeleted<E>(this, e);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="lockRun(fun):Object">
    /**
     * Выполнение кода в блокировке
     * @param run код
     * @return результат выполнения кода
     */
    protected Object lockRun( Func0 run ){
        if( run==null )throw new IllegalArgumentException("run==null");
        Object r = run.apply();
        return r;
    }

    /**
     * Выполнение кода в блокировке
     * @param run код
     * @param method метод map для которого вызывается блокировка
     * @return результат выполнения кода
     */
    protected Object lockRun( Func0 run, LockMethod method){
        if( run==null )throw new IllegalArgumentException("run==null");
        return lockRun(run);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="add(e)">
    // TODO v1 Протестировать что передается AddedItemEvent
    /* (non-Javadoc) @see SetWrapper */
    @Override
    public boolean add(final E e)
    {
        boolean r = (boolean)(Boolean)( lockRun(new Func0() {
            @Override
            public Object apply() {
                if( cancelForAll || cancelForEach )
                {
                    EventSetArgs<E> esa = createInserting(e);
                    fireEventSet(esa);
                    if( esa.isCancel() )
                        return false;
                }
                
                boolean r = (boolean)(Boolean)lockRun( 
                    new Func0(){ @Override public Object apply(){
                        return add0( e );
                    }}
                    ,new LockMethod("add",true)
                );
                
                return r;
            }
        },new LockMethod("add#precheck",false)
        ));
        
        fireQueueEvents();
        return r;
    }
    
    private boolean add0(final E e){
        boolean r = BasicEventSet.super.add(e);
        if( r ){
            EventSetArgs<E> args = createInserted(e);
            addEventToQueue(args);
        }
        return r;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="addAll(c)">
    // TODO v1 Протестировать что передается AddedItemEvent
    /* (non-Javadoc) @see SetWrapper */
    @Override
    public boolean addAll(final Collection<? extends E> c)
    {
        if( cancelForEach && !cancelForAll )
        {
            if( c==null )return false;
            
            Func0<Boolean> fn = new Func0<Boolean>() {
                @Override
                public Boolean apply() {
                    int count = 0;
                    int addedCount = 0;
                    for( final E e : c )
                    {
                        count++;
                        
                        EventSetArgs<E> esa = createInserting(e);
                        fireEventSet(esa);
                        if( esa.isCancel() )return false;

                        Func0 fn = new Func0() {
                            @Override
                            public Object apply() {
                                return add0(e);
                            }
                        };
                        
                        boolean added = 
                            (boolean)(Boolean)
                            lockRun(fn,new LockMethod("addAll", true));
                        
                        if( added ){
                            addedCount++;
                        }
                    }
                    
                    return (addedCount>0 && count>0) || (count==0);
                }
            };
            
            boolean r = (boolean)(Boolean)lockRun(
                fn,
                new LockMethod("addAll#precheck", false));
            
            fireQueueEvents();
            
            return r;
        }
        
        Func0<Boolean> fnCancelForAll = null;
        
        if( cancelForAll )
        {
            fnCancelForAll = new Func0() {
                @Override
                public Object apply() {
                    if( c==null )
                        return false;

                    boolean cancel = false;
                    for( E e : c )
                    {
                        EventSetArgs<E> esa = createInserting(e);
                        fireEventSet(esa);

                        if( esa.isCancel() )
                        {
                            cancel = true;
                            break;
                        }
                    }

                    if( cancel )
                        return false;
                    
                    return null;
                }
            };
        }
        
        @SuppressWarnings("UnusedAssignment")
        boolean r = false;
        
        final Func0 fnAddAllSuper = new Func0() {
            @Override
            public Object apply() {
                int count = 0;
                int added = 0;
                for( E e : c ){
                    count++;
                    if( add0( e ) ){
                        added++;
                    }
                }
                
                return count > 0 && added > 0 ;
            }
        };
        
        if( fnCancelForAll==null ){
            r = (boolean)(Boolean)(lockRun( 
                fnAddAllSuper, 
                new LockMethod("addAll",true)
            ));
        }else{
            final Func0<Boolean> ffnCancelForAll = fnCancelForAll;
            
            Func0 fn = new Func0() {
                @Override
                public Object apply() {
                    Boolean preCheck = ffnCancelForAll.apply();
                    if( preCheck!=null && !((boolean)preCheck) ){
                        return false;
                    }
                    
                    return lockRun(fnAddAllSuper, new LockMethod("addAll",true));
                }
            };
            
            r = (boolean)(Boolean)lockRun(fn, new LockMethod("addAll#precheck", false));
        }
        
        fireQueueEvents();
        
        return r;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="clear()">
    // TODO v1 Протестировать что передается RemovedItemEvent
    /* (non-Javadoc) @see SetWrapper */
    @Override
    public void clear()
    {
        if( cancelForAll )
        {
            final Func0<Boolean> delPreCheck = new Func0<Boolean>() {
                @Override
                public Boolean apply() {
                    for( E e : BasicEventSet.this )
                    {
                        EventSetArgs<E> args = createDeleting(e);
                        fireEventSet(args);
                        if( args.isCancel() )return false;
                    }
                    
                    return true;
                }
            };
            
            lockRun(new Func0(){
                @Override
                public Object apply() {
                    Boolean r = delPreCheck.apply();
                    if( r!=null && !(boolean)r )return null;
                    
                    Object[] deleted = BasicEventSet.super.toArray();
                    
                    lockRun(new Func0() {
                        @Override
                        public Object apply() {
                            BasicEventSet.super.clear();
                            return null;
                        }
                    }, new LockMethod("clear",true));
                        
                    for( Object oe : deleted )
                    {
                        try{
                            E e = (E)oe;
                            EventSetArgs<E> args = createDeleted(e);
                            addEventToQueue(args);
                        }catch(ClassCastException ex){
                            logException(ex);
                        }
                    }
                    
                    return null;
                }
            }, new LockMethod("clear#precheck",false));
            
            fireQueueEvents();
            return;
        }
        
        if( cancelForEach )
        {
            Func0 fnPrecheck = new Func0() {
                @Override
                public Object apply() {
                    for( final E e : BasicEventSet.this )
                    {
                        EventSetArgs<E> args = createDeleting(e);
                        fireEventSet(args);
                        if( args.isCancel() )continue;

                        @SuppressWarnings("UnusedAssignment")
                        boolean r = false;
                        r = (boolean)(Boolean)(lockRun(new Func0() {
                            @Override
                            public Object apply() {
                                return BasicEventSet.this.remove0(e);
                            }
                        }, new LockMethod("clear",true)
                        ));
                    }

                    return null;
                }
            };
            
            lockRun(fnPrecheck, new LockMethod("clear#precheck", false));
            fireQueueEvents();
            return;
        }
        
        if( !cancelForAll || !cancelForEach ){
            Func0 clearFn = new Func0() {
                @Override
                public Object apply() {
                    Object[] deleted = BasicEventSet.super.toArray();
                    
                    BasicEventSet.super.clear();
                    
                    for( Object oe : deleted )
                    {
                        try{
                            E e = (E)oe;
                            EventSetArgs<E> args = createDeleted(e);
                            addEventToQueue(args);
                        }catch(ClassCastException ex){
                            logException(ex);
                        }
                    }
                    
                    return null;
                }
            };
            lockRun(clearFn, new LockMethod("clear", true));
            fireQueueEvents();
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="remove(e)">
    // TODO v1 Протестировать что передается RemovedItemEvent
    /* (non-Javadoc) @see SetWrapper */
    @Override
    public boolean remove(final Object o)
    {
        final Func0 fnPrecheck = new Func0() {
            @Override
            public Object apply() {
                if( cancelForAll || cancelForEach )
                {
                    try
                    {
                        EventSetArgs<E> args = createDeleting((E)o);
                        fireEventSet(args);

                        if( args.isCancel() )return false;
                        return null;
                    }
                    catch(ClassCastException ex)
                    {
                        ex.printStackTrace();
                        return null;
                    }
                }
                return null;
            }
        };
        
        final Func0 fnMain = new Func0() {
            @Override
            @SuppressWarnings("element-type-mismatch")
            public Object apply() {
                Object rc = fnPrecheck.apply();
                
                if( rc instanceof Boolean && !((Boolean)rc) ){
                    return false;
                }
                
                boolean r = (boolean)(Boolean)lockRun(new Func0() {
                    @Override
                    public Object apply() {
                        return remove0(o);
                    }
                }, new LockMethod("remove", true));
                
                return r;
            }
        };
        
        boolean r;
        r = (boolean)(Boolean)lockRun(fnMain, new LockMethod("remove#precheck",false));

        fireQueueEvents();
        
        return r;
    }
    
    protected boolean remove0( Object o ){
        boolean r = super.remove(o);
        if( r ){
            try{
                EventSetArgs<E> args = createDeleted((E)o);
                addEventToQueue(args);
            }catch(ClassCastException ex){
                logException(ex);
            }
        }
        return r;
    }    
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="removeAll()">
    private boolean removeAll_checkContains = false;
    
    // TODO v1 Протестировать что передается RemovedItemEvent
    /* (non-Javadoc) @see SetWrapper */
    @Override
    public boolean removeAll(final Collection<?> c)
    {
        if( cancelForAll )
        {
            //<editor-fold defaultstate="collapsed" desc="precheck removing">
            final AtomicInteger changed = new AtomicInteger(0);
            if( c!=null )
            {
                Func0 fnPrecheck = new Func0(){
                    @Override public Object apply() {
                        //<editor-fold defaultstate="collapsed" desc="prechek">
                        for( Object o2del : c )
                        {
                            try
                            {
                                E e = (E)o2del;
                                if( removeAll_checkContains && !BasicEventSet.this.contains(e) )continue;
                                
                                EventSetArgs<E> args = createDeleting(e);
                                fireEventSet(args);
                                if( args.isCancel() )
                                {
                                    return false;
                                }
                            }catch(ClassCastException ex)
                            {
                                ex.printStackTrace();
                            }
                        }
                        //</editor-fold>
                        //<editor-fold defaultstate="collapsed" desc="deleting">
                        for( final Object o2del : c )
                        {
                            
                            boolean r;
                            r = (boolean)(Boolean)lockRun(new Func0() {
                                @Override
                                @SuppressWarnings("element-type-mismatch")
                                public Object apply() {
                                    boolean res = BasicEventSet.this.remove0(o2del);
                                    if( res ){
                                        changed.incrementAndGet();
                                    }
                                    return res;
                                }
                            }, new LockMethod("removeAll",true)
                            );
                        }
                        //</editor-fold>
                        return null;
                    }};
                
                lockRun(fnPrecheck, new LockMethod("removeAll#precheck", false));
                
                fireQueueEvents();
            }
            return changed.get() > 0;
            //</editor-fold>
        }
        
        if( cancelForEach )
        {
            //<editor-fold defaultstate="collapsed" desc="precheck removing">
            final AtomicInteger changed = new AtomicInteger(0);
            if( c!=null )
            {
                Func0 fn = new Func0() { @Override public Object apply(){
                    for( final Object o2del : c ){
                        //<editor-fold defaultstate="collapsed" desc="precheck">
                        try
                        {
                            E e = (E)o2del;
                            if( removeAll_checkContains && !BasicEventSet.this.contains(e) )continue;
                            EventSetArgs<E> args = createDeleting(e);
                            fireEventSet(args);
                            if( args.isCancel() ){
                                continue;
                            }
                        }
                        catch( ClassCastException ex )
                        {
                            logException(ex);
                        }
                        //</editor-fold>
                        //<editor-fold defaultstate="collapsed" desc="deleting">
                        lockRun(new Func0() {
                            @Override
                            @SuppressWarnings("element-type-mismatch")
                            public Object apply() {
                                boolean res = BasicEventSet.this.remove0(o2del);
                                if( res ){
                                    changed.incrementAndGet();
                                }
                                return res;
                            }
                        }, new LockMethod("removeAll",true));
                        //</editor-fold>
                    }
                    return null;
                }};
                lockRun(fn, new LockMethod("removeAll#precheck", false));
            }
            
            fireQueueEvents();
            return changed.get()>0;
            //</editor-fold>
        }
        
        final AtomicInteger changed = new AtomicInteger(0);
        
        lockRun(new Func0() {
            @Override
            @SuppressWarnings("element-type-mismatch")
            public Object apply() {
                if( c!=null ){
                    for( Object o2del : c ){
                        if( BasicEventSet.this.remove0(o2del) ){
                            changed.incrementAndGet();
                        }
                    }
                }
                return null;
            }
        }, new LockMethod("removeAll",true));

        fireQueueEvents();
        
        return changed.get()>0;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="retainAll()">
    protected void retainsPrepare2Delete(Collection<?> coll2retains, Collection toDelete)
    {
        for( E e : this )
        {
            if( !coll2retains.contains(e) )
                toDelete.add(e);
        }
    }

    // TODO v1 Протестировать что передается RemovedItemEvent
    /* (non-Javadoc) @see SetWrapper */
    @Override
    public boolean retainAll(final Collection<?> c)
    {
        if( cancelForEach || cancelForAll )
        {
            Func0 fn = new Func0() {
                @Override
                public Object apply() {
                    if( c==null )
                    {
                        clear();
                        return true;
                    }

                    Collection toDel = new LinkedList();
                    retainsPrepare2Delete(c, toDel);

                    boolean r = removeAll(toDel);
                    return r;
                }
            };
            
            Object r = lockRun(fn, new LockMethod("retainAll", false));
            if( r instanceof Boolean ){
                return (Boolean)r;
            }
            
            return true;
        }
        
        return retainsStd(c);
    }
    
    protected boolean retainsStd(final Collection<?> c) {
        Func0 fn = new Func0() {
                @Override
                public Object apply() {
            Object[] before = toArray();

            boolean res;
            res = (boolean)(Boolean)lockRun(new Func0() { 
                @Override public Object apply() {
                    return BasicEventSet.super.retainAll(c);
                }}
                , new LockMethod("retainAll#std",true)
            );

            Object[] after = toArray();
            List removed = new ArrayList();
            
            for (Object o1 : before) {
                if (o1 == null) {
                    continue;
                }
                
                boolean isRemoved = true;
                for (Object o2 : after) {
                    if (o2 == null) {
                        continue;
                    }
                    if (o1.equals(o2)) {
                        isRemoved = false;
                        break;
                    }
                }
                
                if (isRemoved) {
                    removed.add(o1);
                }
            }
            
            for (Object ro : removed) {
                if (ro == null) {
                    continue;
                }
                try {
                    E e = (E) ro;
                    EventSetArgs<E> args = createDeleted(e);
                    addEventToQueue(args);
                } catch (ClassCastException ex) {
                }
            }

            return res;
        }};
        
        boolean r = (boolean)(Boolean)this.lockRun(
            fn, 
            new LockMethod("retainAll#prepare", false));
        return r;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contains()">
    @Override
    public boolean contains( final Object o ) {
        boolean res = (boolean)(Boolean)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventSet.super.contains(o);
            }
        }
        , new LockMethod("contains",false)   
        );
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="containsAll()">
    @Override
    public boolean containsAll( final Collection<?> c ) {
        boolean res = (boolean)(Boolean)lockRun(
            new Func0() { @Override public Object apply() {
                return BasicEventSet.super.containsAll(c);
            }}
            , new LockMethod("containsAll",false)
        );
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="size()">
    @Override
    public int size() {
        int res = (int)(Integer)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventSet.super.size();
            }}
            , new LockMethod("size",false)
        );
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="toArray()">
    @Override
    public Object[] toArray() {
        Object[] res = (Object[])lockRun(
            new Func0() { @Override public Object apply() {
                return BasicEventSet.super.toArray();
            }}
            , new LockMethod("toArray",false)
        );
        return res;
    }
    
    @Override
    public <T> T[] toArray( final T[] a ) {
        T[] res = (T[])lockRun(
            new Func0() { @Override public Object apply() {
                return BasicEventSet.super.toArray(a);
            }}
            , new LockMethod("toArray",false)
        );
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="isEmpty()">
    @Override
    public boolean isEmpty() {
        boolean r = (boolean)(Boolean)lockRun(
            new Func0() { @Override public Object apply() {
                return BasicEventSet.super.isEmpty();
            }}
            , new LockMethod("isEmpty",false)
        );
        return r;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="iterator()">
    @Override
    public Iterator<E> iterator() {
        Iterator<E> i = (Iterator<E>)lockRun(
            new Func0() { @Override public Object apply() {
                return BasicEventSet.super.iterator();
            }}
            , new LockMethod("iterator",false)
        );
        return i;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getWrappedSet()">
    @Override
    public Set<E> getWrappedSet() {
        Set<E> s = (Set<E>)lockRun(
            new Func0() { @Override public Object apply() {
                return BasicEventSet.super.getWrappedSet();
            }}, new LockMethod("getWrappedSet",false)
        );
        return s;
    }
    //</editor-fold>
}
