/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.table;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Map;
import java.util.TreeMap;
import java.util.TreeSet;
import xyz.cofe.collection.BasicPair;
import xyz.cofe.collection.Convertor;
import xyz.cofe.collection.Iterators;
import xyz.cofe.collection.Pair;
import xyz.cofe.collection.Predicate;

/**
 * Таблица с данными
 * @author Камнев Георгий Павлович
 * @param <Row> Тип строки таблицы
 * @param <Column> Тип колонки таблицы
 * @param Value Элемент таблицы
 */
public class Table<Row, Column, Value>
{
    // <editor-fold defaultstate="collapsed" desc="Listener">
    /**
     * Подписчик на события
     */
    public interface Listener<R, C, V>
    {

        /**
         * Событие таблицы
         *
         * @param event Событие
         */
        void tableEvent(Event<R, C, V> event);
    }
    // </editor-fold>
    
    // <editor-fold defaultstate="collapsed" desc="Event">
    /**
     * Событие таблицы
     *
     * @param <R> Тип строки
     * @param <C> Тип колонки
     * @param <V> Тип значения
     */
    public static class Event<R, C, V>
    {

        protected Table<R, C, V> table = null;

        /**
         * Конструктор
         *
         * @param table таблица
         */
        public Event(Table<R, C, V> table) {
            this.table = table;
        }
        
        public Table<R, C, V> getTable() {
            return table;
        }
    }
    // </editor-fold>
    
    // <editor-fold defaultstate="collapsed" desc="ValueChangedEvent">
    /**
     * Событие - изменение значения ячейки
     *
     * @param <R> Тип строки
     * @param <C> Тип колонки
     * @param <V> Тип значения
     */
    public static class ValueChagnedEvent<R, C, V> extends Event<R, C, V>
    {

        protected R row = null;
        protected C column = null;
        protected V value = null;
        
        public ValueChagnedEvent(Table<R, C, V> table, R row, C column, V value) {
            super(table);
            this.row = row;
            this.column = column;
            this.value = value;
        }
        
        public R getRow() {
            return row;
        }

        public C getColumn() {
            return column;
        }

        public V getValue() {
            return value;
        }
    }
    // </editor-fold>
    
    // <editor-fold defaultstate="collapsed" desc="RowChagnedEvent">
    /**
     * Событие табилцы - изменение колонки
     *
     * @param <R> Тип строки
     * @param <C> Тип колонки
     * @param <V> Тип значения
     */
    public static class RowChagnedEvent<R, C, V> extends Event<R, C, V>
    {

        protected R row = null;

        /**
         * Констркутор
         *
         * @param table Таблица
         * @param row Строка
         */
        public RowChagnedEvent(Table<R, C, V> table, R row) {
            super(table);
            this.row = row;
        }

        /**
         * Строка таблицы
         *
         * @return Строка
         */
        public R getRow() {
            return row;
        }
    }

    /**
     * Событие - строка добавлена
     *
     * @param <R> Тип строки
     * @param <C> Тип колонки
     * @param <V> Тип значения
     */
    public static class RowAddedEvent<R, C, V> extends RowChagnedEvent<R, C, V>
    {

        /**
         * Конструктор
         *
         * @param table Таблица
         * @param row Строка
         */
        public RowAddedEvent(Table<R, C, V> table, R row) {
            super(table, row);
        }
    }

    /**
     * Событие - строка удалена
     *
     * @param <R> Тип строки
     * @param <C> Тип колонки
     * @param <V> Тип значения
     */
    public static class RowDeletedEvent<R, C, V> extends RowChagnedEvent<R, C, V>
    {

        /**
         * Конструктор
         *
         * @param table Табилца
         * @param row Строка
         */
        public RowDeletedEvent(Table<R, C, V> table, R row) {
            super(table, row);
        }
    }// </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="ColumnChagnedEvent">
    /**
     * Событие изменения таблицы - колонки
     *
     * @param <R> Тип строки
     * @param <C> Тип колонки
     * @param <V> Тип значения
     */
    public static class ColumnChagnedEvent<R, C, V> extends Event<R, C, V>
    {

        protected C column = null;

        /**
         * Конструктор
         *
         * @param table таблица
         * @param column колонка
         */
        public ColumnChagnedEvent(Table<R, C, V> table, C column) {
            super(table);
            this.column = column;
        }

        /**
         * Колонка таблицы
         *
         * @return колонка
         */
        public C getColumn() {
            return column;
        }
    }

    /**
     * Колонка добавлена
     *
     * @param <R> Тип строки
     * @param <C> Тип колонки
     * @param <V> Тип значения
     */
    public static class ColumnAddedEvent<R, C, V> extends ColumnChagnedEvent<R, C, V>
    {

        /**
         * Конструктор
         *
         * @param table таблица
         * @param column колонка
         */
        public ColumnAddedEvent(Table<R, C, V> table, C column) {
            super(table, column);
        }
    }

    /**
     * Колонка удалена
     *
     * @param <R> Тип строки
     * @param <C> Тип колонки
     * @param <V> Тип значения
     */
    public static class ColumnDeletedEvent<R, C, V> extends ColumnChagnedEvent<R, C, V>
    {

        /**
         * Конструктор
         *
         * @param table таблица
         * @param column колонка
         */
        public ColumnDeletedEvent(Table<R, C, V> table, C column) {
            super(table, column);
        }
    }
    // </editor-fold>
    
    // <editor-fold defaultstate="collapsed" desc="ClearedEvent">
    /**
     * Событие - очистка таблицы
     *
     * @param <R> Тип строки
     * @param <C> Тип колонки
     * @param <V> Тип значения
     */
    public static class ClearedEvent<R, C, V> extends Event<R, C, V>
    {

        /**
         * Конструктор
         *
         * @param table Таблица
         */
        public ClearedEvent(Table<R, C, V> table) {
            super(table);
        }
    }// </editor-fold>
    
    /**
     * Реализация Listener
     * @param <Row> Тип строки таблицы
     * @param <Column> Тип колонки таблицы
     * @param <Value> Тип значения таблицы
     */
    public static class Adapter<Row,Column,Value> implements Listener<Row, Column, Value> {
        /* (non-javaDoc) @see Listener */
        @Override
        public void tableEvent(Event<Row, Column, Value> event) {
            if( event instanceof ValueChagnedEvent ){
                ValueChagnedEvent<Row,Column,Value> ve = (ValueChagnedEvent<Row, Column, Value>)event;
                valueChanged(ve.getTable(), ve.getRow(), ve.getColumn(), ve.getValue());
            }
            if( event instanceof RowAddedEvent ){
                RowAddedEvent<Row,Column,Value> e = (RowAddedEvent<Row, Column, Value>)event;
                rowAdded(e.getTable(), e.getRow());
            }
            if( event instanceof RowDeletedEvent ){
                RowDeletedEvent<Row,Column,Value> e = (RowDeletedEvent<Row, Column, Value>)event;
                rowDeleted(e.getTable(), e.getRow());
            }
            if( event instanceof ColumnAddedEvent ){
                ColumnAddedEvent<Row,Column,Value> e = (ColumnAddedEvent<Row, Column, Value>)event;
                columnAdded(e.getTable(), e.getColumn());
            }
            if( event instanceof ColumnDeletedEvent ){
                ColumnDeletedEvent<Row,Column,Value> e = (ColumnDeletedEvent<Row, Column, Value>)event;
                columnDeleted(e.getTable(), e.getColumn());
            }
            if( event instanceof ClearedEvent ){
                ClearedEvent<Row,Column,Value> e = (ClearedEvent<Row,Column,Value>)event;
                cleared(e.getTable());
            }
        }
        
        /**
         * Таблица очищена
         * @param table таблица
         */
        protected void cleared(Table<Row,Column,Value> table){
        }
        
        /**
         * Значение в таблице изменилось
         * @param table Таблица
         * @param row Строка
         * @param column Колонка
         * @param value Значение
         */
        protected void valueChanged(Table<Row,Column,Value> table,Row row,Column column,Value value){
        }

        /**
         * Добавлена строка в таблицу
         * @param table Таблица
         * @param row Строка
         */
        protected void rowAdded(Table<Row,Column,Value> table,Row row){
        }
        
        /**
         * Удалена строка из таблицы
         * @param table Таблица
         * @param row Строка
         */
        protected void rowDeleted(Table<Row,Column,Value> table,Row row){
        }
        
        /**
         * Добавлена колонка в таблицу
         * @param table Таблица
         * @param column Колонка
         */
        protected void columnAdded(Table<Row,Column,Value> table,Column column){
        }
        
        /**
         * Удалена колонка из таблицы
         * @param table Таблица
         * @param column Колонка
         */
        protected void columnDeleted(Table<Row,Column,Value> table,Column column){
        }
    }
    
    /**
     * Значение таблицы по умолчанию
     */
    protected Value defItem = null;
    
    /**
     * Элементы таблицы
     */
    protected Map<Row, Map<Column, Value>> items = new TreeMap<Row, Map<Column, Value>>();
    
    /**
     * Строки таблицы
     */
    protected Collection<Row> rows = new TreeSet<Row>();
    
    /**
     * Колонки таблицы
     */
    protected Collection<Column> columns = new TreeSet<Column>();
    
    /**
     * Подписчики на события таблицы
     */
    protected Collection<Listener<Row,Column,Value>> listeners = new HashSet<Listener<Row, Column, Value>>();
    
    /**
     * Порядок вывода строк
     */
    protected Comparator<Row> rowComparator = null;

    /**
     * Конструктор
     * @param defaultItem Элемент по умолчанию используемый в качестве пустой ячейки
     */
    public Table(Value defaultItem)
    {
        this.defItem = defaultItem;
    }
    
    // <editor-fold defaultstate="collapsed" desc="Работа с подписчиками listeners / fireEvents">
    /**
     * Добавляет подписчика на события
     *
     * @param listener Подписчик
     */
    public void addListener(Listener<Row, Column, Value> listener) {
        if (listener != null)
            listeners.add(listener);
    }

    /**
     * Удаляет подписчика
     *
     * @param listener Подписчик
     */
    public void removeListener(Listener<Row, Column, Value> listener) {
        listeners.remove(listener);
    }

    /**
     * Возвращает список подписчиков на события
     *
     * @return Список подписчиков
     */
    public Collection<Listener<Row, Column, Value>> getListeners() {
        return listeners;
    }

    /**
     * Рассылает подписчикам событие
     *
     * @param event Событие
     */
    protected void fireEvent(Event<Row, Column, Value> event) {
        for (Listener<Row, Column, Value> l : getListeners()) {
            l.tableEvent(event);
        }
    }

    /**
     * Рассылает уведомление о измении значения таблицы
     *
     * @param row Строка
     * @param column Колонка
     * @param value Новое значение
     */
    protected void fireValueChanged(Row row, Column column, Value value) {
        ValueChagnedEvent<Row, Column, Value> e = new ValueChagnedEvent<Row, Column, Value>(this, row, column, value);
        fireEvent(e);
    }

    /**
     * Рассылает уведомление о добавлении строки
     * @param row Строка
     */
    protected void fireRowAdded(Row row) {
        RowAddedEvent<Row, Column, Value> e = new RowAddedEvent<Row, Column, Value>(this, row);
        fireEvent(e);
    }

    /**
     * Рассылает уведомление о удалении строки
     * @param row Строка
     */
    protected void fireRowDeleted(Row row) {
        RowDeletedEvent<Row, Column, Value> e = new RowDeletedEvent<Row, Column, Value>(this, row);
        fireEvent(e);
    }

    /**
     * Рассылает уведомление о измении значения таблицы
     * @param column Строка
     */
    protected void fireColumnAdded(Column column) {
        ColumnAddedEvent<Row, Column, Value> e = new ColumnAddedEvent<Row, Column, Value>(this, column);
        fireEvent(e);
    }

    /**
     * Рассылает уведомление о измении значения таблицы
     * @param column Строка
     */
    protected void fireColumnDeleted(Column column) {
        ColumnDeletedEvent<Row, Column, Value> e = new ColumnDeletedEvent<Row, Column, Value>(this, column);
        fireEvent(e);
    }

    /**
     * Рассылка уведомления о очистке таблицы
     */
    protected void fireCleared() {
        ClearedEvent<Row, Column, Value> e = new ClearedEvent<Row, Column, Value>(this);
        fireEvent(e);
    }// </editor-fold>
    
    // <editor-fold defaultstate="collapsed" desc="copyFrom()">
    /**
     * Копирует данные из другой таблицы
     *
     * @param src Исходная таблица
     */
    public void copyFrom(Table<? extends Row, ? extends Column, ? extends Value> src) {
        if (src == null) {
            throw new IllegalArgumentException("src == null");
        }
        
        items = new TreeMap<Row, Map<Column, Value>>();
        if (rowComparator != null) {
            rows = new TreeSet<Row>(rowComparator);
        } else {
            rows = new TreeSet<Row>();
        }
        
        columns = new TreeSet<Column>();
        
        Table tsrc = src;
        Object oi = tsrc.getDefItem();
        setDefItem(oi == null ? null : (Value) oi);
        
        for (Row r : src.rows_direct()) {
            for (Column c : src.columns_direct()) {
                Object or = r;
                Object oc = c;
                oi = tsrc.get(or, oc);
                set(r, c, oi == null ? null : (Value) oi);
            }
        }
    }// </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="rowComparator">
    /**
     * Указывает порядок строк
     *
     * @return порядок строк
     */
    public Comparator<Row> getRowComparator() {
        return rowComparator;
    }

    /**
     * Указывает порядок строк
     *
     * @param newRowOrder порядок строк
     */
    public void setRowComparator(Comparator<Row> newRowOrder) {
        rowComparator = newRowOrder;
        
        Collection<Row> nrows = null;
        if (newRowOrder != null) {
            nrows = new TreeSet<Row>(newRowOrder);
        } else {
            nrows = new TreeSet<Row>();
        }
        
        nrows.addAll(rows);
        rows = nrows;
    }// </editor-fold>
    
    // <editor-fold defaultstate="collapsed" desc="getRowValues / getColumnValues ...">
    /**
     * Возвращает значения для указанной строки
     *
     * @param row Строка
     * @param colPred Фильтр колонок, возможно null
     * @return Пары Значение
     */
    public Iterable<Value> getRowValues(Row row, Predicate<Column> colPred) {
        if (row == null) {
            throw new IllegalArgumentException("row==null");
        }
        final Row r = row;
        Convertor<Column, Value> conv = new Convertor<Column, Value>()
        {

            @Override
            public Value convert(Column col) {
                Value v = get(r, col);
                return v;
            }
        };
        if (colPred != null) {
            return Iterators.convert(
                    Iterators.predicate(columns_direct(), colPred), conv);
        }
        return Iterators.convert(columns_direct(), conv);
    }

    /**
     * Возвращает значения для указанной колонки
     *
     * @param column Колонка
     * @param rowPred Фильтр строк, возможно null
     * @return Пары Значения
     */
    public Iterable<Value> getColumnValues(Column column, Predicate<Row> rowPred) {
        if (column == null) {
            throw new IllegalArgumentException("column==null");
        }
        final Column col = column;
        Convertor<Row, Value> conv = new Convertor<Row, Value>()
        {

            @Override
            public Value convert(Row row) {
                Value v = get(row, col);
                return v;
            }
        };
        if (rowPred != null) {
            return Iterators.convert(
                    Iterators.predicate(rows_direct(), rowPred), conv);
        }
        return Iterators.convert(rows_direct(), conv);
    }

    /**
     * Возвращает пары Строка/Значение для указанной колонки
     *
     * @param column Колонка
     * @param rowPred Фильтр строк, возможно null
     * @return Пары Строка/Значение
     */
    public Iterable<Pair<Row, Value>> getColumnRowValues(Column column, Predicate<Row> rowPred) {
        if (column == null) {
            throw new IllegalArgumentException("column==null");
        }
        final Column col = column;
        Convertor<Row, Pair<Row, Value>> conv = new Convertor<Row, Pair<Row, Value>>()
        {

            @Override
            public Pair<Row, Value> convert(Row row) {
                Value v = get(row, col);
                return new BasicPair<Row, Value>(row, v);
            }
        };
        if (rowPred != null) {
            return Iterators.convert(
                    Iterators.predicate(rows_direct(), rowPred), conv);
        }
        return Iterators.convert(rows_direct(), conv);
    }

    /**
     * Возвращает пары Колонка/Значение для указанной строки
     *
     * @param row Строка
     * @param columnPred Фильтр колонок, возможно null
     * @return Пары Колонка/Значение
     */
    public Iterable<Pair<Column, Value>> getRowColumnValues(Row row, Predicate<Column> columnPred) {
        if (row == null) {
            throw new IllegalArgumentException("row==null");
        }
        final Row r = row;
        Convertor<Column, Pair<Column, Value>> conv = new Convertor<Column, Pair<Column, Value>>()
        {

            @Override
            public Pair<Column, Value> convert(Column col) {
                Value v = get(r, col);
                return new BasicPair<Column, Value>(col, v);
            }
        };
        if (columnPred != null) {
            return Iterators.convert(
                    Iterators.predicate(columns_direct(), columnPred), conv);
        }
        return Iterators.convert(columns_direct(), conv);
    }// </editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Значение по умолчанию">
    /**
     * Возвращает Элемент по умолчанию используемый в качестве пустой ячейки
     * @return Элемент
     */
    public Value getDefItem()
    {
        return defItem;
    }
    
    /**
     * Устанавливает Элемент по умолчанию используемый в качестве пустой ячейки
     * @param defItem Элемент
     */
    public void setDefItem(Value defItem)
    {
        this.defItem = defItem;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="isRowExists() isColumnExists()">
    /**
     * Проверка сушествует ли строка в таблице
     * @param r Строка
     * @return true - существует
     */
    public boolean isRowExists(Row r)
    {
        return items.containsKey(r);
    }
    
    /**
     * Проверка сушествует ли колонка в таблице
     * @param c Колонка
     * @return true - существует
     */
    public boolean isColumnExists(Column c)
    {
        return columns.contains(c);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="rows() columns()">
    /**
     * Возвращает строки таблицы
     * @return Строки
     */
    public Iterable<Row> rows()
    {
        return copyRows() ? rows_copy() : rows_direct();
    }
    
    protected boolean copyRows(){ return false; }
    protected boolean copyColumns(){ return false; }
    
    /**
     * Возвращает строки таблицы, результат копируется в буффер
     * @return Скопированные строки
     */
    protected Iterable<Row> rows_copy()
    {
        return new ArrayList<Row>(rows);
    }
    
    /**
     * Возвращает строки таблицы без копирования в буффер
     * @return Строки таблицы
     */
    protected Iterable<Row> rows_direct()
    {
        return rows;
    }
    
    /**
     * Возвращает колонки таблицы
     * @return Колонки таблицы
     */
    public Iterable<Column> columns()
    {
        return copyColumns() ? columns_copy() : columns_direct();
    }
    
    /**
     * Возвращает колонки таблицы, результат копируется в буффер
     * @return Колонки таблицы
     */
    protected Iterable<Column> columns_copy()
    {
        return new ArrayList<Column>(columns);
    }
    
    /**
     * Возвращает колонки таблицы без копирования в буффер
     * @return Колонки таблицы
     */
    protected Iterable<Column> columns_direct()
    {
        return columns;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="addRow / addColumn">
    /**
     * Добавляет строку в таблицу
     * @param r Строка
     */
    public void addRow(Row r)
    {
        items.put(r, new TreeMap<Column, Value>());
        rows.add(r);
        fireRowAdded(r);
    }

    /**
     * Добавляет колонку в таблицу
     * @param c колонка
     */
    public void addColumn(Column c)
    {
        columns.add(c);
        fireColumnAdded(c);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="removeRow / removeColumn">
    /**
     * Очистка таблицы от всех строк, колонок и значений
     */
    public void clear(){
        items.clear();
        rows.clear();
        columns.clear();
        fireCleared();
    }
    
    /**
     * Удаляет строку таблицы
     * @param r Строка
     */
    public void removeRow(Row r)
    {
        if (items.containsKey(r))
        {
            items.remove(r);
        }

        if (rows.contains(r))
        {
            rows.remove(r);
        }
        
        fireRowDeleted(r);
    }

    /**
     * Удаляет колонку из таблицы
     * @param c Колонка
     */
    public void removeColumn(Column c)
    {
        if (columns.contains(c))
        {
            columns.remove(c);
        }

        for (Row r : items.keySet())
        {
            if (items.get(r).containsKey(c))
            {
                items.get(r).remove(c);
            }
        }
        
        fireColumnDeleted(c);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="rowsCount()">
    /**
     * Возвращает кол-во строк в таблице
     * @return Кол-во строк
     */
    public int rowsCount()
    {
        return rows.size();
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="columnsCount()">
    /**
     * Возвращает кол-во колонок в таблице
     * @return Кол-во колонок
     */
    public int columnsCount()
    {
        return columns.size();
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="normalize()">
    /**
     * Добавляет необходимые строки в items и значения по умолчанию.
     */
    protected void normalize()
    {
        for (Row r : rows_direct())
        {
            if (!items.containsKey(r))
            {
                items.put(r, new TreeMap<Column, Value>());
            }
        }
        
        for (Row r : rows_direct())
        {
            Map<Column, Value> rData = items.get(r);
            for (Column c : columns_direct())
            {
                if (!rData.containsKey(c))
                {
                    rData.put(c, defItem);
                }
            }
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="set / get">
    /**
     * Устанавливает элемент таблицы
     * @param r Строка
     * @param c Колонка
     * @param i Значение
     */
    public void set(Row r, Column c, Value i)
    {
        boolean needNorm = false;
        
        if (!isRowExists(r))
        {
            addRow(r);
            needNorm = true;
        }
        
        if (!isColumnExists(c))
        {
            addColumn(c);
            needNorm = true;
        }
        
        if (needNorm)
        {
            normalize();
        }
        
        items.get(r).put(c, i);
        fireValueChanged(r, c, i);
    }
    
    /**
     * Возвращает элемент таблицы
     * @param r Строка
     * @param c Колонка
     * @return Значение
     */
    public Value get(Row r, Column c)
    {
        if (!isRowExists(r))
        {
            return defItem;
        }
        if (items.get(r).containsKey(c))
        {
            return items.get(r).get(c);
        }
        
        return defItem;
    }
    //</editor-fold>
}
