/*
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.tree;

import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Stack;
//import java.util.function.Consumer;
import xyz.cofe.collection.BasicVisitor;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.NodesExtracter;
import xyz.cofe.collection.Visitor;
import xyz.cofe.collection.iterators.TreeIterable;
import xyz.cofe.collection.iterators.TreeWalk;
import xyz.cofe.collection.iterators.TreeWalkItreator;
import xyz.cofe.collection.iterators.TreeWalkType;
import xyz.cofe.common.Reciver;

/**
 * Абстрактный узел дерева
 * @author gocha
 * @param <Node> Тип узла
 */
public abstract class AbstractTreeNode<Node extends TreeNode> 
    implements TreeNode<Node>, 
               TreeNodeSetParent<Node>, 
               TreeNodeUpdateParent<Node>,
               TreeNodeGetNodePath<Node>
{
    /**
     * Синхронное выполнение участка кода
     * @param run код
     * @param method имя метода или null
     * @param args аргументы или null
     * @return результат
     */
    protected Object syncrun( Func0 run, String method, Object ... args ){
        if( run==null )throw new IllegalArgumentException( "run==null" );
        return run.apply();
    }
    
    /**
     * Возвращает путь от корня к узлу
     * @param node узел
     * @return путь от корня к узлу
     */
    public static List getNodePath( TreeNode node ){
        LinkedList path = new LinkedList();
        
        TreeNode n = (TreeNode)node;
        while( true ){
            if( path.contains((Object)n) ){
                throw new Error( "cycle detected" );
            }
            
            path.add(0, n);
            
            TreeNode p = (TreeNode)n.getParent();
            if( p==null ){
                break;
            }
            
            n = p;
        }
        
        return path;
    }

    @Override
    public List<Node> getNodePath() {
        // java 8
        /*
        return (List)
            syncrun( ()->{
                return TreeNodeGetNodePath.super.getNodePath();
            }, "getNodePath" );
        */
        return (List)
            syncrun( new Func0() {
                @Override
                public Object apply() {
                    return getNodePath( AbstractTreeNode.this );
            }}, "getNodePath" );
    }
    
    /**
     * Указывает на родительский узел или null, если нет роди
     */
    protected Node parent = null;

    /* (non-Javadoc@ @see TreeNode */
    @Override
    public Node getParent() {
        // java 8
        //return (Node) syncrun( ()->{ return parent; }, "getParent" );
        return (Node) syncrun( 
                new Func0() {
                @Override
                public Object apply() {
                    return parent;
                }
            }
            ,"getParent" );
    }

    /* (non-Javadoc@ @see TreeNode */
    @Override
    public void setParent(final Node tn) {
        // java 8
        //syncrun( ()->{ this.parent = tn; return null; }, "setParent", tn );
        syncrun( new Func0() {
            @Override
            public Object apply() {
                parent = tn;
                return null;
            }
        }, "setParent", tn );        
    }

    @Override
    public boolean updateParent(final Node oldparent, final Node newparent)
    {
        // java 8
        /*
        return (Boolean)syncrun( ()->{ 
            if( this.parent == oldparent ){
                this.parent = newparent;
                return true; 
            }
            return false;
        }, "updateParent", oldparent, newparent );
        */
        
        return (Boolean)syncrun( new Func0() {
            @Override
            public Object apply() { 
            if( parent == oldparent ){
                parent = newparent;
                return true; 
            }
            return false;
        }}, "updateParent", oldparent, newparent );
    }
    
	/**
	 * Возвращает собственный дочерний индекс
     * @param thisNode Узел для которого возвращает индекс
	 * @return Индекс или -1, если нет родительского элемента
	 */
	public static int getIndex(TreeNode thisNode){
		if (thisNode== null) {
			throw new IllegalArgumentException("thisNode==null");
		}

		Object parent = thisNode.getParent();
		if( parent==null )return -1;
		if( !(parent instanceof TreeNode) )return -1;

		TreeNode parentTN = (TreeNode)parent;
		Object[] siblings = parentTN.getChildren();
		if( siblings==null )return -1;

		int idx = -1;
		for( idx=0; idx<siblings.length; idx++ ){
			Object sibling = siblings[idx];
			if( sibling!=null && sibling==thisNode )return idx;
		}

		return -1;
	}

    /* (non-Javadoc@ @see TreeNode */
	@Override
	public int getIndex() {        
		// java 8
		//return (Integer)syncrun( ()->{ return getIndex(this); }, "getIndex" );
        
        
		return (Integer)syncrun( 
            new Func0() {
                @Override 
                public Object apply() { 
                    return AbstractTreeNode.getIndex(AbstractTreeNode.this); 
                } 
            }, 
            "getIndex" 
        );
	}

    /* (non-Javadoc@ @see TreeNode */
	@Override
	public Node getSibling(int offset) {
        // java 8
        /*
        return (Node)syncrun( ()->{ 
            Object o = getSibling(this, offset);
            return (Node)o;
        }, "getSibling", offset); */
        
        final TreeNode self = this;
        final int foffset = offset;

        return (Node) syncrun(new Func0() {
            @Override
            public Object apply() {
                Object o = getSibling(self, foffset);
            return (Node)o;
            }
        }, "getSibling", offset);
    }

    /* (non-Javadoc@ @see TreeNode */
    @Override
    public Node getNextSibling() {
        return getSibling(1);
    }

    /* (non-Javadoc@ @see TreeNode */
    @Override
    public Node getPreviousSibling() {
        return getSibling(-1);
    }
    
	/**
	 * Возвращает соседний узел
     * @param thisNode Узел относительно которого происходит смещение
	 * @param offset Смещение от узла
	 * @return Узел или null в случаи достижения края
	 */
	public static Object getSibling(TreeNode thisNode, int offset){
		if (thisNode== null) {
			throw new IllegalArgumentException("thisNode==null");
		}

		if( offset==0 )return thisNode;

		Object parent = thisNode.getParent();
		if( parent==null )return null;
		if( !(parent instanceof TreeNode) )return null;

		TreeNode parentTN = (TreeNode)parent;
		Object[] siblings = parentTN.getChildren();
		if( siblings==null )return null;

		int selfIdx = -1;
		int idx = -1;
		for( idx=0; idx<siblings.length; idx++ ){
			Object sibling = siblings[idx];
			if( sibling!=null && sibling==thisNode ){
				selfIdx = idx;
			}
		}

		if( selfIdx<0 )return null;

		int targetIdx = selfIdx + offset;
		if( targetIdx<0 )return null;
		if( targetIdx>=siblings.length )return null;
		return siblings[targetIdx];
	}

    /**
     * Возвращает итератор по узлам дерева
     * @param walkType Тип обхода
     * @return Обход дерева
     */
    public Iterable<TreeWalk<Node>> tree( final TreeWalkType walkType ){
        // java 8
        //return (Iterable)syncrun( ()->{ return tree( this, walkType ); }, "tree", walkType );
        return (Iterable)syncrun( new Func0() {
            @Override
            public Object apply() {
                return tree( AbstractTreeNode.this, walkType ); 
            }}, "tree", walkType );
    }

    /**
     * Возвращает итератор по узлам дерева
     * @return Обход дерева
     */
    @Override
    public Iterable<TreeWalk<Node>> tree(){
        return tree( this, null );
    }

    /**
     * Обход вершин, начиная с указанной в определенном порядке
     * @param <Node> Тип вершины (узла дерева)
     * @param start Начальная вершина
     * @param walkType Тип обхода, null - ByBranchForward
     * @return Итератор
     */
    public static <Node extends TreeNode> Iterable<TreeWalk<Node>> tree( TreeNode<Node> start, TreeWalkType walkType ){
        if( start==null )throw new IllegalArgumentException( "start==null" );
        if( walkType==null )walkType = TreeWalkType.ByBranchForward;

        final TreeWalkType ftwt = walkType;
        final NodesExtracter next = defaultNodesExtracter;
        final Object root = start;

        Iterable itr = new Iterable() {
            @Override
            public Iterator iterator() {
                return
                    new TreeWalkItreator(
                        root,
                        next,
                        0, 1,
                        ftwt, null, null);
            }
        };

        return itr;
    }

    /**
     * Обход вершин, начиная с указанной в определенном порядке
     * @param <Node> Тип вершины (узла дерева)
     * @param start Начальная вершина
     * @param walkType Тип обхода
     * @param follow Функция проверки обхода (возможно null)
     * @return Итератор
     */
    public static <Node extends TreeNode> Iterable<TreeWalk<Node>>
        tree(
            TreeNode<Node> start,
            TreeWalkType walkType,
            final Func1<Boolean,TreeNode<Node>> follow
        )
    {
        if( start==null )throw new IllegalArgumentException( "start==null" );
        if( walkType==null )walkType = TreeWalkType.ByBranchForward;

        final TreeWalkType ftwt = walkType;
        final NodesExtracter next = new NodesExtracter(){
            @Override
            public Iterable extract(Object from) {
                if( from instanceof TreeNode ){
                    TreeNode tn = (TreeNode)from;
                    if( follow!=null ){
                        if( !follow.apply(tn) ){
                            return null;
                        }
                    }
                    return Arrays.asList(tn.getChildren());
                }
                return null;
            }
        };

        final Object root = start;

        Iterable itr = new Iterable() {
            @Override
            public Iterator iterator() {
                return
                    new TreeWalkItreator(
                        root,
                        next,
                        0, 1,
                        ftwt, null, null);
            }
        };

        return itr;
    }

    /**
     * Возвращает итератор по узлам дерева
     * @param <Node> Тип узла
     * @param start Начальный узел
     * @param follow Функция проверки обхода (возможно null)
     * @return Обход дерева
     */
    public static <Node extends TreeNode> Iterable<Node>
        walk(
            TreeNode<Node> start,
            final Func1<Boolean,TreeNode<Node>> follow )
    {
        if( start==null )throw new IllegalArgumentException( "start==null" );
//        NodesExtracter next = defaultNodesExtracter;
        final NodesExtracter next = new NodesExtracter(){
            @Override
            public Iterable extract(Object from) {
                if( from instanceof TreeNode ){
                    TreeNode tn = (TreeNode)from;
                    if( follow!=null ){
                        if( !follow.apply(tn) ){
                            return null;
                        }
                    }
                    return Arrays.asList(tn.getChildren());
                }
                return null;
            }
        };
        TreeIterable titr = new TreeIterable(start, next);
        return titr;
    }

    /**
     * Возвращает итератор по узлам дерева
     * @param <Node> Тип узла
     * @param start Начальный узел
     * @return Обход дерева
     */
    public static <Node extends TreeNode> Iterable<Node> walk( TreeNode<Node> start ){
        if( start==null )throw new IllegalArgumentException( "start==null" );
        NodesExtracter next = defaultNodesExtracter;
        TreeIterable titr = new TreeIterable(start, next);
        return titr;
    }

    /**
     * Возвращает итератор по узлам дерева
     * @return Обход дерева
     */
    @Override
    public Iterable<Node> walk(){
        // java 8
        // return (Iterable)syncrun( ()->{ return walk( this ); }, "walk" );
        return (Iterable)syncrun( new Func0() {
            @Override
            public Object apply() {
                return walk( AbstractTreeNode.this ); 
            }}, "walk" );
    }

    protected static NodesExtracter defaultNodesExtracter =
        new NodesExtracter(){
            @Override
            public Iterable extract(Object from) {
                if( from instanceof TreeNode ){
                    return Arrays.asList(((TreeNode)from).getChildren());
                }
                return null;
            }
        };

    public static <Node extends TreeNode> void visit( final Visitor<Node> visitor, TreeNode<Node> startNode ){
        NodesExtracter next;

//        next = new NodesExtracter(){
//            @Override
//            public Iterable extract(Object from) {
//                if( from instanceof TreeNode ){
//                    return Arrays.asList(((TreeNode)from).getChildren());
//                }
//                return null;
//            }
//        };

        next = defaultNodesExtracter;

        BasicVisitor.visit((Visitor)visitor, startNode, next);
    }
    
    @Override
    public void visit( final Reciver<Stack<Node>> visiter ){
        if (visiter== null) {
            throw new IllegalArgumentException("visiter==null");
        }
        
        // java 8
        /* syncrun( ()->{ 
            visit( stack -> {
                visiter.accept(stack);
                return true;
                }, 
                stack -> {}
            );
            return null;
        }, "visit", visiter, visiter );
        */ 
        syncrun( new Func0() {
            @Override
            public Object apply() {
                visit(
                    new Func1<Boolean,Stack<Node>>(){
                        @Override
                        public Boolean apply( Stack<Node> stack ){
                            visiter.recive(stack);
                            return true;
                        }
                    }
                );
            return null;
        }}, "visit", visiter, visiter );
    }

    @Override
    public void visit( final Reciver<Stack<Node>> visiterEnter, final Reciver<Stack<Node>> visiterExit ){
        if (visiterEnter== null) throw new IllegalArgumentException("visiterEnter==null");
        if (visiterExit== null) throw new IllegalArgumentException("visiterExit==null");

        /*
        java 8
        
        syncrun( ()->{ 
            visit( stack -> {                
                    visiterEnter.accept(stack);
                    return true;
                }, 
                stack -> {
                    visiterExit.accept(stack);
                    return null;
                }
            );
            return null;
        }, "visit", visiterEnter, visiterExit );
        */
        
        syncrun( new Func0() {
            @Override
            public Object apply() {
                visit( 
                    new Func1<Boolean, Stack<Node>>() {
                    @Override
                    public Boolean apply(Stack<Node> stack) {
                        visiterEnter.recive(stack);
                        return true;
                    }
                    },
                    new Func1<Object, Stack<Node>>() {
                    @Override
                    public Object apply(Stack<Node> stack) {
                        visiterExit.recive(stack);
                        return null;
                    }
                    }
                );
                return null;
            }
        }, "visit", visiterEnter, visiterExit );
    }

    /**
     * Обход дерева
     * @param visiter функция посещения узла, если вернет true - то будет продолжен движение вглубь по дереву.
     * возможно null.
     * @param exitVisiter функция выхода из узла. возможно null.
     */
    @Override
    public void visit( final Func1<Boolean,Stack<Node>> visiter, final Func1<Object,Stack<Node>> exitVisiter ){
        final TreeNode<Node> self = this;
        
        syncrun( new Func0() {
            @Override
            public Object apply() {
                visit( visiter, exitVisiter, self );
                return null;
            }
        }, "visit", visiter, exitVisiter );

        // java 8
        /*
        syncrun( ()->{ 
            visit( visiter, exitVisiter, this );
            return null;
        }, "visit", visiter, exitVisiter );
        */
    }

    /**
     * Обход дерева
     * @param visiter функция посещения узла, если вернет true - то будет продолжен движение вглубь по дереву.
     * возможно null.
     */
    public void visit( final Func1<Boolean,Stack<Node>> visiter ){
        // java 8
        // syncrun( ()->{ visit( visiter, null, this ); return null; }, "visit", visiter );
        
        final TreeNode<Node> self = this;
        syncrun( new Func0() {
            @Override
            public Object apply() {
                visit( visiter, null, self ); return null;
            }
        }, "visit", visiter);
    }

    public static <Node extends TreeNode> void visit(
        final Func1<Boolean,Stack<Node>> visiter,
        final Func1<Object,Stack<Node>> exitVisiter,
        TreeNode<Node> startNode
    ){
        visit(new BasicVisitor<Node>(){
                Stack<Node> path = new Stack<Node>();

                @Override
                public void exit(Node obj) {
                    if( exitVisiter!=null )exitVisiter.apply(path);
                    path.pop();
                    super.exit(obj);
                }


                @Override
                public boolean enter(Node obj) {
                    path.push(obj);
                    if( visiter==null )return true;
                    return visiter.apply(path);
                }
            },
            startNode);
    }

    public static <Node extends TreeNode> void visit( 
        final Func1<Boolean,Stack<Node>> visiter, 
        TreeNode<Node> startNode 
    ){
        visit(new BasicVisitor<Node>(){
                Stack<Node> path = new Stack<Node>();

                @Override
                public void exit(Node obj) {
                    path.pop();
                    super.exit(obj);
                }


                @Override
                public boolean enter(Node obj) {
                    if( visiter==null )return true;
                    path.push(obj);
                    return visiter.apply(path);
                }
            },
            startNode);
    }
}
