/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.collection.tree;


import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
//import xyz.cofe.collection.list.BasicEventList;
import xyz.cofe.collection.list.EventList;
import xyz.cofe.collection.list.IndexEventList;
import xyz.cofe.collection.list.SimpleListAdapter;

/**
 * Базовая реализация tree node. 
 * <ul>
 * <li>
 * Дочерниее узлы храняться виде списка (EventList).
 * </li><li>
 * После добавления/удаления дочернего узла вызываются методы onTreeNodeAdded / onTreeNodeRemoved
 * </li><li>
 * Для каждого добавляемого узла устанавливается родитель (TreeNodeSetParent)
 * </li><li>
 * onTreeNodeAdded - устанавливает родителя для дочернего узла
 * </ul>
 * Пример использования:
 * <div class="font-family: monospace;">
 public static class TNode extends BasicTreeNode&lt;TNode&gt; { <br>
 &nbsp;       public String name = null; <br>
        <br>
 &nbsp;       &#64;Override <br>
 &nbsp;       public TNode[] getChildren()<br>
 &nbsp;       {<br>
 &nbsp; &nbsp;           return getChildrenList().toArray( new TNode[]{} );<br>
 &nbsp;       }<br>
<br>
 &nbsp;       &#64;Override<br>
 &nbsp;       public String toString()<br>
 &nbsp;       {<br>
 &nbsp; &nbsp;           return "TNode{" + "name=" + name + '}';<br>
 &nbsp;       }<br>
    }
 * </div>
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 * @param <Node> Тип узла
 */
public abstract class BasicTreeNode<Node extends TreeNode> 
extends AbstractTreeNode<Node>
implements 
    TreeNodeAppendChild<Node>,
    TreeNodeChildrenList<Node>,
    TreeNodeInsertChild<Node>,
    TreeNodeRemoveChild<Node>,
    TreeNodeRemoveChildAt,
    TreeNodeReplaceChild<Node>,
    OnTreeNodeAdded<Node>,
    OnTreeNodeAdding<Node>,
    OnTreeNodeRemoved<Node>,
    OnTreeNodeRemoving<Node>,
    TreeNodeAssignSelfParent<Node>
//  ,TreeNodeNotifier<Node>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(BasicTreeNode.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    /**
     * Дочерние элементы
     */
    private EventList<Node> childrenList;
    
    /**
     * Получение списка дочерних элементов
     * @return список дочерних элементов
     */
    @Override
    public EventList<Node> getChildrenList(){
        return (EventList)syncrun( new Func0() {
            @Override
            public Object apply() {
            if( childrenList!=null )return childrenList;
            childrenList = createEventList();
            if( childrenList!=null ){
                attachListeners(childrenList);
            }
            return childrenList;
        }}, "getChildrenList" );
    }
    
    /**
     * Созданисе списка дочерних узлов.
     * @return список, по умолчанию IndexEventList
     */
    protected EventList<Node> createEventList(){
        //return new IndexEventList<>();
        return (EventList)syncrun( new Func0() {
            @Override
            public Object apply() {
            return new IndexEventList<Node>(); 
            }}, "createEventList" );
    }
    
    /**
     * Добавляет подписчиков (notifyListener) на список узлов
     * @param elist список узлов
     * @see #notifyListener
     */
    protected void attachListeners( final EventList<Node> elist ){
        syncrun( new Func0() {
            @Override
            public Object apply() {
            if( elist!=null ){
    //            elist.addEventListListener(assignParentListener,true);
                elist.addEventListListener(notifyListener, true);
            }
            return null;
        }}, "attachListeners", elist );
    }
    
    /**
     * Подписчик уведомляющий о изменении структуры: <br>
     * При добавлении узла вызывается метод onTreeNodeAdded <br>
     * При удалении узла вызывается метод onTreeNodeRemoved <br>
     */
    protected final SimpleListAdapter<Node> notifyListener = new SimpleListAdapter<Node>(){
        @Override
        protected void added(Node e, EventList<Node> list, Integer position)
        {
            onTreeNodeAdded(position, e);
        }

        @Override
        protected void adding(Node e, EventList<Node> list, Integer position)
        {
            onTreeNodeAdding(position, e);
        }

        @Override
        protected void removed(Node e, EventList<Node> list, Integer position)
        {
            onTreeNodeRemoved(position, e);
        }

        @Override
        protected void removing(Node e, EventList<Node> list, Integer position)
        {
            onTreeNodeRemoving(position, e);
        }
    };
    
    @Override
    public void onTreeNodeAdded(Integer index,final Node child){
        syncrun( new Func0() {
            @Override
            public Object apply() {         
            //TreeNodeAssignSelfParent.super.onTreeNodeAdded(index, child);
            // TreeNodeNotifier.super.onTreeNodeAdded(index, child);
                assignParentTo(child);
            return null;
            }}, "onTreeNodeAdded", index, child );
    }
    
    /**
     * Устанавливает this в качестве родитеской ссылки
     * добавленному дочернему узлу
     * @param inserted дочерний узел
     * @see #onTreeNodeAdded
     */
    protected void assignParentTo( Object inserted ){
        if( inserted instanceof TreeNodeSetParent ){
            TreeNodeSetParent<TreeNode> tn = (TreeNodeSetParent<TreeNode>)inserted;
            tn.setParent(this);
        }
    }
    
    @Override
    public void onTreeNodeAdding(Integer index,Node child){
    }
    
    @Override
    public void onTreeNodeRemoved(Integer index, final Node child){
        syncrun( new Func0() {
            @Override
            public Object apply() {
            //TreeNodeAssignSelfParent.super.onTreeNodeRemoved(index, child);
            resetParent( child );
            return null;
            }}, "onTreeNodeRemoved", index, child );
//        TreeNodeNotifier.super.onTreeNodeRemoved(index, child);
    }
    
    /**
     * Сбрасывает ссылку parent дочернего узела, при удалении (onTreeNodeRemoved)
     * @param child дочерний узел, должен поддерживать TreeNodeUpdateParent
     * @see #onTreeNodeRemoved
     */
    protected void resetParent( Node child ){
        if( child instanceof TreeNodeUpdateParent ){
            ((TreeNodeUpdateParent)child).updateParent(this, null);
        }
    }

    @Override
    public void onTreeNodeRemoving(Integer index,Node child){
    }    
    
//    /**
//     * Подписчик устанавливающий BasicTreeNode.this в качестве родитеской ссылки
//     * добавленному дочернему узлу3
//     * @see #assignParentTo(java.lang.Object) 
//     */
//    protected final SimpleListAdapter assignParentListener = new SimpleListAdapter(){
//        @Override
//        protected void added(Object e, EventList list, Integer position) {
//            assignParentTo(e);
//        }
//    };
//    
//    /**
//     * Устанавливает BasicTreeNode.this в качестве родитеской ссылки
//     * добавленному дочернему узлу
//     * @param inserted дочерний узел
//     */
//    protected void assignParentTo( Object inserted ){
//        if( inserted instanceof TreeNodeSetParent ){
//            TreeNodeSetParent<TreeNode> tn = (TreeNodeSetParent<TreeNode>)inserted;
//            tn.setParent(BasicTreeNode.this);
//        }
//    }
    
    @Override
    public Node appendChild(final Node child) {
        if( child==null )throw new IllegalArgumentException( "child==null" );
        syncrun( new Func0() {
            @Override
            public Object apply() {         
            getChildrenList().add(child); return null; 
            }}, "appendChild", child );
        return (Node)this;
    }

    @Override
    public void insertChildAt(final int index, final Node child) {
        if( child==null )throw new IllegalArgumentException( "child==null" );
        syncrun( new Func0() {
            @Override
            public Object apply() {         
            getChildrenList().add(index, child); return null; 
            }}, "insertChildAt", index, child );
    }

    @Override
    public void removeChild(final Node child) {
        syncrun( new Func0() {
            @Override
            public Object apply() {         
            getChildrenList().remove(child); return null; 
            }}, "removeChild", child );
    }

    @Override
    public void removeChildAt(final int idx) {
        syncrun( new Func0() {
            @Override
            public Object apply() {         
            getChildrenList().remove(idx); return null; 
            }}, "removeChildAt", idx );
    }

    @Override
    public void setChild(final int index, final Node node) {
        syncrun( new Func0(){
            @Override
            public Object apply() {
            if( node==null )throw new IllegalArgumentException( "node==null" );
            getChildrenList().set(index, node);
            return null;
        }}, "setChild", index, node );
    }
}
