/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.collection.tree;


import java.io.Closeable;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Iterators;
import xyz.cofe.collection.NodesExtracter;
import xyz.cofe.collection.iterators.EmptyIterable;
import xyz.cofe.collection.map.BasicEventMap;
import xyz.cofe.collection.map.EventMap;
import xyz.cofe.collection.map.LockEventMap;
import xyz.cofe.collection.map.SimpleMapAdapter;
import xyz.cofe.collection.set.ClassSet;

/**
 *
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class ClassNode
implements NodesExtracter
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(ClassNode.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    protected final Lock lock;
    protected final Map<Class,Set<NodesExtracter>> typedExtracter;
    protected final ClassSet types;
    
    public ClassNode(){
        lock = new java.util.concurrent.locks.ReentrantLock();
        
        Map<Class, Set<NodesExtracter>>
        typedExtracter = new LinkedHashMap<Class, Set<NodesExtracter>>();
        types = new ClassSet(true);
        
        BasicEventMap<Class,Set<NodesExtracter>> ev_typedExtracter 
            = new LockEventMap<Class,Set<NodesExtracter>>(typedExtracter, lock);
        typedExtracter = ev_typedExtracter;

        this.typedExtracter = typedExtracter;
        syncTypes(types, ev_typedExtracter);
    }
    
    public ClassNode(Lock lock){
        this( (ClassNode)null, lock );
    }
    
    public ClassNode(ClassNode src, Lock lock){
        this.lock = lock==null ? new java.util.concurrent.locks.ReentrantLock() : lock;

        Map<Class, Set<NodesExtracter>>
        typedExtracter = new LinkedHashMap<Class, Set<NodesExtracter>>();
        types = new ClassSet(true);
        
        BasicEventMap<Class,Set<NodesExtracter>> ev_typedExtracter 
            = new LockEventMap<Class,Set<NodesExtracter>>(typedExtracter, lock);
        typedExtracter = ev_typedExtracter;

        this.typedExtracter = typedExtracter;
        syncTypes(types, ev_typedExtracter);
        
        if( src!=null ){
            try{
                src.lock.lock();
                this.typedExtracter.putAll(src.typedExtracter);
            }finally{
                src.lock.unlock();
            }
        }
    }
    
    @Override
    public ClassNode clone(){
        return new ClassNode(this, null);
    }
    public ClassNode clone(Lock lock){
        return new ClassNode(this, lock);
    }
    
    private Closeable syncTypes( final ClassSet types, EventMap<Class,Set<NodesExtracter>>  map ){
        SimpleMapAdapter<Class,Set<NodesExtracter>> mapListener
            = new SimpleMapAdapter<Class,Set<NodesExtracter>>(){
                @Override
                protected void removed(
                    Set<NodesExtracter> value, 
                    EventMap<Class, Set<NodesExtracter>> map, 
                    Class key) 
                {
                    try {
                        lock.lock();
                        if( types!=null && key!=null ){
                            types.remove(key);
                        }
                    }
                    finally {
                        lock.unlock();
                    }
                }
                @Override
                protected void added(
                    Set<NodesExtracter> value, 
                    EventMap<Class, Set<NodesExtracter>> map, 
                    Class key) {
                    try {
                        lock.lock();
                        if( types!=null && key!=null ){
                            types.add(key);
                        }
                    }
                    finally {
                        lock.unlock();
                    }
                }
            };
        return map.addEventMapListener(mapListener);
    }
    
    public <T> ClassNode adds( Class<T> cls, NodesExtracter<T,?> ne ){
        if( cls==null )throw new IllegalArgumentException( "cls==null" );
        if( ne==null )throw new IllegalArgumentException( "ne==null" );
        add( cls, ne );
        return this;
    }
    
    public void add( Class cls, NodesExtracter ne ){
        if( cls==null )throw new IllegalArgumentException( "cls==null" );
        if( ne==null )throw new IllegalArgumentException( "ne==null" );
        
        try {
            lock.lock();
            Set<NodesExtracter> setne = typedExtracter.get(cls);
            if( setne==null ){
                setne = new LinkedHashSet<NodesExtracter>();
                typedExtracter.put(cls, setne);
            }
            
            setne.add(ne);
        }
        finally {
            lock.unlock();
        }
    }
    
    public void remove( Class cls, NodesExtracter ne ){
        if( cls==null )throw new IllegalArgumentException( "cls==null" );
        if( ne==null )throw new IllegalArgumentException( "ne==null" );
        
        try {
            lock.lock();
            Set<NodesExtracter> setne = typedExtracter.get(cls);
            if( setne==null ) return;
            
            setne.remove(ne);
            if( setne.isEmpty() ){
                typedExtracter.remove(cls);
            }
        }
        finally {
            lock.unlock();
        }
    }
    
    public void clear( Class cls ){
        if( cls==null )throw new IllegalArgumentException( "cls==null" );
        
        try {
            lock.lock();
            Set<NodesExtracter> setne = typedExtracter.get(cls);
            if( setne==null ) return;
            
            setne.clear();
            if( setne.isEmpty() ){
                typedExtracter.remove(cls);
            }
        }
        finally {
            lock.unlock();
        }
    }
    public void clear(){
        try {
            lock.lock();
            typedExtracter.clear();
        }
        finally {
            lock.unlock();
        }
    }
    
    public boolean hasClass( Class cls ){
        if( cls==null )throw new IllegalArgumentException( "cls==null" );
        try {
            lock.lock();
            return typedExtracter.containsKey(cls);
        }
        finally {
            lock.unlock();
        }
    }
    public Class[] getClasses(){
        try {
            lock.lock();
            return types.toArray(new Class[]{});
        }
        finally {
            lock.unlock();
        }
    }
    
    public NodesExtracter[] getNodeExtracters( Class cls ){
        if( cls==null )throw new IllegalArgumentException( "cls==null" );
        try {
            lock.lock();
            Set<NodesExtracter> childrenExtr = typedExtracter.get(cls);
            if( childrenExtr==null )return null;
            return childrenExtr.toArray(new NodesExtracter[]{});
        }
        finally {
            lock.unlock();
        }
    }
    public NodesExtracter[] extractersOf( Class cls ){
        if( cls==null )throw new IllegalArgumentException( "cls==null" );
        try {
            lock.lock();
            Set<NodesExtracter> childrenExtr = typedExtracter.get(cls);
            if( childrenExtr!=null ){
                return childrenExtr.toArray(new NodesExtracter[]{});
            }
         
            Collection<Class> matchParents = types.getAssignableFrom(cls, true, false);
            Set<Class> matchedCls = new LinkedHashSet<Class>();
            if( matchParents!=null ){
                for( Class c : matchParents ){
                    if( c!=null ){
                        matchedCls.add(c);
                        if( c.isInterface() ){
                            continue;
                        }
                        break;
                    }
                }
            }
            
            childrenExtr = new LinkedHashSet<NodesExtracter>();
            for( Class c : matchedCls ){
                Set<NodesExtracter> snet = typedExtracter.get(c);
                childrenExtr.addAll(snet);
            }
            
            if( childrenExtr!=null && childrenExtr.size()>0 ){
                return childrenExtr.toArray(new NodesExtracter[]{});
            }
            
            return new NodesExtracter[]{};
        }
        finally {
            lock.unlock();
        }
    }
    
    public Iterable fetch( Object node ){
        if( node==null )return Iterators.empty();
        
        try {
            lock.lock();
            Class cls = node.getClass();
            
            NodesExtracter[] extrs = extractersOf(cls);
            return fetch(node, extrs);
            
//            Set<NodesExtracter> childrenExtr = typedExtracter.get(cls);
//            if( childrenExtr!=null && childrenExtr.size()>0 ){
//                return fetch(node, childrenExtr);
//            }
//            
//            Collection<Class> matchParents = types.getAssignableFrom(cls, true, false);
//            Class matchedCls = null;
//            if( matchParents!=null ){
//                for( Class c : matchParents ){
//                    if( c!=null ){
//                        matchedCls = c;
//                        break;
//                    }
//                }
//            }
//            
//            childrenExtr = matchedCls!=null ? typedExtracter.get(matchedCls) : null;
//            if( childrenExtr!=null && childrenExtr.size()>0 ){
//                return fetch(node, childrenExtr);
//            }
        }
        finally {
            lock.unlock();
        }
        
//        return Iterators.empty();
    }
    private Iterable fetch( Object node, NodesExtracter ... set ){
        Iterable itr = null;
        for( NodesExtracter ne : set ){
            if( ne!=null ){
                Iterable i = ne.extract(node);
                if( i!=null && !(i instanceof EmptyIterable) ){
                    if( itr==null ){
                        itr = i;
                    }else{
                        itr = Iterators.sequence(itr, i);
                    }
                }
            }
        }
        return itr==null ? Iterators.empty() : itr;
    }

    @Override
    public Iterable extract(Object from) {
        return fetch(from);
    }
}
