/*
 * The MIT License
 *
 * Copyright 2017 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.collection.tree;

import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import static xyz.cofe.collection.tree.TreeNodeDeepCounterDef.getDeepCounterOf;
import static xyz.cofe.collection.tree.TreeNodeDeepCounterDef.setDeepCounterOf;

/**
 * Реализация узла древа с поддержкой сквозного индексирования узлов. <br><br>
 * 
 * <b>Основные возможности:</b>
 * <ul>
 * <li>
 * Дочерниее узлы храняться виде списка (EventList).
 * </li><li>
 * После добавления/удаления дочернего узла вызываются методы onTreeNodeAdded / onTreeNodeRemoved
 * </li><li>
 * Для каждого добавляемого узла устанавливается родитель (TreeNodeSetParent)
 * </li><li>
 * onTreeNodeAdded - устанавливает родителя для дочернего узла
 * </li><li>
 * Уведомления о измении стркутуры древа. События TreeNodeAdded / TreeNodeRemoved
 * </li>
 * <li>
 * Скозная нумерация узлов
 * </li>
 * </ul>
 * 
 * 
 * Все узлы древа нумеруются в порядке обхода в глубину. <br>
 * Пример 1(а):
 * <ul>
 * <li>А (0)
 * <ul>
 * <li>B (1)
 * <ul>
 * <li>D (2)</li>
 * <li>E (3)</li>
 * </ul>
 * </li>
 * <li>C (4)
 * <ul>
 * <li>F (5)
 * <ul>
 * <li>H (6)</li>
 * <li>I (7)</li>
 * <li>J (8)</li>
 * </ul>
 * </li>
 * <li>G (9)</li>
 * </ul>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * Пример 1(б):
 * <table border='1' summary="Пример дерева">
 * 
 * <tr>
 * <td align='center' colspan='6'>A (0)</td>
 * </tr>
 * 
 * <tr>
 * <td align='center' colspan='2'>B (1)</td>
 * <td align='center' colspan='4'>C (4)</td>
 * </tr>
 * 
 * <tr>
 * <td align='center' colspan='1'>D (2)</td>
 * <td align='center' colspan='1'>E (3)</td>
 * <td align='center' colspan='3'>F (5)</td>
 * <td align='center' colspan='1'>G (9)</td>
 * </tr>
 * 
 * <tr>
 * <td align='center' colspan='2'> </td>
 * <td align='center' colspan='1'>H (6)</td>
 * <td align='center' colspan='1'>I (7)</td>
 * <td align='center' colspan='1'>J (8)</td>
 * <td align='center' colspan='1'> </td>
 * </tr>
 * </table>
 * <br>
 * <b>deepOffset()</b><br>
 * Смещение +3 относительно узла A вернет узел E, а для узла E вернет узел H<br>
 * <code style='font-size:100%'>a.deepOffset(3) == e</code> <br>
 * <code style='font-size:100%'>e.deepOffset(3) == h</code> <br><br>
 * 
 * Смещение -5 относительно узла I вернет узел D: <br>
 * <code style='font-size:100%'>i.deepOffset(-5) == d</code> <br><br>
 * 
 * <b>getRootOffset()</b><br>
 * getRootOffset - возвращает смещение относительно корня.<br>
 * Так для узла A значение будет 0, а для узла H - 6:<br>
 * <code style='font-size:100%'>a.getRootOffset() == 0</code> <br>
 * <code style='font-size:100%'>h.getRootOffset() == 6</code> <br><br>
 * 
 * <b>Пример реализации класса:</b> <br>
 * 
 * <code style="font-size:100%">
 * public static class TNode <br>
  &nbsp; extends IndexTreeNode&lt;TNode&gt; { <br>
 &nbsp;       &#64;Override <br>
 &nbsp;       public TNode[] getChildren()<br>
 &nbsp;       {<br>
 &nbsp; &nbsp;           return getChildrenList().toArray( new TNode[]{} );<br>
 &nbsp;       }<br>
<br>
 &nbsp;       // Полезная нагрузка <br>
 &nbsp;       public String name = null; <br>
        <br>
 &nbsp;       &#64;Override<br>
 &nbsp;       public String toString()<br>
 &nbsp;       {<br>
 &nbsp; &nbsp;           return "TNode{" + "name=" + name + '}';<br>
 &nbsp;       }<br>
    }
    * </code>
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 * @param <Node> Тип узла
 */
public abstract class IndexTreeNode<Node extends TreeNode>
    extends PopupTreeNode<Node>
    implements 
            TreeNodeDeepCounter<Node>
            ,TreeNodeDeepOffset<Node>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(IndexTreeNode.class.getName());

    private static Level logLevel(){ 
        return logger.getLevel() ;
    }
    
    private static boolean isLogSevere(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level level = logLevel();
        return level==null 
        ? false
        : level.intValue() <= Level.FINER.intValue();
    }    
    
    private static boolean isLogFinest(){
        Level level = logLevel();
        return level==null 
        ? false
        : level.intValue() <= Level.FINEST.intValue();
    }    
    
    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    
    private static void logEntering(String method,Object ... params){
        logger.entering(IndexTreeNode.class.getName(),method,params);
    }

    private static void logExiting(String method,Object result){
        logger.exiting(IndexTreeNode.class.getName(),method,result);
    }

    private static void logExiting(String method){
        logger.exiting(IndexTreeNode.class.getName(),method);
    }
    //</editor-fold>
    
    /**
     * кол-во узлов включая себя, или null
     */
    protected Integer nodesCount = null;
    
    /**
     * Возвращает кол-во узлов включая себя и всех вложенных
     * @return кол-во узлов (всегда больше 0)
     */
    @Override
    public int getNodesCount(){
        return (int)(Integer)syncrun( new Func0() {
            @Override
            public Object apply() {
            Integer ncnt = nodesCount;
            if( ncnt!=null ){
                return ncnt;
            }
            
            int sum = 0;
            
            TreeNode[] children = getChildren();
            for( TreeNode ctn : children ){
                if( ctn==null )continue;
                
                if( ctn instanceof TreeNodeDeepCounter ){
                    TreeNodeDeepCounter tndc = (TreeNodeDeepCounter)ctn;
                    int count = tndc.getNodesCount();
                    if( count<0 ){
                        logFiner( "can't eval deep counter of {0} - return {1}",tndc,count );
                        return -3;
                    }
                    sum += count;
                }else{
                    // can't eval
                    logFiner( "can't eval deep counter of {0} not instanceof TreeNodeDeepCounter",ctn );
                    return -2;
                }
            }
            
            // self
            sum += 1;
            
            // cache
            logFiner( "cache deep counter of {0} - sum={1}",this, sum );
            
            setNodesCount(sum);
            
            return sum;
            }},"getNodesCount");
    }
    
    /*
    @Override
    public void onTreeNodeRemoved(Integer index, Node child)
    {
        syncrun( ()->{ 
            super.onTreeNodeRemoved(index, child);
            TreeNodeDeepOffset.super.onTreeNodeRemoved(index, child);
            return null;
        }, "onTreeNodeRemoved", index, child );
    }

    @Override
    public void onTreeNodeAdded(Integer index, Node child)
    {
        syncrun( ()->{ 
            super.onTreeNodeAdded(index, child);
            TreeNodeDeepOffset.super.onTreeNodeAdded(index, child);
            return null;
        }, "onTreeNodeAdded", index, child );
    }
    */
    
    @Override
    public Node deepOffset(int offset) {
        return (Node)(new TreeNodeDeepOffsetDef()).deepOffset(this, offset);
    }

    @Override
    public int getRootOffset() {
        return (new TreeNodeDeepOffsetDef()).getRootOffsetOf(this);
    }

    protected void resetNodesCount(){
        syncrun( new Func0() {
            @Override
            public Object apply() {
                nodesCount = null; return null; 
            }}, "resetNodesCount" );
    }

    protected void setNodesCount(final int cnt){
        syncrun( new Func0() {
            @Override
            public Object apply() { nodesCount = cnt; return null; }}, "resetNodesCount" );
    }
    
    /**
     * Передача события вверх по древу
     * @param ev Событие
     */
    @Override
    public void popup( final TreeNodeEvent<Node> ev ){
        syncrun( new Func0() {
            @Override
            public Object apply() {
         
            // Сброс счетчика узлов
            if( ev instanceof TreeNodeAdded || 
                ev instanceof TreeNodeRemoving ||
                ev instanceof TreeNodeRemoved ||
                ev instanceof TreeNodeAdding
            ){
                resetNodesCount();
            }

            // Передача сообщения вверх
            if( ev instanceof TreeNodePopupEvent ){
                TreeNodePopupEvent tnpe
                    = (TreeNodePopupEvent)ev;

                List l = tnpe.getPopupPath();

                // Проверка цикла
                if( l!=null ){
                    int cycles = 0;
                    for( Object o : l ){
                        if( this == o ){
                            // найден цикл
                            cycles++;
                        }
                    }
                    if( cycles>1 ){
                        throw new Error( "cycle detected" );
                        // return;
                    }
                }

                l.add(this);

                fireTreeNodeEvent(ev);

                Node pnode = getParent();
                if( pnode instanceof TreeNodeNotifier ){
                    ((TreeNodeNotifier)pnode).popup(ev);
                }
            }
            
            return null;
        }}, "popup", ev );
    }
    
    @Override
    public void onTreeNodeAdded(final Integer index,final Node child){
        syncrun( new Func0() {
            @Override
            public Object apply() { 
            resetNodesCount();
            
            //super.onTreeNodeAdded(index, child);
            //TreeNodeAssignSelfParent.super.onTreeNodeAdded(index, child);            
            // Установка parent
            if( child instanceof TreeNodeSetParent ){
                TreeNodeSetParent<TreeNode> tn = (TreeNodeSetParent<TreeNode>)child;
                tn.setParent(IndexTreeNode.this);
            }
            
            //TreeNodeNotifier.super.onTreeNodeAdded(index, child);
            // Уведомление выше стоящих
            TreeNodeAdded<Node> ev = new TreeNodeAdded(IndexTreeNode.this,child,index);
            ev.getPopupPath().add(child);

            popup( ev );
            
            return null;
        }}, "onTreeNodeAdded", index, child );
    }
    
    @Override
    public void onTreeNodeAdding(final Integer index,final Node child){
        syncrun( new Func0() {
            @Override
            public Object apply() { 
            resetNodesCount();
            
            //super.onTreeNodeAdding(index, child);
            
            //TreeNodeNotifier.super.onTreeNodeAdding(index, child);            
            // Уведомление выше стоящих
            TreeNodeAdding<Node> ev = new TreeNodeAdding(IndexTreeNode.this,child,index);
            ev.getPopupPath().add(child);
            popup( ev );
            
            return null;
        }}, "onTreeNodeAdding", index, child );
    }
    
    @Override
    public void onTreeNodeRemoved(final Integer index,final Node child){
        syncrun( new Func0() {
            @Override
            public Object apply() {
            resetNodesCount();
            
            //super.onTreeNodeRemoved(index, child);
            // Обновление parent
            if( child instanceof TreeNodeUpdateParent ){
                ((TreeNodeUpdateParent)child).updateParent(IndexTreeNode.this, null);
            }

            // TreeNodeNotifier.super.onTreeNodeRemoved(index, child);
            // Уведомление выше стоящих
            TreeNodeRemoved<Node> ev = new TreeNodeRemoved(IndexTreeNode.this,child,index);
            ev.getPopupPath().add(child);
            popup( ev );

            return null;
        }}, "onTreeNodeRemoved", index, child );
    }    
    
    @Override
    public void onTreeNodeRemoving(final Integer index,final Node child){
        syncrun( new Func0() {
            @Override
            public Object apply() {
            resetNodesCount();
            
            //super.onTreeNodeRemoving(index, child);
            
            // TreeNodeNotifier.super.onTreeNodeRemoving(index, child);
            // Уведомление выше стоящих
            TreeNodeRemoving<Node> ev = new TreeNodeRemoving(IndexTreeNode.this,child,index);
            ev.getPopupPath().add(child);
            popup( ev );

            return null;
        }}, "onTreeNodeRemoving", index, child );
    }
}
