/*
 * The MIT License
 *
 * Copyright 2017 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.collection.tree;


import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
//import java.util.stream.Collectors;

/**
 * Событие добавления узла
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class TreeNodeAdded<Node extends TreeNode>
    extends BasicTreeNodeEvent<Node>
    implements TreeNodeAddedEvent<Node>, TreeNodePopupEvent<Node>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(TreeNodeAdded.class.getName());

    private static Level logLevel(){ 
        return logger.getLevel() ;
    }
    
    private static boolean isLogSevere(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level level = logLevel();
        return level==null 
        ? false
        : level.intValue() <= Level.FINER.intValue();
    }    
    
    private static boolean isLogFinest(){
        Level level = logLevel();
        return level==null 
        ? false
        : level.intValue() <= Level.FINEST.intValue();
    }    
    
    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    
    private static void logEntering(String method,Object ... params){
        logger.entering(TreeNodeAdded.class.getName(),method,params);
    }

    private static void logExiting(String method,Object result){
        logger.exiting(TreeNodeAdded.class.getName(),method,result);
    }

    private static void logExiting(String method){
        logger.exiting(TreeNodeAdded.class.getName(),method);
    }
    //</editor-fold>

    /**
     * Конструктор
     * @param source источник события
     */
    public TreeNodeAdded(TreeNode<Node> source){
        super(source);
    }
    
    /**
     * Конструктор
     * @param parent Родительский узел в который добавлен дочерний узел
     * @param child Дочерний узел
     * @param childIndex Позиция (от нуля) в котором распологается дочерний узел
     */
    public TreeNodeAdded(TreeNode<Node> parent, TreeNode<Node> child, Integer childIndex){
        super(child);
        this.child = child;
        this.parent = parent;
        this.childIndex = childIndex;
    }
    
    protected final TreeNodePopupEventDef popupDef = new TreeNodePopupEventDef(this);

    @Override
    public List<TreeNode<Node>> getPopupPath() {
        return popupDef.getPopupPath();
    }
    
    /**
     * Родительский узел в который добавлен дочерний узел
     */
    protected TreeNode<Node> parent;

    /**
     * Возвращает родительский узел в который добавлен дочерний узел
     * @return родительский узел
     */
    @Override
    public TreeNode<Node> getParent()
    {
        return parent;
    }

    /**
     * Дочерний узел
     */
    protected TreeNode<Node> child;

    /**
     * Возвращает дочерний узел
     * @return Дочерний узел
     */
    @Override
    public TreeNode<Node> getChild()
    {
        return child;
    }
    
    /**
     * Позиция (от нуля) в котором распологается дочерний узел
     */
    protected Integer childIndex;

    /**
     * Позиция (от нуля) в котором распологается дочерний узел
     * @return индекс дочернего узла
     */
    @Override
    public Integer getChildIndex()
    {
        return childIndex;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        
        List<String> l = new LinkedList<String>();
        
        for( Object n : getPopupPath() ){
            if( n!=null ){
                l.add( 0, n.toString() );
            }else{
                l.add( 0, "" );
            }
        }
        
        /* java 8
        List<String> l2 =
        getPopupPath()
            .stream()
            .map( n -> n!=null ? n.toString() : "" )
            .collect(Collectors.toList());
        
        Collections.reverse(l);
        
        String popupPath = l.stream().collect(Collectors.joining("/"));
        */
        
        StringBuilder sb2 = new StringBuilder();
        int ssi = -1;
        for( String ss : l ){
            ssi++;
            if( ssi>0 )sb2.append("/");
            sb2.append(ss);
        }
        String popupPath = sb2.toString();
        
        //if( popupPath.length() > 40 ){
        //    popupPath = l.stream().collect(Collectors.joining("/\n  "));
        // }
        
        sb.append("{");
        
        sb.append("popup path:").append(popupPath).append("\n");
        sb.append(", ").append("parent=").append(parent).append("\n");
        sb.append(", ").append("child=").append(child).append("\n");
        sb.append(", ").append("childIndex=").append(childIndex).append("\n");
        
        sb.append("}");
        
        return sb.toString();
    }
}
