/*
 * The MIT License
 *
 * Copyright 2017 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.collection.tree;


import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Смещение к узлу относительно данного узла. <br>
 * Реализация по умолчанию
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 * @param <Node> Тип узла
 */
public class TreeNodeDeepOffsetDef<Node extends TreeNodeDeepOffset>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(TreeNodeDeepOffsetDef.class.getName());

    private static Level logLevel(){ 
        return logger.getLevel() ;
    }
    
    private static boolean isLogSevere(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level level = logLevel();
        return level==null 
        ? false
        : level.intValue() <= Level.FINER.intValue();
    }    
    
    private static boolean isLogFinest(){
        Level level = logLevel();
        return level==null 
        ? false
        : level.intValue() <= Level.FINEST.intValue();
    }    
    
    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    
    private static void logEntering(String method,Object ... params){
        logger.entering(TreeNodeDeepOffsetDef.class.getName(),method,params);
    }

    private static void logExiting(String method,Object result){
        logger.exiting(TreeNodeDeepOffsetDef.class.getName(),method,result);
    }

    private static void logExiting(String method){
        logger.exiting(TreeNodeDeepOffsetDef.class.getName(),method);
    }
    //</editor-fold>
    
    public int getRootOffsetOf( Node node ){
        return getRootOffsetOf( node, null );
    }
    
    private int getRootOffsetOf( Node node, Set visited ){
        if( node==null )throw new IllegalArgumentException("node==null");
        if( visited==null )visited = new LinkedHashSet();
        if( visited.contains(node) ){
            throw new Error( "cycle detect" );
        }
        visited.add(node);
        
        Node parent = (Node)node.getParent();
        if( parent==null )return 0;
        
        Node sib = (Node)node.getPreviousSibling();
        if( sib==null ){
            int off = getRootOffsetOf(parent,visited);
            off++;
            return off;
        }
        
        int off = getRootOffsetOf(sib,visited);
        int sibncnt = sib.getNodesCount();
        off += sibncnt;
        return off;
    }
    
    public Node deepOffset( Node node, int offset ) {
        /*if( node==null )throw new IllegalArgumentException("node==null");
        if( offset == 0 )return node;
        
        if( offset>0 ){
            return offsetRight(node, offset);
        }
        
        return offsetLeft(node, offset);*/
        
        return deepOffset(node, offset, null);
    }
    
    private Node deepOffset( Node node, int offset, Set visited ) {
        if( node==null )throw new IllegalArgumentException("node==null");

        if( visited==null )visited = new LinkedHashSet();
        if( visited.contains(node) )throw new Error( "cycle detect" );
        visited.add(node);

        if( offset == 0 )return node;
        
        if( offset>0 ){
            return offsetRight(node, offset, visited);
        }
        
        return offsetLeft(node, offset, visited);
    }
    
    private Node moveNext( Node node, Set visited ){
        if( node==null )throw new IllegalArgumentException("node==null");
        
        if( visited==null )visited = new LinkedHashSet();
//        if( visited.contains(node) )throw new Error( "cycle detect" );
        
        TreeNode[] children = node.getChildren();
        if( children!=null && children.length>0 ){
            Node n = (Node)children[0];
            return n;
        }
        
        TreeNode sib = node.getNextSibling();
        if( sib!=null )return (Node)sib;
        
        // move up to sib
        TreeNode p = node.getParent();
        while ( true ){
            if( p==null )break;
            
            sib = p.getNextSibling();
            if( sib!=null )return (Node)sib;
            
            p = p.getParent();
        }
        
        return null;
    }
    
    private Node movePrevious( Node node, Set visited ){
        if( node==null )throw new IllegalArgumentException("node==null");

        if( visited==null )visited = new LinkedHashSet();
        // if( visited.contains(node) )throw new Error( "cycle detect" );
        
        Node sib = (Node)node.getPreviousSibling();
        if( sib!=null ){
            int ncount = sib.getNodesCount();
            if( ncount<1 ) return null; // Error
            if( ncount==1 )return (Node)sib;
            return offsetRight((Node)sib, ncount-1, visited);
        }
        
        return (Node)node.getParent();
    }
    
    private Node offsetLeft( Node node, int offset, Set visited ){
        if( node==null )throw new IllegalArgumentException("node==null");

        if( visited==null )visited = new LinkedHashSet();
        // if( visited.contains(node) )throw new Error( "cycle detect" );
        visited.add(node);
        
        if( offset == 0 )return node;
        if( offset > 0 )return offsetRight(node, offset, visited);
        
        Node sib = (Node)node.getPreviousSibling();
        if( sib==null ){
            Node prev = movePrevious(node, visited);
            if( prev==null )return null;
            
            //if( visited.contains(prev) )throw new Error( "cycle detect" );
            return offsetLeft(prev, offset+1, visited);
        }
        
        int sibncount = sib.getNodesCount();
        int toff = offset + sibncount;
        
        if( (toff)==0 )return sib;        
        if( (toff)>0 ){
            // if( visited.contains(sib) )throw new Error( "cycle detect" );
            // visited.add(sib);
            return offsetRight(sib, toff, visited);
        }
        
        //if( visited.contains(sib) )throw new Error( "cycle detect" );
        return offsetLeft( sib, toff, visited );
    }
    
    private Node offsetRight( Node node, int offset, Set visited ){
        if( node==null )throw new IllegalArgumentException("node==null");

        if( visited==null )visited = new LinkedHashSet();
        //if( visited.contains(node) )throw new Error( "cycle detect" );
        
        if( offset == 0 )return node;
        if( offset < 0 )return null;
        
        int cnt = node.getNodesCount();
        
        TreeNode[] children = node.getChildren();
        if( children.length <= 0 && cnt > 1 ){
            logWarning("bag!!! cnt="+cnt+" children.length="+children.length);
            
            TreeNodeDeepCounterDef.setDeepCounterOf(node, null);
            cnt = node.getNodesCount();
        }
        
        if( cnt<1 )return null;
        
        //offset--;
        
        int minoff = 0;
        int maxoff = cnt - 1;
        
        if( offset < minoff )return null; // движение влево
        if( offset > maxoff ){
            if( offset < cnt ){ // не возможная ситуация
                throw new Error( "bag!!" );
            }

            // движение вправо
            Node nextNode = moveNext(node, visited);
            if( nextNode==null )return null;
            
            // return (Node)nextNode.deepOffset(offset-1);
            return (Node)deepOffset( (Node)nextNode, offset-1, visited);
        }
        
        // движение вниз
        
//        TreeNode[] children = node.getChildren();
        if( children==null )throw new Error( "bag!!!" );
        if( children.length<0 )throw new Error( "bag!!!!" );
        if( children.length<1 ){
            logWarning("!! offsetRight, children.length={0}", children.length);
        }
        
        int offsum = 0;
        
        TreeNode child = null;
        for( int ci=0; ci<children.length; ci++ ){
            child = children[ci];
            if( child==null )continue;

            if( !(child instanceof TreeNodeDeepOffset) )throw new Error( "bag!!!!!" );

            TreeNodeDeepOffset tndo = (TreeNodeDeepOffset)child;
            int ncount = tndo.getNodesCount();
            if( ncount<=0 )throw new Error( "bag!!!!!" );
            
            minoff = offsum+1;
            int mmxoff = minoff+ncount;
            
            if( !(offset >= minoff && offset < mmxoff) ){
                // вне диапазона мин/макс доч.смещения
                offsum += ncount;
                continue;
            }
            
            // return (Node)tndo.deepOffset(offset - offsum - 1);
            return (Node)deepOffset( (Node)tndo,offset - offsum - 1, visited );
        }
        
        return null;
    }
}
