/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.collection.tree;

import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Дефолт реализация
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 * @param <Node> Тип узла
 */
public class TreeNodeNotifierDef<Node extends TreeNode> {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(TreeNodeNotifierDef.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(TreeNodeNotifierDef.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(TreeNodeNotifierDef.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(TreeNodeNotifierDef.class.getName(), method, result);
    }
    //</editor-fold>
    
    protected final TreeNodeNotifier<Node> self;
    
    public TreeNodeNotifierDef(TreeNodeNotifier<Node> self){
        this.self = self;
    }

    /**
     * Передача события вверх по древу
     * @param ev Событие
     */
    public void popup( TreeNodeEvent ev ){
        if( ev instanceof TreeNodePopupEvent ){
            TreeNodePopupEvent tnpe
                = (TreeNodePopupEvent)ev;
            
            List l = tnpe.getPopupPath();
            
            // Проверка цикла
            if( l!=null ){
                int cycles = 0;
                for( Object o : l ){
                    if( self == o ){
                        // найден цикл
                        cycles++;
                    }
                }
                if( cycles>1 ){
                    throw new Error( "cycle detected" );
                    // return;
                }
            }
            
            l.add(self);
            
            self.fireTreeNodeEvent(ev);

            Object pnode = self.getParent();
            if( pnode instanceof TreeNodeNotifier ){
                ((TreeNodeNotifier)pnode).popup(ev);
            }
        }
    }
    
    public void onTreeNodeAdded(Integer index, Node child)
    {
        TreeNodeAdded ev = new TreeNodeAdded(self, child,index);
        ev.getPopupPath().add(child);

        popup( ev );
    }

    public void onTreeNodeAdding(Integer index, Node child)
    {
        TreeNodeAdding ev = new TreeNodeAdding(self, child,index);
        ev.getPopupPath().add(child);

        popup( ev );
    }
    
    public void onTreeNodeRemoved(Integer index, Node child)
    {
        TreeNodeRemoved ev = new TreeNodeRemoved(self, child,index);
        ev.getPopupPath().add(child);
        
        popup( ev );
    }
    
    public void onTreeNodeRemoving(Integer index, Node child)
    {
        TreeNodeRemoving ev = new TreeNodeRemoving(self, child,index);
        ev.getPopupPath().add(child);
        
        popup( ev );
    }
}
