/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.common;


import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;

//TODO требует тестирования
/**
 * Коллекция java.io.closeable, обеспечивает синхронизацию для публичных методов
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class CloseableSet
    implements Closeable
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(CloseableSet.class.getName()).log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        Logger.getLogger(CloseableSet.class.getName()).log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        Logger.getLogger(CloseableSet.class.getName()).log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        Logger.getLogger(CloseableSet.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(CloseableSet.class.getName()).log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        Logger.getLogger(CloseableSet.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(CloseableSet.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    protected Object sync = null;
    protected HashSet<Object> links = new LinkedHashSet<Object>();
    protected WeakHashMap<Object,Object> weaklinks = new WeakHashMap<Object,Object>();
    
    /**
     * Конструктор, по умолч синхронизация включена
     * По умолчанию синхронизация включена, объект по которому происходит синх - this.
     */
    @SuppressWarnings("LeakingThisInConstructor")
    public CloseableSet(){
        sync = this;
    }

    /**
     * Конструктор, по умолч синхронизация включена
     * По умолчанию синхронизация включена, объект по которому происходит синх - this.
     * @param closeables Объекты которые требуется закрывать
     */
    @SuppressWarnings("LeakingThisInConstructor")
    public CloseableSet( Closeable ... closeables ){
        sync = this;
        if( closeables!=null )
            for( Closeable c : closeables )
                if( c!=null )
                    add((Closeable)c);
    }
    
    /**
     * Конструктор.
     * @param sync использовать синхронизацию
     */
    @SuppressWarnings("LeakingThisInConstructor")
    public CloseableSet(boolean sync){
        if( sync ){
            this.sync = this;
        }
    }
    
    /**
     * Конструктор.
     * @param sync использовать синхронизацию
     * @param syncObject Объект по которому происходи синхронизация
     */
    public CloseableSet(boolean sync, Object syncObject){
        if( syncObject==null )throw new IllegalArgumentException( "syncObject==null" );
        if( sync ){
            this.sync = syncObject;
        }
    }
    
    //<editor-fold defaultstate="collapsed" desc="getCloseables()">
    /**
     * Получение массива объектов для закрытия
     * @return массив объектов
     */
    @SuppressWarnings({"LeakingThisInConstructor", "SynchronizeOnNonFinalField"})
    public Object[] getCloseables(){
        if( sync!=null ){
            synchronized(sync){
                ArrayList<Object> cl = new ArrayList<Object>();
                cl.addAll(links);
                cl.addAll(weaklinks.keySet());
                return cl.toArray();
            }
        }else{
            ArrayList<Object> cl = new ArrayList<Object>();
            cl.addAll(links);
            cl.addAll(weaklinks.keySet());
            return cl.toArray();
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="add()">
    /**
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @return интерфейс удяления из списка обрабатываемых
     */
    public Closeable add( Closeable closeable ){
        return add( (Closeable)closeable, false );
    }
    
    /**
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @return интерфейс удяления из списка обрабатываемых
     */
    public Closeable add( Runnable closeable ){
        return add( closeable, false );
    }
    
    /* java 7
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @return интерфейс удяления из списка обрабатываемых
    public Closeable add( AutoCloseable closeable ){
        return add( closeable, false );
    }*/
    
    /**
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @return интерфейс удяления из списка обрабатываемых
     */
    public Closeable add( Func0 closeable ){
        return add( closeable, false );
    }
    
    /**
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @param weak true - добавить как weak ссылку / false - как hard
     * @return интерфейс удяления из списка обрабатываемых
     */
    public Closeable add( Closeable closeable, boolean weak ){
        Object s = sync;
        if( s!=null ){
            synchronized (s) {
                return add0(closeable,weak);
            }
        }else{
            return add0(closeable,weak);
        }
    }
    
    /**
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @param weak true - добавить как weak ссылку / false - как hard
     * @return интерфейс удяления из списка обрабатываемых
     */
    public Closeable add( Runnable closeable, boolean weak ){
        Object s = sync;
        if( s!=null ){
            synchronized (s) {
                return add0(closeable,weak);
            }
        }else{
            return add0(closeable,weak);
        }
    }
    
    /* java 7
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @param weak true - добавить как weak ссылку / false - как hard
     * @return интерфейс удяления из списка обрабатываемых
    public Closeable add( AutoCloseable closeable, boolean weak ){
        Object s = sync;
        if( s!=null ){
            synchronized (s) {
                return add0(closeable,weak);
            }
        }else{
            return add0(closeable,weak);
        }
    }*/
    
    /**
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @param weak true - добавить как weak ссылку / false - как hard
     * @return интерфейс удяления из списка обрабатываемых
     */
    public Closeable add( Func0 closeable, boolean weak ){
        Object s = sync;
        if( s!=null ){
            synchronized (s) {
                return add0(closeable,weak);
            }
        }else{
            return add0(closeable,weak);
        }
    }
    
    private static final Closeable dummy = new Closeable() {
                @Override
                public void close() throws IOException {
                }
            };
    
    /**
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @param weak true - добавить как weak ссылку / false - как hard
     * @return интерфейс удяления из списка обрабатываемых
     */
    protected Closeable add0( Closeable closeable, boolean weak ){
        if( closeable==null ){ return dummy; }
        
        if( weak )weaklinks.put(closeable, true);else links.add(closeable);
        
        final Closeable cl = closeable;
        return new Closeable() {
            private Closeable c = cl;
            @Override
            public void close() throws IOException {
                if (c != null) {
                    CloseableSet.this.remove(c);
                    c = null;
                }
            }
        };
    }
    
    /* java 7
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @param weak true - добавить как weak ссылку / false - как hard
     * @return интерфейс удяления из списка обрабатываемых
    protected Closeable add0( AutoCloseable closeable, boolean weak ){
        if( closeable==null ){ return dummy; }
        
        if( weak )weaklinks.put(closeable, true);else links.add(closeable);
        
        final AutoCloseable cl = closeable;
        return new Closeable() {
            private AutoCloseable c = cl;
            @Override
            public void close() throws IOException {
                if (c != null) {
                    CloseableSet.this.remove(c);
                    c = null;
                }
            }
        };
    } */
    
    /**
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @param weak true - добавить как weak ссылку / false - как hard
     * @return интерфейс удяления из списка обрабатываемых
     */
    protected Closeable add0( Runnable closeable, boolean weak ){
        if( closeable==null ){ return dummy; }
        
        if( weak )weaklinks.put(closeable, true);else links.add(closeable);
        
        final Runnable cl = closeable;
        return new Closeable() {
            private Runnable c = cl;
            @Override
            public void close() throws IOException {
                if (c != null) {
                    CloseableSet.this.remove(c);
                    c = null;
                }
            }
        };
    }
    
    /**
     * Добавляет объект в список обрабатываемых
     * @param closeable объект
     * @param weak true - добавить как weak ссылку / false - как hard
     * @return интерфейс удяления из списка обрабатываемых
     */
    protected Closeable add0( Func0 closeable, boolean weak ){
        if( closeable==null ){ return dummy; }
        
        if( weak )weaklinks.put(closeable, true);else links.add(closeable);
        
        final Func0 cl = closeable;
        return new Closeable() {
            private Func0 c = cl;
            @Override
            public void close() throws IOException {
                if (c != null) {
                    CloseableSet.this.remove(c);
                    c = null;
                }
            }
        };
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="remove()">
    /**
     * Удалить объект из списка обрабатываемых
     * @param closeable объект
     */
    public void remove( Object closeable ){
        Object s = sync;
        if( s!=null ){
            synchronized (s) {
                remove0(closeable);
            }
        }else{
            remove0(closeable);
        }
    }
    
    /**
     * Удалить объект из списка обрабатываемых
     * @param closeable объект
     */
    protected void remove0( Object closeable ){
        if (closeable != null) {
            links.remove(closeable);
            weaklinks.remove(closeable);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="removeAll()">
    /**
     * Удалить все объекты из списка обрабатываемых
     */
    public void removeAll(){
        Object s = sync;
        if( s!=null ){
            synchronized (s) {
                removeAll0();
            }
        }else{
            removeAll0();
        }
    }
    
    /**
     * Удалить все объекты из списка обрабатываемых
     */
    public void removeAll0(){
        links.clear();
        weaklinks.clear();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="closeAll()">
    /**
     * Вызвать для всех объектов close и удалить из списка обрабатываемых
     */
    public void closeAll(){
        closeAll(true);
    }
    
    /**
     * Вызвать для всех объектов close
     * @param removeAll удалять из списка обрабатываемых
     */
    public void closeAll(boolean removeAll){
        Object s = sync;
        if( s!=null ){
            synchronized (s) {
                closeAll0(removeAll);
            }
        }else{
            closeAll0(removeAll);
        }
    }
    
    /**
     * Вызвать для всех объектов close
     * @param removeAll удалять из списка обрабатываемых
     */
    protected void closeAll0(boolean removeAll){
        for( Object c : getCloseables() ){
            if( c instanceof Closeable){
                try {
                    ((Closeable)c).close();
                    if( removeAll )remove(c);
                } catch (IOException ex) {
                    Logger.getLogger(CloseableSet.class.getName()).log(Level.SEVERE, null, ex);
                }
            }else if( c instanceof Runnable ){
                ((Runnable)c).run();
            }else if( c instanceof Func0 ){
                ((Func0)c).apply();
            /*}else if( c instanceof AutoCloseable ){
                try {
                    ((AutoCloseable)c).close();
                } catch (Exception ex) {
                    Logger.getLogger(CloseableSet.class.getName()).log(Level.SEVERE, null, ex);
                }*/
            }
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="close()">
    /**
     * Закрывает объекты
     * @throws IOException Ошибка IO
     * @see #closeAll()
     */
    @Override
    public void close() throws IOException {
        Object s = sync;
        if( s!=null ){
            synchronized (s) {
                closeAll();
            }
        }else{
            closeAll();
        }
    }
//</editor-fold>
}
