/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.fs;

import java.nio.charset.Charset;
import xyz.cofe.collection.Func2;

/**
 * Абстрактный файл
 */
public abstract class AbstractFile
implements
    File,
    FileAppendBytes,
    FileAppendText,
    FileWriteBytes,
    FileWriteText,
    FileReadText,
    FileReadBytes
{
    /**
     * Создание клона объекта
     * @return клон
     */
    @Override
    public abstract File clone();
    
    private FileHelper fileHelper;

    /**
     * Помошник для работы с файлом
     * @return Помошник
     */
    protected synchronized FileHelper getFileHelper(){
        if( fileHelper!=null ){
            return fileHelper;
        }
        fileHelper = new FileHelper(this);
        return fileHelper;
    }

    /**
     * Чтение текстового содержимого файла
     * @param cs Кодировка (возможно null, тогда будет использоваться кодировка по умолчанию)
     * @param maxSize Максимальное кол-во байт или -1
     * @param blockSize Размер блока читаемого за раз или -1
     * @param progress Функция уведомления чтения или null. <br>
     * Первый аргумент - прочитаное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть прочтено <br>
     * @return Текстовое содержимое
     */
    @Override
    public String readText(Charset cs, int maxSize, int blockSize, Func2<Object, Long, Long> progress) {
        return getFileHelper().readText(cs, maxSize, blockSize, progress);
    }

    /**
     * Чтение текстового содержимого файла
     * @param cs Кодировка (возможно null, тогда будет использоваться кодировка по умолчанию)
     * @param maxSize Максимальное кол-во байт или -1
     * @return Текстовое содержимое
     */
    @Override
    public String readText(Charset cs, int maxSize) {
        return getFileHelper().readText(cs, maxSize);
    }

    /**
     * Чтение текстового содержимого файла
     * @param cs Кодировка (возможно null, тогда будет использоваться кодировка по умолчанию)
     * @return Текстовое содержимое
     */
    @Override
    public String readText(Charset cs) {
        return getFileHelper().readText(cs);
    }

    /**
     * Запись текста в файл
     * @param text Текст
     * @param cs Кодировка
     * @param blockSize Размер блока или -1
     * @param progress Функция уведомления записи или null. <br>
     * Первый аргумент - записанное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть записано
     */
    @Override
    public void writeText(String text, Charset cs, int blockSize, Func2<Object, Long, Long> progress) {
        getFileHelper().writeText(text, cs, blockSize, progress);
    }

    /**
     * Запись текста в файл
     * @param text Текст
     * @param cs Кодировка
     */
    @Override
    public void writeText(String text, Charset cs) {
        getFileHelper().writeText(text, cs);
    }

    /**
     * Запись текста в конец файла
     * @param text Текст
     * @param cs Кодировка
     * @param blockSize Размер блока или -1
     * @param progress Функция уведомления записи или null. <br>
     * Первый аргумент - записанное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть записано
     */
    @Override
    public void appendText(String text, Charset cs, int blockSize, Func2<Object, Long, Long> progress) {
        getFileHelper().appendText(text, cs, blockSize, progress);
    }

    /**
     * Запись текста в конец файла
     * @param text Текст
     * @param cs Кодировка
     */
    @Override
    public void appendText(String text, Charset cs) {
        getFileHelper().appendText(text, cs);
    }

    /**
     * Чтение байтового содержимого файла
     * @param maxSize Максимальное кол-во байт или -1
     * @param blockSize Размер блока читаемого за раз или -1
     * @param progress Функция уведомления чтения или null. <br>
     * Первый аргумент - прочитаное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть прочтено <br>
     * @return Текстовое содержимое
     */
    @Override
    public byte[] readBytes(int maxSize, int blockSize, Func2<Object, Long, Long> progress) {
        return getFileHelper().readBytes(maxSize, blockSize, progress);
    }

    /**
     * Чтение байтового содержимого файла
     * @param maxSize Максимальное кол-во байт или -1
     * @return Текстовое содержимое
     */
    @Override
    public byte[] readBytes(int maxSize) {
        return getFileHelper().readBytes(maxSize);
    }

    /**
     * Чтение байтового содержимого файла
     * @return Текстовое содержимое
     */
    @Override
    public byte[] readBytes() {
        return getFileHelper().readBytes();
    }

    /**
     * Запись данных
     * @param data Данные кторые должны быть записаны
     * @param offset Смещение
     * @param dataSize Кол-во записываемых данных
     * @param blockSize Размер блока записываемых данных за раз или -1
     * @param progress Функция уведомления записи или null. <br>
     * Первый аргумент - записанное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть записано
     */
    @Override
    public void writeBytes(byte[] data, int offset, int dataSize, int blockSize, Func2<Object, Long, Long> progress) {
        getFileHelper().writeBytes(data, offset, dataSize, blockSize, progress);
    }

    /**
     * Запись данных
     * @param data Данные кторые должны быть записаны
     * @param offset Смещение
     * @param dataSize Кол-во записываемых данных
     */
    @Override
    public void writeBytes(byte[] data, int offset, int dataSize) {
        getFileHelper().writeBytes(data, offset, dataSize);
    }

    /**
     * Запись данных
     * @param data Данные кторые должны быть записаны
     */
    @Override
    public void writeBytes(byte[] data) {
        getFileHelper().writeBytes(data);
    }

    /**
     * Запись данных в конец файла
     * @param data Данные кторые должны быть записаны
     * @param offset Смещение
     * @param dataSize Кол-во записываемых данных
     * @param blockSize Размер блока записываемых данных за раз или -1
     * @param progress Функция уведомления записи или null. <br>
     * Первый аргумент - записанное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть записано
     */
    @Override
    public void appendBytes(byte[] data, int offset, int dataSize, int blockSize, Func2<Object, Long, Long> progress) {
        getFileHelper().appendBytes(data, offset, dataSize, blockSize, progress);
    }

    /**
     * Запись данных в конец файла
     * @param data Данные кторые должны быть записаны
     * @param offset Смещение
     * @param dataSize Кол-во записываемых данных
     */
    @Override
    public void appendBytes(byte[] data, int offset, int dataSize) {
        getFileHelper().appendBytes(data, offset, dataSize);
    }

    /**
     * Запись данных в конец файла
     * @param data Данные кторые должны быть записаны
     */
    @Override
    public void appendBytes(byte[] data) {
        getFileHelper().appendBytes(data);
    }
}
