/*
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.fs;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import xyz.cofe.common.Reciver;

/**
 * Файл/каталог файловой системы
 * @author nt.gocha@gmail.com
 */
public interface File extends Path,
    FileReadBytes, FileReadText,
    FileWriteBytes, FileWriteText,
    FileAppendBytes, FileAppendText
{
    /**
     * Создание клона объекта
     * @return клон
     */
    public File clone();
    
    /**
     * Возвращает ссылку на файловую систему
     * @return ссылка на файловую систему
     */
    public FileSystem getFileSystem();

    /**
     * Возвращает путь соответствующий предку файла/каталога
     * @return Предок или null
     */
    @Override
    public File getParent();

    /**
     * Возвращает дочерний путь файла / каталога
     * @param name Имя файла/каталога
     * @return Дочерний элемент
     */
    @Override
    public File getChild( String name );

    /**
     * Возвращает каноническую форму
     * @return Каноническая форма
     */
    @Override
    public File getCanonical();

    /**
     * Возвращает абсолютный путь
     * @return Абсолютный путь
     */
    @Override
    public File getAbsolute();

    /**
     * Возвращает true если файл является каталогом
     * @return true - каталог
     */
    public boolean isDirectory();

    /**
     * Возвращает true если файл является обычным файлом
     * @return true - файл
     */
    public boolean isFile();

    /**
     * Возвращает true если файл существует
     * @return true - существует
     */
    public boolean isExists();

    /**
     * Возвращает список дочерних файлов
     * @return дочерние файлы
     */
    public List<File> listFiles();

    /**
     * Возвращает дату модификации файла
     * @return дата модификации файла
     */
    public Date getModifyDate();

    /**
     * Устанавливает дату модификации файла
     * @param modifyDate дата модификации файла
     */
    public void setModifyDate( Date modifyDate );

    /**
     * Возвращает длину файла
     * @return длина файла
     */
    public long getLength();

    /**
     * Открывает файл для чтения
     * @return Поток байтов для чтения
     */
    public BufferedInputStream openRead();

    /**
     * Открывает файл для записи
     * @return Поток для записи
     */
    public BufferedOutputStream openWrite();

    /**
     * Открывает для дозаписи в конец файла
     * @return Поток для записи
     */
    public BufferedOutputStream openAppend();

    /**
     * Открытие файла для чтения/запичи
     * @param mode Режим открытия
     * @return Случайный доступ к файлу
     */
    public java.io.RandomAccessFile openReadWrite(OpenMode mode);

    /**
     * Возвращает последовательность файлов/каталогов для обхода содержимого
     * @return последовательность файлов/каталогов
     */
    public Iterable<File> walk();

    /**
     * Возвращает последовательность файлов/каталогов для обхода содержимого
     * @param opts Опции обхода каталога
     * @return последовательность файлов/каталогов
     */
    public Iterable<File> walk(VisitOptions opts);

    /**
     * Проверяет возможно ли читать файл
     * @return true - чтение разрешено
     */
    public boolean isReadable();

    /**
     * Устанавливает права чтения
     * @param readable true - чтение доступно
     */
    public void setReadable( boolean readable );

    /**
     * Устанавливает права чтения
     * @param readable true - чтение доступно
     * @param ownerOnly true - только для владельца
     */
    public void setReadable(boolean readable, boolean ownerOnly);

    /**
     * Проверяет права записи
     * @return true - запись разрешена
     */
    public boolean isWritable();

    /**
     * Устанавливает права записи
     * @param writable true - запись разрешена
     */
    public void setWritable( boolean writable );

    /**
     * Устанавливает права записи
     * @param writable true - запись разрешена
     * @param ownerOnly true - только для владельца
     */
    public void setWritable(boolean writable, boolean ownerOnly);

    /**
     * Проверяет права исполнения
     * @return true - исполнение разрешено
     */
    public boolean isExecutable();

    /**
     * Устанавливает права исполнения
     * @param executable true - исполнение разрешено
     */
    public void setExecutable( boolean executable );

    /**
     * Устанавливает права исполнения
     * @param executable true - исполнение разрешено
     * @param ownerOnly true - только для владельца
     */
    public void setExecutable(boolean executable, boolean ownerOnly);

    /**
     * Удаляет файл/каталог с содержимым
     */
    public void delete();

    /**
     * Удаляет файл/каталог с содержимым
     * @param deleted удаленные файлы/каталоги
     */
    public void delete(Reciver<File> deleted);

    /**
     * Удаляет файл/каталог с содержимым
     * @param deleted удаленные файлы/каталоги
     * @param opts опции обхода каталога
     */
    public void delete(Reciver<File> deleted,VisitOptions opts);

    /**
     * Создает каталог
     */
    public void mkdir();

    /**
     * Создает каталог и недостающие родит. каталоги
     */
    public void mkdirs();

    /**
     * Переименовывает файл/каталог
     * @param target новое имя
     */
    public void renameTo( File target );
}
