/*
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.fs;

import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.Func2;
import xyz.cofe.io.IOFun;

/**
 * Функции помошники при работе с файлами
 */
public class FileHelper {
    /**
     * Ссылка на файл
     */
    public final File file;

    /**
     * Конструктор
     * @param file файл
     */
    public FileHelper( File file ){
        if( file==null )throw new IllegalArgumentException( "file==null" );
        this.file = file;
    }

    //<editor-fold defaultstate="collapsed" desc="readText()">
    /**
     * Чтение текстового содержимого файла
     * @param cs Кодировка (возможно null, тогда будет использоваться кодировка по умолчанию)
     * @param maxSize Максимальное кол-во байт или -1
     * @param blockSize Размер блока читаемого за раз или -1
     * @param progress Функция уведомления чтения или null. <br>
     * Первый аргумент - прочитаное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть прочтено <br>
     * @return Текстовое содержимое
     */
    public String readText(
        Charset cs,
        int maxSize,
        int blockSize,
        Func2<Object,Long,Long> progress
    )
    {
        try {
            final Func2<Object,Long,Long> fprogress = progress;
            final int max = maxSize;

            InputStream input = file.openRead();

            String txt = IOFun.readText(input, cs, maxSize, blockSize, new Func1<Object, Long>() {
                @Override
                public Object apply(Long readed) {
                    try{
                        if( fprogress==null )return null;
                        long tot = max >= 0 ? max : file.getLength();
                        fprogress.apply(readed, tot);
                    }catch(Throwable err ){
                        Logger.getLogger(FileHelper.class.getName())
                                .log(Level.WARNING, err.getMessage(), err);
                    }
                    return null;
                }
            });

            input.close();

            return txt;
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    /**
     * Чтение текстового содержимого файла
     * @param cs Кодировка (возможно null, тогда будет использоваться кодировка по умолчанию)
     * @param maxSize Максимальное кол-во байт или -1
     * @return Текстовое содержимое
     */
    public String readText( Charset cs, int maxSize ) {
        return readText(cs, maxSize, -1, null);
    }

    /**
     * Чтение текстового содержимого файла
     * @param cs Кодировка (возможно null, тогда будет использоваться кодировка по умолчанию)
     * @return Текстовое содержимое
     */
    public String readText( Charset cs ) {
        return readText(cs, -1, -1, null);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="writeText()">
    /**
     * Запись текста в файл
     * @param text Текст
     * @param cs Кодировка
     * @param blockSize Размер блока или -1
     * @param progress Функция уведомления записи или null. <br>
     * Первый аргумент - записанное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть записано
     */
    public void writeText( String text, Charset cs, int blockSize, Func2<Object,Long,Long> progress )
    {
        try {
            if( text==null )throw new IllegalArgumentException( "text==null" );
            OutputStream out = file.openWrite();
            IOFun.writeText(out, text, cs, blockSize, progress);
            out.flush();
            out.close();
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    /**
     * Запись текста в файл
     * @param text Текст
     * @param cs Кодировка
     */
    public void writeText( String text, Charset cs )
    {
        writeText(text, cs, -1, null);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="appendText()">
    /**
     * Запись текста в конец файла
     * @param text Текст
     * @param cs Кодировка
     * @param blockSize Размер блока или -1
     * @param progress Функция уведомления записи или null. <br>
     * Первый аргумент - записанное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть записано
     */
    public void appendText( String text, Charset cs, int blockSize, Func2<Object,Long,Long> progress )
    {
        try {
            if( text==null )throw new IllegalArgumentException( "text==null" );
            OutputStream out = file.openAppend();
            IOFun.writeText(out, text, cs, blockSize, progress);
            out.flush();
            out.close();
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    /**
     * Запись текста в конец файла
     * @param text Текст
     * @param cs Кодировка
     */
    public void appendText( String text, Charset cs )
    {
        appendText(text, cs, -1, null);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="readBytes()">
    /**
     * Чтение байтового содержимого файла
     * @param maxSize Максимальное кол-во байт или -1
     * @param blockSize Размер блока читаемого за раз или -1
     * @param progress Функция уведомления чтения или null. <br>
     * Первый аргумент - прочитаное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть прочтено <br>
     * @return Текстовое содержимое
     */
    public byte[] readBytes(
        int maxSize,
        int blockSize,
        Func2<Object,Long,Long> progress
    )
    {
        try {
            final Func2<Object,Long,Long> fprogress = progress;
            final int max = maxSize;

            InputStream input = file.openRead();

            byte[] data = IOFun.readBytes(input, maxSize, blockSize, new Func1<Object, Long>() {
                @Override
                public Object apply(Long readed) {
                    try{
                        if( fprogress==null )return null;

                        long tot = max >= 0 ? max : file.getLength();
                        fprogress.apply(readed, tot);
                    }catch(Throwable err ){
                        Logger.getLogger(FileHelper.class.getName())
                                .log(Level.WARNING, err.getMessage(), err);
                    }
                    return null;
                }
            });

            input.close();

            return data;
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    /**
     * Чтение байтового содержимого файла
     * @param maxSize Максимальное кол-во байт или -1
     * @return Текстовое содержимое
     */
    public byte[] readBytes(
        int maxSize
    )
    {
        return readBytes(maxSize, -1, null);
    }

    /**
     * Чтение байтового содержимого файла
     * @return Текстовое содержимое
     */
    public byte[] readBytes()
    {
        return readBytes(-1, -1, null);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="writeBytes()">
    /**
     * Запись данных
     * @param data Данные кторые должны быть записаны
     * @param offset Смещение
     * @param dataSize Кол-во записываемых данных
     * @param blockSize Размер блока записываемых данных за раз или -1
     * @param progress Функция уведомления записи или null. <br>
     * Первый аргумент - записанное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть записано
     */
    public void writeBytes(
        byte[] data,
        int offset,
        int dataSize,
        int blockSize,
        Func2<Object,Long,Long> progress
    )
    {
        try {
            if( data==null )throw new IllegalArgumentException( "data==null" );
            OutputStream out = file.openWrite();
            IOFun.writeBytes(out, data, offset, dataSize, blockSize, progress);
            out.flush();
            out.close();
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    /**
     * Запись данных
     * @param data Данные кторые должны быть записаны
     * @param offset Смещение
     * @param dataSize Кол-во записываемых данных
     */
    public void writeBytes(
        byte[] data,
        int offset,
        int dataSize
    )
    {
        writeBytes(data, offset, dataSize, -1, null);
    }

    /**
     * Запись данных
     * @param data Данные кторые должны быть записаны
     */
    public void writeBytes(
        byte[] data
    )
    {
        if( data==null )throw new IllegalArgumentException( "data==null" );
        writeBytes(data, 0, data.length, -1, null);
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="appendBytes()">
    /**
     * Запись данных в конец файла
     * @param data Данные кторые должны быть записаны
     * @param offset Смещение
     * @param dataSize Кол-во записываемых данных
     * @param blockSize Размер блока записываемых данных за раз или -1
     * @param progress Функция уведомления записи или null. <br>
     * Первый аргумент - записанное кол-во байт <br>
     * Второй аргумент - всего кол-во байт, которое должно быть записано
     */
    public void appendBytes(
        byte[] data,
        int offset,
        int dataSize,
        int blockSize,
        Func2<Object,Long,Long> progress
    )
    {
        try {
            if( data==null )throw new IllegalArgumentException( "data==null" );
            OutputStream out = file.openAppend();
            IOFun.writeBytes(out, data, offset, dataSize, blockSize, progress);
            out.flush();
            out.close();
        } catch( IOException ex ) {
            throw new IOError(ex);
        }
    }

    /**
     * Запись данных в конец файла
     * @param data Данные кторые должны быть записаны
     * @param offset Смещение
     * @param dataSize Кол-во записываемых данных
     */
    public void appendBytes(
        byte[] data,
        int offset,
        int dataSize
    ) {
        if( data==null )throw new IllegalArgumentException( "data==null" );
        appendBytes(data, offset, dataSize, -1, null);
    }

    /**
     * Запись данных в конец файла
     * @param data Данные кторые должны быть записаны
     */
    public void appendBytes(
        byte[] data
    )
    {
        if( data==null )throw new IllegalArgumentException( "data==null" );
        appendBytes(data, 0, data.length, -1, null);
    }
//</editor-fold>
}
