/*
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.fs;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.Stack;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Convertor;
import xyz.cofe.collection.Iterators;
import xyz.cofe.collection.NodesExtracter;
import xyz.cofe.collection.Predicate;
import xyz.cofe.collection.iterators.TreeWalk;
import xyz.cofe.collection.iterators.TreeWalkItreator;

/**
 * Осуществляет обход каталога
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class FileVisitor{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(FileVisitor.class.getName()).log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        Logger.getLogger(FileVisitor.class.getName()).log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        Logger.getLogger(FileVisitor.class.getName()).log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        Logger.getLogger(FileVisitor.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(FileVisitor.class.getName()).log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        Logger.getLogger(FileVisitor.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(FileVisitor.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    /**
     * Конструктор по умолчанию
     */
    public FileVisitor(){
    }
    
    /**
     * Поведение при возниконовеии ошибки
     */
    public static enum ErrorBehavior {
        /**
         * Продолжить выполнение
         */
        Continue,
        
        /**
         * Завершить работу
         */
        Stop
    }
    
    /**
     * Опции копирования
     */
    public static class CopyOptions {
        public boolean checkHistory = true;
        public boolean history = false;
        public boolean followLink = true;
        public boolean resolveLink = true;
        public boolean maxRecursiveLinkLevel = true;
        public boolean useCanonicalPath = true;
        public boolean useAbsolutePath = true;
        public boolean errorBehavior = true;
        public boolean filter = true;
    }
    
    /**
     * Конструктор копирования
     * @param src образец
     * @param opts Опции копирования
     */
    public FileVisitor( FileVisitor src, CopyOptions opts ){
        if( src!=null && opts!=null ){
            if( opts.checkHistory )checkHistory = src.isCheckHistory();
            if( opts.history ){
                visitHistory.clear();
                visitHistory.addAll(src.getVisitHistory());
            }
            if( opts.resolveLink ){
                this.resolveLink = src.isResolveLink();
            }
            if( opts.followLink ){
                this.followLink = src.isFollowLink();
            }
            if( opts.maxRecursiveLinkLevel ){
                this.maxRecusiveLinkLevel = src.getMaxRecusiveLinkLevel();
            }
            if( opts.useCanonicalPath ){
                this.useCanonicalPath = src.isUseCanonicalPath();
            }
            if( opts.useAbsolutePath ){
                this.useAbsolutePath = src.isUseAbsolutePath();
            }
            if( opts.filter ){
                this.filter = src.filter;
            }
        }
    }
    
    /**
     * Клонирует объект
     * @return копия объекта
     */
    @Override
    public FileVisitor clone(){
        CopyOptions opts = new CopyOptions();
        return clone(opts);
    }
    
    /**
     * Клонирует объект
     * @param opts Опции копирования
     * @return копия объекта
     */
    public FileVisitor clone(CopyOptions opts){
        return new FileVisitor( this, opts );
    }
    
    /**
     * Вызывается при входе в каталог/файл
     * @param path Путь/файл в который осуществляется вход
     */
    public void enter(Stack<File> path){
    }
    
    /**
     * Вызывается при выходе из каталога/файла
     * @param path Путь/файл из которого осуществляется выход
     */
    public void exit(Stack<File> path){
    }
    
    /**
     * Вызывается при возниконовении ошибки
     * @param err Ошибка
     * @return true - продолжить выполнение, false - завершить работу
     */
    public ErrorBehavior error(Throwable err){
        System.err.println("error "+err.getMessage());
        return errorBehavior;
    }
    
    //<editor-fold defaultstate="collapsed" desc="visitHistory">
    private Set<File> visitHistory = new TreeSet<File>();
    
    /**
     * Возвращает набор посещенных файлов/каталогов
     * @return Список файлов
     */
    public Set<File> getVisitHistory() {
        if( visitHistory==null )visitHistory = new TreeSet<File>();
        return visitHistory;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="errorBehavior">
    private ErrorBehavior errorBehavior = ErrorBehavior.Continue;
    
    /**
     * Поведение при возникновении ошибки
     * @return поведение
     */
    public ErrorBehavior getErrorBehavior() {
        return errorBehavior;
    }
    
    /**
     * Поведение при возникновении ошибки
     * @param errorBehavior поведение
     */
    public void setErrorBehavior(ErrorBehavior errorBehavior) {
        this.errorBehavior = errorBehavior;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="checkHistory">
    private boolean checkHistory = true;
    
    /**
     * Проверять историю посещенных каталог/файлов, что бы избежать бесконечных циклов.
     * <b>По умолчанию true.</b>
     * @return Проверять историю посещенных каталог/файлов
     */
    public boolean isCheckHistory() {
        return checkHistory;
    }
    
    /**
     * Проверять историю посещенных каталог/файлов, что бы избежать бесконечных циклов.
     * @param checkHistory Проверять историю посещенных каталог/файлов
     */
    public void setCheckHistory(boolean checkHistory) {
        this.checkHistory = checkHistory;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="followLink">
    private boolean followLink = true;

    /**
     * Следовать символическим ссылкам. <b>По умолчанию true.</b> <br>
     * Если true - то, с. ссылки будут испоьзоватся в качестве ссылок на каталоги
     * @return Следовать символическим ссылкам.
     */
    public boolean isFollowLink() {
        return followLink;
    }
    
    /**
     * Следовать символическим ссылкам.
     * @param followLink Следовать символическим ссылкам.
     */
    public void setFollowLink(boolean followLink) {
        this.followLink = followLink;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="resolveLink">
    private boolean resolveLink = false;
    
    /**
     * "Разрешать" символические ссылки. <b>По умолчанию false.</b><br>
     * Если установлено в true - то, при нахождении символической ссылки будет установлено куда она указывает
     * и использовано разрешенное имя.<br>
     * <br>
     * Пример:<br>
     * Есть следующая структура файлов<br>
     * <br>
     * /A/<br>
     * /A/1<br>
     * /A/link -&gt; /B<br>
     * /B/<br>
     * /B/2<br>
     * /B/3<br>
     * <br>
     * <table border="1" summary="Пример разрешения ссылок">
     * <tr>
     * <td>resolveLink = false</td><td>resolveLink = true</td>
     * </tr>
     * <tr>
     * <td>
     * /A/<br>
     * /A/1<br>
     * /A/link/<br>
     * /A/link/2<br>
     * /A/link/3<br>
     * /B/<br>
     * /B/2<br>
     * /B/3<br>
     * </td>
     * <td>
     * /A/<br>
     * /A/1<br>
     * /B/<br>
     * /B/2<br>
     * /B/3<br>
     * </td>
     * </tr>
     * </table>
     * @return "Разрешать" символические ссылки.
     */
    public boolean isResolveLink() {
        return resolveLink;
    }
    
    /**
     * "Разрешать" символические ссылки.
     * @param resolveLink "Разрешать" символические ссылки.
     * @see #isResolveLink() 
     */
    public void setResolveLink(boolean resolveLink) {
        this.resolveLink = resolveLink;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="maxRecusiveLinkLevel">
    private int maxRecusiveLinkLevel = 100;
    
    /**
     * Максимальная глубина вложенности символических ссылок. <br>
     * -1 - не органичено.
     * <b>По умолчанию 100</b>
     * @return Максимальная глубина вложенности символических ссылок
     */
    public int getMaxRecusiveLinkLevel() {
        return maxRecusiveLinkLevel;
    }
    
    /**
     * Максимальная глубина вложенности символических ссылок. <br>
     * @param maxRecusiveLinkLevel Максимальная глубина вложенности символических ссылок. <br> -1 - не органичено.
     */
    public void setMaxRecusiveLinkLevel(int maxRecusiveLinkLevel) {
        this.maxRecusiveLinkLevel = maxRecusiveLinkLevel;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="useCanonicalPath">
    private boolean useCanonicalPath = true;
    
    /**
     * Приводить пути файлов к канонической форме, <b>по умолч. true.</b>
     * @return Приводить пути файлов к канонической форме
     */
    public boolean isUseCanonicalPath() {
        return useCanonicalPath;
    }
    
    /**
     * Приводить пути файлов к канонической форме
     * @param useCanonicalPath Приводить пути файлов к канонической форме
     */
    public void setUseCanonicalPath(boolean useCanonicalPath) {
        this.useCanonicalPath = useCanonicalPath;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="useAbsolutePath">
    private boolean useAbsolutePath = false;
    
    /**
     * Приводить пути файлов к абсолютной форме, <b>по умолч. false.</b>
     * @return Приводить пути файлов к абсолютной форме
     */
    public boolean isUseAbsolutePath() {
        return useCanonicalPath;
    }
    
    /**
     * Приводить пути файлов к абсолютной форме
     * @param useCanonicalPath Приводить пути файлов к абсолютной форме
     */
    public void setUseAbsolutePath(boolean useCanonicalPath) {
        this.useCanonicalPath = useCanonicalPath;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="resolveLink">
    /**
     * Рекурсивно разрешает ссылки
     * @param rl Получение ссылки
     * @param currentResolve текущая итерация
     * @param maxResolve макс. кол-во итераций или -1
     * @return Ссылка или null
     */
    protected File resolveLink( ResolveLink<File> rl, int currentResolve, int maxResolve ){
        if( currentResolve>maxResolve && maxResolve>=0 )return null;
        File f = rl.resolveLink();
        if( f==null )return null;
        if( (f instanceof IsSymbolicLink) && (f instanceof ResolveLink) ){
            ResolveLink<File> frl = (ResolveLink)f;
            IsSymbolicLink fis = (IsSymbolicLink)f;
            if( fis.isSymbolicLink() ){
                File rf = resolveLink(frl,currentResolve+1,maxResolve);
                if( rf==null ){
                    return f;
                }
                return rf;
            }
        }
        return f;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="filter">
    protected Predicate<Stack<File>> filter = null;
    
    /**
     * Вызывается перед входом в каталог для проверки необходимости войти
     * @return Предикат входа в каталог
     */
    public Predicate<Stack<File>> getFilter() {
        return filter;
    }
    
    /**
     * Вызывается перед входом в каталог для проверки необходимости войти
     * @param filter Предикат входа в каталог
     */
    public void setFilter(Predicate<Stack<File>> filter) {
        this.filter = filter;
    }
    //</editor-fold>
    
    private List<File> getChildrenOf_post( List<File> children ){
        for( int i=0; i<children.size(); i++ ){
            File c = children.get(i);
            if( isUseAbsolutePath() ){
                c = (File)c.getAbsolute();
            }
            if( isUseCanonicalPath() ){
                c = (File)c.getCanonical();
            }
            children.set(i, c);
        }
        return children;
    }
    
    /**
     * Возвращает список дочерних файлов, для последующего обхода
     * @param parent Каталог/файл в которых осуществляется вход
     * @return дочерние файлы/каталоги
     */
    private Iterable<File> getChildrenOf( File parent){
        ArrayList<File> res = new ArrayList<File>();
        if( parent==null )return res;
        
        if( isCheckHistory() ){
            Set<File> hist = getVisitHistory();
            if( hist.contains(parent) )return res;
            hist.add(parent);
        }
        
        if( parent instanceof IsSymbolicLink ){
            boolean islink = ((IsSymbolicLink)parent).isSymbolicLink();
            if( islink ){
                if( !isFollowLink() ){
                    return res;
                }                
                if( isResolveLink() && (parent instanceof ResolveLink) ){
                    File resolved = resolveLink((ResolveLink<File>)parent,1,getMaxRecusiveLinkLevel());
                    if( resolved!=null ){
                        if( resolved.isDirectory() ){
                            List files = resolved.listFiles();
                            return getChildrenOf_post(files);
                        }else{
//                            res.add( isUseCanonicalPath() ? 
//                                (T)resolved.getCanonical() : 
//                                resolved );
                            return getChildrenOf_post(res);
                        }
                    }
                }
            }
        }
        
        if( parent.isDirectory() ){
            List<File> files = parent.listFiles();
            for( File f : files ){
                res.add( f );
            }
        }
        
        return getChildrenOf_post(res);
    }
    
    private boolean walk_isRunning = false;
    
    /**
     * Осуществляет обход каталога,
     * вызывает методы enter/exit
     * @param root точка с которой начиается обход
     * @see #enter(java.util.Stack) 
     * @see #exit(java.util.Stack) 
     */
    public void walk( File root ){
        if( root==null )throw new IllegalArgumentException( "root==null" );
        if( walk_isRunning )throw new IllegalStateException("walk is running");
        try{
            walk_isRunning = true;
            getVisitHistory().clear();
            
            Stack<File> s = new Stack<File>();
            s.push( isUseCanonicalPath() ? (File)root.getCanonical() : root);
            
            walk(s);
            
            s.pop();
        }finally{
            walk_isRunning = false;
        }
    }
    
    private int walk( Stack<File> s ){
        int res = 0;
        
        if( filter!=null && !filter.validate(s) ){
            return res;
        }
        
        enter(s);
        if( !s.empty() ){
            File p = s.peek();
            
            boolean hasErr = false;
            boolean stop = false;
            
            Iterable<File> children = null;
            try{
                children = getChildrenOf(p);
            }catch( Throwable err ){
                hasErr = true;
                ErrorBehavior eb = error(err);
                if( eb!=null ){
                    switch( eb ){
                        case Continue:
                            stop = false;
                            break;
                        case Stop:
                            stop = true;
                            break;
                    }
                }
            }
            
            if( !hasErr ){
                if( children!=null ){
                    for( File f : children ){
                        s.push(f);
                        int r = walk(s);
                        if( r!=0 ){
                            s.pop();
                            res = 1;
                            break;
                        }
                        s.pop();
                    }
                }
            }else{
                if( stop ){
                    res = 1;
                }else{
                    res = 0;
                }
            }
        }
        exit(s);
        
        return res;
    }
    
    public Iterable<TreeWalk<File>> treeWalkIterable( File root ){
        if( root==null )throw new IllegalArgumentException( "root==null" );
        final FileVisitor fv = clone();
        NodesExtracter<File,File> ne = new NodesExtracter<File, File>() {
            @Override
            public Iterable<File> extract(File from) {
                Iterable<File> itr = null;
                try{
                    itr = fv.getChildrenOf(from);
                }catch(Throwable err){
                    fv.error(err);
                    itr = null;
                }
                return itr;
            }
        };
        fv.getVisitHistory().clear();
        
        File froot = root;
        if( isUseAbsolutePath() ){
            froot = (File)froot.getAbsolute();
        }
        if( isUseCanonicalPath() ){
            File canon = (File)froot.getCanonical();
            froot = canon;
        }
        
//        return Iterators.tree(froot, ne);
//        TreeIterable treeItreable = new TreeIterable(ne, ne);
        
        final Stack<File> stack = new Stack<File>();
        
        Iterable<TreeWalk<File>> itr = TreeWalkItreator.createIterable(froot, 
            ne,
            new Predicate<TreeWalk<File>>(){
                @Override
                public boolean validate(TreeWalk<File> value) {
                    if( fv.filter==null )return true;
                    stack.clear();
                    for( File n : value.nodePath() ){
                        stack.add(n);
                    }
                    return fv.filter.validate(stack);
                }
            }
        );
        
        return itr;
    }
    
    /**
     * Возвращает последовательность файлов/каталогов (включая вложенгные)
     * @param root точка с которой начиается обход
     * @return последовательность файлов/каталогов
     */
    public Iterable<xyz.cofe.fs.File> walkIterable( File root ){
        if( root==null )throw new IllegalArgumentException( "root==null" );
//        final FileVisitor fv = clone();
//        NodesExtracter<File,File> ne = new NodesExtracter<File, File>() {
//            @Override
//            public Iterable<File> extract(File from) {
//                Iterable<File> itr = null;
//                try{
//                    itr = fv.getChildrenOf(from);
//                }catch(Throwable err){
//                    fv.error(err);
//                    itr = null;
//                }
//                return itr;
//            }
//        };
//        fv.getVisitHistory().clear();
//        
//        File froot = root;
//        if( isUseAbsolutePath() ){
//            froot = (File)froot.getAbsolute();
//        }
//        if( isUseCanonicalPath() ){
//            File canon = (File)froot.getCanonical();
//            froot = canon;
//        }
//        
////        return Iterators.tree(froot, ne);
////        TreeIterable treeItreable = new TreeIterable(ne, ne);
//        
//        final Stack<File> stack = new Stack<File>();
//        
//        Iterable<TreeWalk<File>> itr = TreeWalkItreator.createIterable(froot, 
//            ne,
//            new Predicate<TreeWalk<File>>(){
//                @Override
//                public boolean validate(TreeWalk<File> value) {
//                    if( fv.filter==null )return true;
//                    stack.clear();
//                    for( File n : value.nodePath() ){
//                        stack.add(n);
//                    }
//                    return fv.filter.validate(stack);
//                }
//            }
//        );
        Iterable<TreeWalk<File>> itr = treeWalkIterable(root);
        Iterable<File> resitr = Iterators.convert(itr, new ConvertTreeWalk());
        return resitr;
    }
    
    public static class ConvertTreeWalk <T extends File>
    implements Convertor<TreeWalk<T>, T>
    {
        @Override
        public T convert(TreeWalk<T> from) {
            return from.currentNode();
        }
//        @Override
//        public T convert(Iterable<TreeWalk<T>> from) {
//            return from.
//        }
    }
}
