/*
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.fs;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;
//import xyz.cofe.common.Text;

/**
 * Путь в нотации UNIX. <br>
 *
 * <font style="font-size:18pt">Примеры с пояснениями</font><br>
 *
 * <b>/home/user</b> - путь состоит из компонент, в данном случаи их две: home и user. <br>
 * Компоненты разделены между собой слешем (/). <br>
 * Этот путь является абсолютным, т.к. начинается с слеша. <br>
 *
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class UnixPath
implements Comparable<UnixPath>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(UnixPath.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(UnixPath.class.getName()).log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        Logger.getLogger(UnixPath.class.getName()).log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(UnixPath.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(UnixPath.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(UnixPath.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(UnixPath.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    // <editor-fold defaultstate="collapsed" desc="split">
    /**
     * Разделяет строку на подстроки, где в качестве разделитя указана строка. 
     * <table border="1" summary="Пример">
     * 
     * <tr style="font-weight:bold"><td>Исходная строка</td><td>Разделитель</td><td>Результат</td></tr>
     * 
     * <tr><td>"delim"</td><td>"delim"</td><td>""</td></tr>
     * 
     * <tr> <td>"delimabc"</td><td>"delim"</td><td>""<br>"abc"</td></tr>
     * <tr> <td>"abcdelimabc"</td><td>"delim"</td><td>"abc"<br>"abc"</td></tr>
     * 
     * <tr><td>"abcdelim"</td><td>"delim"</td><td>"abc"<br>""</td> </tr>
     * 
     * <tr> <td>"abc"</td><td>"delim"</td><td>"abc"</td></tr>
     * 
     * <tr> <td>"abc"</td><td>""</td><td>"abc"</td> </tr>
     * </table>
     *
     * @param src Исходня строка
     * @param splitter Разделитель
     * @return Подстроки
     */
    private static String[] split(String src, String splitter) {
        if (src == null) {
            throw new IllegalArgumentException("src == null");
        }
        if (splitter == null) {
            throw new IllegalArgumentException("splitter == null");
        }

        ArrayList<String> result = new ArrayList<String>();

        if (splitter.equals(src)) {
            result.add("");
        } else if (splitter.length() > src.length()) {
            result.add(src);
        } else {
            int offset = 0;
            while (true) {
                if (offset > src.length())
                    break;
                if (offset == src.length()) {
                    result.add("");
                    break;
                }
                int next = src.indexOf(splitter, offset);
                if (next < 0) {
                    String s = src.substring(offset, src.length());
                    result.add(s);
                    break;
                } else {
                    String s = src.substring(offset, next);
                    result.add(s);
                    offset = next + splitter.length();
                }
            }
        }
        return result.toArray(new String[]{});
    }

    private static Iterable<String> splitIterable(String src,String splitter){
        String[] arr = split(src,splitter);
        return xyz.cofe.collection.Iterators.array(arr);
    }

    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="join">
    /**
     * Объединяет строки вставляя между ними заданную строку
     *
     * @param lines Строки
     * @param glue Вставка
     * @return Результат склейки
     */
    private static String join(Iterable<String> lines, String glue) {
        if (lines == null) {
            throw new IllegalArgumentException("lines == null");
        }
        if (glue == null) {
            throw new IllegalArgumentException("glue == null");
        }
        StringBuilder res = new StringBuilder();
        int idx = -1;
        for( String line : lines ){
            idx++;
            if( idx>0 )res.append(glue);
            if( line!=null )res.append(line);
        }
        return res.toString();
    }

    /**
     * Объединяет строки вставляя между ними заданную строку
     *
     * @param lines Строки
     * @param glue Вставка
     * @param from С какой строки начать
     * @param count Сколько строк объединять
     * @return Результат склейки
     */
    private static String join(String[] lines, String glue, int from, int count) {
        if (lines == null) {
            throw new IllegalArgumentException("lines == null");
        }
        if (glue == null) {
            throw new IllegalArgumentException("glue == null");
        }
        String res = "";
        for (int i = 0; i < count; i++) {
            int idx = from + i;
            String line = (idx >= 0 && idx < lines.length) ? lines[idx] : "";
            if (line == null)
                line = "";
            if (i > 0)
                res += glue;
            res += line;
        }
        return res;
    }

    /**
     * Объединяет строки вставляя между ними заданную строку
     *
     * @param lines Строки
     * @param glue Вставка
     * @return Результат склейки
     */
    private static String join(String[] lines, String glue) {
        if (lines == null) {
            throw new IllegalArgumentException("lines == null");
        }
        if (glue == null) {
            throw new IllegalArgumentException("glue == null");
        }
        return join(lines, glue, 0, lines.length);
    }
    // </editor-fold>
    
    /**
     * Возвращает строку из указанной позиции заданной или меньше длинны
     * @param source исходный текст
     * @param beginIndex позиция, больше или равно 0, может быть больше длины строки
     * @param len длина искомого текста (может быть за пределами границ текста), больше или равен 0
     * @return текст, если за границами вернет пустую строку
     */
    private static String lookupText(String source,int beginIndex,int len){
        if( source==null )throw new IllegalArgumentException( "source==null" );
        if( len<0 )throw new IllegalArgumentException( "len<0" );
        if( beginIndex<0 )throw new IllegalArgumentException( "beginIndex<0" );
        
        if( beginIndex>=source.length() )return "";
        if( (beginIndex+len)>source.length() ){
            len = source.length() - beginIndex;
        }
        
        return source.substring(beginIndex, beginIndex+len);
    }
    
    private final static String filesep = "/"; //System.getProperty("file.separator", "/");
    private final static String current = ".";
    private final static String parent = "..";
    private final static String empty = "";

    private String[] components = null;
    private final boolean useEscape = true;
    private final char escape = '\\';

    /**
     * Конструктор
     * @param components компоненты пути
     */
    protected UnixPath( String[] components ){
        if( components==null )throw new IllegalArgumentException( "components==null" );
//        if( components.length<1 )throw new IllegalArgumentException( "components.length<1" );
        for( int i=0; i<components.length; i++ ){
            if( components[i]==null )throw new IllegalArgumentException( "components["+i+"]==null" );
        }
        this.components = components;
    }

    /**
     * Конструктор
     * @param path путь, текстовое представление
     * @param useEscape Использовать символ экронирования
     * @param escapeChar Символ экранирования
     * @param fileSepChar Символ разделения имен каталога/файла
     */
    public UnixPath( String path, boolean useEscape, char escapeChar, char fileSepChar ){
        if( path==null )throw new IllegalArgumentException( "path==null" );
        if( useEscape ){
//            components = parse(path, escape, filesep.charAt(0));
            components = parse(path, escapeChar, fileSepChar);
        }else{
            if( path.contains(""+fileSepChar) ){
//                components = Text.split(path, filesep);
                components = split(path, ""+fileSepChar);
            }else{
                components = new String[]{ path };
            }
        }
    }
    
    /**
     * Конструктор
     * @param path путь, текстовое представление
     */
    public UnixPath( String path ){
        if( path==null )throw new IllegalArgumentException( "path==null" );
        if( useEscape ){
            components = parse(path, escape, filesep.charAt(0));
        }else{
            if( path.contains(filesep) ){
//                components = Text.split(path, filesep);
                components = split(path, filesep);
            }else{
                components = new String[]{ path };
            }
        }
    }
    
//    private static String[] parse( 
//        String path, 
//        LinkedHashMap<String,String> escapeRewrite, 
//        String escapeAnyPrefix,
//        String filesep 
//    ){
//        ArrayList<String> list = new ArrayList<String>();
//        StringBuilder buff = new StringBuilder();
//        for( int i=0; i<path.length(); i++ ){
//            
//            if( escapeRewrite!=null ){
//                for( Map.Entry<String,String> escapeEntry : escapeRewrite.entrySet() ){
//                    if( escapeEntry==null )continue;
//                    String escapeChars = escapeEntry.getKey();
//                    String escapeRwrite = escapeEntry.getValue();
//                    if( escapeChars==null || escapeChars.length()==0 )continue;
//                    String lt = lookupText(path, i, escapeChars.length());
//                    if( escapeChars.equals(lt) ){
//                        if( escapeRwrite!=null ){
//                            buff.append(escapeRwrite);
//                        }
//                        i += escapeChars.length()-1;
//                    }
//                }
//            }
//            
//            if( escapeAnyPrefix!=null ){
//                
//            }
//            
////            String lt = Text.lookupText(path, i, 2);
//            String lt = lookupText(path, i, 2);
//            if( lt.length()==2 ){
//                char c0 = lt.charAt(0);
//                char c1 = lt.charAt(1);
//                if( c0==escape ){
//                    buff.append(c1);
//                    i+=1;
//                } else if( c0==filesep ){
//                    list.add(buff.toString());
//                    buff.setLength(0);
//                }else{
//                    buff.append(c0);
//                }
//            }else if( lt.length()==1 ){
//                char c0 = lt.charAt(0);
//                if( c0==escape ){
//                }else if( c0==filesep ){
//                    list.add(buff.toString());
//                    buff.setLength(0);
//                }else{
//                    buff.append(c0);
//                }
//            }
//        }
//        if( buff.length()>0 )list.add( buff.toString() );
//        return list.toArray( new String[]{} );
//    }

    private static String[] parse( String path, char escape, char filesep ){
        ArrayList<String> list = new ArrayList<String>();
        StringBuilder buff = new StringBuilder();
        for( int i=0; i<path.length(); i++ ){
//            String lt = Text.lookupText(path, i, 2);
            String lt = lookupText(path, i, 2);
            if( lt.length()==2 ){
                char c0 = lt.charAt(0);
                char c1 = lt.charAt(1);
                if( c0==escape ){
//                    if( c1==''
                    buff.append(c1);
                    i+=1;
                } else if( c0==filesep ){
                    list.add(buff.toString());
                    buff.setLength(0);
                }else{
                    buff.append(c0);
                }
            }else if( lt.length()==1 ){
                char c0 = lt.charAt(0);
                if( c0==escape ){
                }else if( c0==filesep ){
                    list.add(buff.toString());
                    buff.setLength(0);
                }else{
                    buff.append(c0);
                }
            }
        }
        if( buff.length()>0 )list.add( buff.toString() );
        return list.toArray( new String[]{} );
    }

    /**
     * Конструктор копирования
     * @param src исходный образец
     */
    public UnixPath( UnixPath src ){
        if( src==null )throw new IllegalArgumentException( "src==null" );
        this.components = src.components;
    }

    /**
     * Создает дочерний путь
     * @param base базовый путь
     * @param child дочернее имя или относительный путь
     */
    public UnixPath( UnixPath base, String child ){
        if( base==null )throw new IllegalArgumentException( "base==null" );
        if( child==null )throw new IllegalArgumentException( "child==null" );
        UnixPath childPath = new UnixPath(child);
        String[] cmpts = Arrays.copyOf(base.components, base.components.length + childPath.components.length);
        System.arraycopy(childPath.components, 0, cmpts, base.components.length, childPath.components.length);
        this.components = cmpts;
    }

    /**
     * Возвращает компонеты пути
     * @return компоненты пути
     */
    public String[] getComponents(){ return components; }

    /**
     * Возвращает компоненты пути с указанного индекса
     * @param beginIndex индекс компоненты
     * @return Компоненты пути
     */
    public String[] getComponents(int beginIndex){
        if( beginIndex<=0 )return components;
        if( beginIndex>=components.length )return new String[]{};
        return Arrays.copyOfRange(components, beginIndex, components.length);
    }

    private String encodeComponent( String component ){
        StringBuilder sb = new StringBuilder();
        char file_sep = filesep.charAt(0);
        for( int i=0; i<component.length(); i++ ){
            char c0 = component.charAt(i);
            if( c0==escape ){
                sb.append(escape).append(escape);
            }else if( c0==file_sep ){
                sb.append(escape).append(filesep);
            }else{
                sb.append(c0);
            }
        }
        return sb.toString();
    }

    /**
     * Возвращает текстовое представление
     * @return текстовое представление
     */
    @Override
    public String toString(){
        if( useEscape ){
            if( components.length==1 && components[0]!=null && components[0].equals("") )return filesep;
            StringBuilder path = new StringBuilder();
            String sep = filesep;
            int i=-1;
            for( String cmpt : components ){
                i++;
                if( i>0 )path.append(sep);
                path.append(encodeComponent(cmpt));
            }
            return path.toString();
        }else{
            if( components.length==1 && components[0]!=null && components[0].equals("") )return filesep;
//            return Text.join(components, filesep);
            return join(components, filesep);
        }
    }

    /**
     * Возвращает признак что путь является абсолютным
     * @return true - абсолютный путь
     */
    public boolean isAbsolute(){
        return components.length>0 && components[0]!=null && components[0].length()==0;
    }

    private boolean getAbsolute_allowRetNull(){ return false; }

    /**
     * Возвращает абсолютный путь относительно текущего каталога
     * @param currentDir текущий каталог
     * @return абсолютный путь
     */
    public UnixPath getAbsolute( UnixPath currentDir ){
        if( this.isAbsolute() )return this;
        if( currentDir==null )throw new IllegalArgumentException( "currentDir==null" );
        if( !currentDir.isAbsolute() )throw new IllegalArgumentException( "!currentDir.isAbsolute()" );
//        ArrayList<String> cmpts = new ArrayList<String>();
//        cmpts.addAll(Arrays.asList(currentDir.getComponents()));
//        for( String cmpt : components ){
//            if( cmpt.equals(empty) )continue;
//            if( cmpt.equals(current) )continue;
//            if( cmpt.equals(parent) ){
//                if( cmpts.size()>1 ){
//                    cmpts.remove(cmpts.size()-1);
//                    continue;
//                }else{
//                    if( getAbsolute_allowRetNull() ){
//                        return null;
//                    }else{
//                        continue;
//                    }
//                }
//            }
//            cmpts.add(cmpt);
//        }
//        return new UnixPath(cmpts.toArray(new String[]{}));

        ArrayList<String> cmpts = new ArrayList<String>();
        for( String c : currentDir.getComponents() ){
            cmpts.add(c);
        }
        for( String c : this.getComponents() ){
            cmpts.add(c);
        }
        return new UnixPath(cmpts.toArray(new String[]{}));
    }

//    public boolean isCanonical(){
//        if( components.length>0 ){
//            if( components[0]==null )return false;
//            if( components[0].equals(current) )return false;
//            if( components[0].equals(parent) )return false;
//        }
//        for( int i=1; i<components.length; i++ ){
//            String cmpt = components[i];
//            if( cmpt==null )return false;
//            if( cmpt.equals(current) )return false;
//            if( cmpt.equals(parent) )return false;
//            if( cmpt.equals(empty) )return false;
//        }
//        return true;
//    }

    /**
     * Возвращает признак что путь является относительным.
     * Антогонист к isAbsolute
     * @return true - путь является относительным
     */
    public boolean isRelative(){
        return !isAbsolute();
    }

    /**
     * Возвращает родительский путь
     * @return Родительский путь
     */
    public UnixPath getParent(){
        if( components.length<=1 ){
            return null;
        }
        return new UnixPath(Arrays.copyOf(components, components.length-1));
    }

    /**
     * Создает клон
     * @return клон
     */
    @Override
    public UnixPath clone(){
        return new UnixPath(this);
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 13 * hash + Arrays.deepHashCode(this.components);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final UnixPath other = (UnixPath) obj;
        if (!Arrays.deepEquals(this.components, other.components)) {
            return false;
        }
        return true;
    }

//    public UnixPath getCanonical(){
//        if( isCanonical() )return this;
//        ArrayList<String> cmpts = new ArrayList<String>();
//        if( components.length>0 ){
//            if( components[0].equals(current) ){
//                //skip
//            }else if( components[0].equals(parent) ){
//                return null;
//            }else if( components[
//        }
//        return new UnixPath(cmpts.toArray(new String[]{}));
//    }

    private boolean resolveLink_allowRetNull(){ return false; }

    /**
     * "Разрешает" ссылку. <br>
     * Примеры: <br>
     * <table summary="Пример преобразования">
     * <tr>
     * <td>current dir</td>
     * <td>this</td>
     * <td>link</td>
     * <td>result</td>
     * </tr>
     *
     * <tr>
     * <td>/opt/jdk/default</td>
     * <td>/opt/jdk/default</td>
     * <td>jdk1.8.0_05</td>
     * <td>/opt/jdk/jdk1.8.0_05</td>
     * </tr>
     *
     * <tr>
     * <td>/opt/jdk/default</td>
     * <td>/opt/jdk/default</td>
     * <td>./jdk1.8.0_05</td>
     * <td>/opt/jdk/jdk1.8.0_05</td>
     * </tr>
     *
     * <tr>
     * <td>/opt/jdk/default</td>
     * <td>/opt/jdk/default</td>
     * <td>../jdk1.8.0_05</td>
     * <td>/opt/jdk1.8.0_05</td>
     * </tr>
     *
     * <tr>
     * <td>/opt/jdk/default</td>
     * <td>/opt/jdk/default</td>
     * <td>../../jdk1.8.0_05</td>
     * <td>/jdk1.8.0_05</td>
     * </tr>
     *
     * <tr>
     * <td>/opt/jdk/default</td>
     * <td>/opt/jdk/default</td>
     * <td>../../../../jdk1.8.0_05</td>
     * <td>/jdk1.8.0_05</td>
     * </tr>
     *
     * </table>
     * @param link куда указывает ссылка
     * @param currentDir текущий каталог
     * @return цель
     */
    public UnixPath resolveLink( UnixPath link, UnixPath currentDir ){
        if( currentDir==null )throw new IllegalArgumentException( "currentDir==null" );
        if( link==null )throw new IllegalArgumentException( "path==null" );
        if( link.isAbsolute() )return link;

        UnixPath p = !isAbsolute() ? getAbsolute(currentDir) : this;

        ArrayList<String> cmpts = new ArrayList<String>();
        cmpts.addAll(Arrays.asList(p.getComponents()));

        if( cmpts.size()>0 )cmpts.remove(cmpts.size()-1);
        for( String cmpt : link.getComponents() ){
            if( cmpt.equals(current) )continue;
            if( cmpt.equals(empty) )continue;
            if( cmpt.equals(parent) ){
                if( cmpts.size()>1 ){
                    cmpts.remove(cmpts.size()-1);
                    continue;
                }else{
                    if( resolveLink_allowRetNull() ){
                        return null;
                    }else{
                        continue;
                    }
                }
            }
            cmpts.add(cmpt);
        }
        return new UnixPath(cmpts.toArray(new String[]{}));
    }

    /**
     * Приведение к каноничному виду, без лишних и пустых компонент пути
     * @return каноничная форма
     */
    public UnixPath getCanonical(){
        if( components!=null &&
            components.length==1 &&
            components[0]!=null &&
            components[0].equals(current)
        ){
            return this;
        }

        if( components!=null &&
            components.length==1 &&
            components[0]!=null &&
            components[0].length()==0
        ){
            return this;
        }

        ArrayList<String> cmpts = new ArrayList<String>();
        cmpts.addAll(Arrays.asList(UnixPath.this.getComponents()));
        cmpts = removeCurrent(cmpts);
        removeEmpties(cmpts);
        removeParent(cmpts);
//        removeTrailEmpties(cmpts);
        return new UnixPath(cmpts.toArray(new String[]{}));
    }

    private void removeTrailEmpties( ArrayList<String> components ){
        ArrayList<String> cmpts = new ArrayList<String>();
        int begin = -1;
        int endExc = -1;
        for( int i=components.size()-1; i>=0; i++ ){
            String cmpt0 = components.get(i);
            if( cmpt0.equals(empty) ){
                if( endExc<0 ){
                    begin = i;
                    endExc = i+1;
                }else{
                    begin = i;
                }
            }else{
                break;
            }
        }
        if( begin>0 && endExc>0 && endExc>begin ){
            for( int i=0; i<endExc - begin; i++ ){
                components.remove(begin);
            }
        }
    }

    private void removeParent( ArrayList<String> components ){
        boolean abs = components.size()>0 && components.get(0).length()==0;
        int begin = -1;
        int endExc = -1;
        int state = 0;
        while( state!=99 ){
            switch( state ){
                case 0: // find parent
                    begin = -1;
                    for( int i=abs ? 1 : 0; i<components.size()-1; i++ ){
                        String cmpt0 = components.get(i);
                        String cmpt1 = components.get(i+1);
                        if( !cmpt0.equals(parent) && cmpt1.equals(parent) ){
                            begin = i;
                            endExc = i+2;
                            break;
                        }
                    }
                    if( begin<0 ){
                        state = 99;
                    }else{
                        state = 2;
                    }
                    break;
                case 2: // remove diapasone begin .. end
                    for( int i=0; i<endExc - begin; i++ ){
                        components.remove(begin);
                    }
                    state = 0;
                    break;
                default:
                    state = 99;
                    break;
            }
        }
    }

    private void removeEmpties( ArrayList<String> components ){
        boolean abs = components.size()>0 && components.get(0).length()==0;
        int del = -1;
        int state = 0;
        while( state!=99 ){
            switch( state ){
                case 0:
                    del = -1;
                    for( int i=abs ? 1 : 0; i<components.size(); i++ ){
                        String cmpt0 = components.get(i);
                        if( cmpt0.equals(empty) ){
                            del = i;
                            break;
                        }
                    }
                    if( del>=0 ){
                        state = 1;
                    }else{
                        state = 99;
                    }
                    break;
                case 1:
                    components.remove(del);
                    state = 0;
                    break;
                default:
                    state = 99;
                    break;
            }
        }
    }

    private void removeDiplicateEmpties( ArrayList<String> components ){
        boolean abs = components.size()>0 && components.get(0).length()==0;
        int begin = -1;
        int endExc = -1;
        int state = 0;
        while( state!=99 ){
            switch( state ){
                case 0: // find begin dupl
                    begin = -1;
                    for( int i=abs ? 1 : 0; i<components.size()-1; i++ ){
                        String cmpt0 = components.get(i);
                        String cmpt1 = components.get(i+1);
                        if( cmpt0.equals(empty) && cmpt1.equals(empty) ){
                            begin = i;
                            break;
                        }
                    }
                    if( begin<0 ){
                        state = 99;
                    }else{
                        state = 1;
                    }
                    break;
                case 1: // find end dupl
                    endExc = -1;
                    for( int i=begin; i<components.size()-1; i++ ){
                        String cmpt0 = components.get(i);
                        String cmpt1 = components.get(i+1);
                        if( cmpt0.equals(empty) && !cmpt1.equals(empty) ){
                            endExc = i+1;
                            break;
                        }
                    }
                    if( endExc<0 )endExc = components.size();
                    state = 2;
                    break;
                case 2: // remove diapasone begin .. end
                    for( int i=0; i<endExc - begin; i++ ){
                        components.remove(begin);
                    }
                    state = 0;
                    break;
                default:
                    state = 99;
                    break;
            }
        }
    }

    private ArrayList<String> removeCurrent( ArrayList<String> components ){
        ArrayList<String> cmpts = new ArrayList<String>();
        for( String cmpt : components ){
            if( cmpt.equals(current) )continue;
            cmpts.add(cmpt);
        }
        return cmpts;
    }

    @Override
    public int compareTo(UnixPath o) {
        String p1 = toString();
        String p2 = o.toString();
        return p1.compareTo(p2);
    }

    public String getName(){
        if( components.length==0 )return "";
        if( components.length==1 )return components[0];
        UnixPath canon = getCanonical();
        if( canon!=null ){
            String[] cmpts = canon.getComponents();
            if( cmpts!=null && cmpts.length>0 )return cmpts[cmpts.length-1];
        }
        return components[components.length-1];
    }
}
