/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.gui.swing;

import java.lang.reflect.InvocationTargetException;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.SwingUtilities;
import xyz.cofe.common.Reciver;

/**
 * Принимает сообщения из любого потока (треда) и делегирует его в AWT/Swing поток
 * @author Kamnev Georgiy nt.gocha@gmail.com
 */
public class AWTReciver<T> implements Reciver<T> {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(AWTReciver.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(AWTReciver.class.getName()).log(Level.FINER, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(AWTReciver.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(AWTReciver.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(AWTReciver.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(AWTReciver.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    /**
     * Конструктор
     */
    public AWTReciver(){
    }

    /**
     * Конструктор
     * @param r приемник AWT, которому посылаются сообщения
     */
    public AWTReciver(Reciver<T> r){
        this.reciver = r;
    }

    /**
     * Конструктор
     * @param r приемник AWT, которому посылаются сообщения
     * @param awtSync true - ожидания получения сообщения (invokeAndWait) / <br>
     *                false - сообщение будет передано используя очередь сообщений (invokeLater)
     */
    public AWTReciver(Reciver<T> r, boolean awtSync){
        this.reciver = r;
        this.awtSync = awtSync;
    }

    //<editor-fold desc="reciver : Reciver<T> - приемник AWT">
    protected Reciver<T> reciver = null;

    /**
     * Возвращает приемник AWT, которому посылаются сообщения
     * @return приемник AWT
     */
    public synchronized Reciver<T> getReciver() {
        return reciver;
    }

    /**
     * Указывает приемник AWT, которому посылаются сообщения
     * @param reciver приемник AWT
     */
    public synchronized void setReciver(Reciver<T> reciver) {
        this.reciver = reciver;
    }
    //</editor-fold>
    //<editor-fold desc="awtSync : boolean">
    protected volatile boolean awtSync = true;

    /**
     * Возвращает как доставлять сообщеия: немедленно true = invokeAndWait / через очередь AWT - false = invokeLater
     * @return true - ожидания получения сообщения (invokeAndWait) / <br>
     *                false - сообщение будет передано используя очередь сообщений (invokeLater)
     */
    public synchronized boolean isAwtSync() {
        return awtSync;
    }

    /**
     * Указывает как доставлять сообщеия: немедленно true = invokeAndWait / через очередь AWT - false = invokeLater
     * @param awtSync true - ожидания получения сообщения (invokeAndWait) / <br>
     *                false - сообщение будет передано используя очередь сообщений (invokeLater)
     */
    public synchronized void setAwtSync(boolean awtSync) {
        this.awtSync = awtSync;
    }
    //</editor-fold>

    @Override
    public synchronized void recive(T obj) {
        if( SwingUtilities.isEventDispatchThread() ){
            if( reciver==null )return;
            reciver.recive(obj);
        }else{
            final Reciver rc = reciver;
            final T o = obj;
            Runnable r = new Runnable() {
                @Override
                public void run() {
                    rc.recive(o);
                }
            };
            if( awtSync ){
                try {
                    SwingUtilities.invokeAndWait(r);
                } catch (InterruptedException ex) {
                    Logger.getLogger(AWTReciver.class.getName()).log(Level.SEVERE, null, ex);
                } catch (InvocationTargetException ex) {
                    Logger.getLogger(AWTReciver.class.getName()).log(Level.SEVERE, null, ex);
                }
            }else{
                SwingUtilities.invokeLater(r);
            }
        }
    }
}
