/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.data;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;
import xyz.cofe.collection.BasicPair;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.Pair;
import xyz.cofe.collection.list.EventList;
import xyz.cofe.collection.list.IndexEventList;
import xyz.cofe.collection.list.SyncEventList;
import xyz.cofe.common.CloseableSet;
import xyz.cofe.common.Reciver;
import xyz.cofe.data.DataCellUpdated;
import xyz.cofe.data.DataColumn;
import xyz.cofe.data.DataColumnAdded;
import xyz.cofe.data.DataColumnRemoved;
import xyz.cofe.data.DataEvent;
import xyz.cofe.data.DataEventListener;
import xyz.cofe.data.DataRow;
import xyz.cofe.data.DataRowDeleted;
import xyz.cofe.data.DataRowErased;
import xyz.cofe.data.DataRowInserted;
import xyz.cofe.data.DataRowState;
import xyz.cofe.data.DataRowStateChanged;
import xyz.cofe.data.DataRowUndeleted;
import xyz.cofe.data.DataTable;
import xyz.cofe.data.DataTableDropped;
import xyz.cofe.gui.swing.table.EventSupport;

/**
 * Модель для таблицы с данными
 * @author Kamnev Georgiy nt.gocha@gmail.com
 */
public class DataTableModel
implements TableModel
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(DataTableModel.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(DataTableModel.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(DataTableModel.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(DataTableModel.class.getName(), method, result);
    }
    //</editor-fold>
    
    private final EventSupport esupport;
    private final Object sync;
    
    /**
     * Конструктор
     */
    public DataTableModel(){
        esupport = new EventSupport(this);
        sync = this;
        dataTableListeners = new CloseableSet();
        newRows = new SyncEventList<>(new ArrayList(),sync);
    }

    /**
     * Конструктор
     * @param dt таблица с данными
     */
    public DataTableModel( DataTable dt ){
        esupport = new EventSupport(this);
        sync = this;
        newRows = new SyncEventList<>(new ArrayList(),sync);
        dataTableListeners = new CloseableSet();
        setDataTable(dt);
    }
    
    protected final ConcurrentLinkedQueue<Runnable> swingRunQueue = new ConcurrentLinkedQueue<>();
    
    /**
     * Выполняет код в потоке SWING/AWT
     * @param run код
     */
    protected void swingRun( Runnable run ){
        if( run==null )return;
        swingRunQueue.add(run);
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                while (true) {                    
                    Runnable r = swingRunQueue.poll();
                    if( r==null )break;
                    r.run();
                }
            }
        });
    }

    //<editor-fold defaultstate="collapsed" desc="event support">
    /**
     * Возвращает поддержку уведомлений о измении свойств
     * @return поддержка уведомления измменения свойств
     */
    protected PropertyChangeSupport propertySupport() {
        return esupport.propertySupport();
    }
    
    /**
     * Возвращает поддержку уведомлений о измении свойств
     * @return поддержка уведомления измменения свойств
     */
    public EventSupport getEventSupport(){ return esupport; }

    /**
     * Уведомляет о измении свойства
     * @param property имя свойства
     * @param oldValue предыдущее значение
     * @param newValue текущее значение
     */
    protected void firePropertyChange(String property, Object oldValue, Object newValue) {
        esupport.firePropertyChange(property, oldValue, newValue);
    }
    
    /**
     * Добавляет подписчика на уведомления о изменении свойств
     * @param listener подписчик
     */
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        esupport.addPropertyChangeListener(listener);
    }
    
    /**
     * Удаляет подписчика от уведомлений о измении свойств
     * @param listener подписчик
     */
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        esupport.removePropertyChangeListener(listener);
    }
    
    /**
     * Указывает уведомлять в потоке SWING/AWT
     * @return true - уведомлять в потокое SWING/AWT
     */
    public boolean isNotifyInAwtThread() {
        return esupport.isNotifyInAwtThread();
    }
    
    /**
     * Указывает уведомлять в потоке SWING/AWT
     * @param notifyInAwtThread true - уведомлять в потокое SWING/AWT
     */
    public void setNotifyInAwtThread(boolean notifyInAwtThread) {
        esupport.setNotifyInAwtThread(notifyInAwtThread);
    }
    
    /**
     * Указывает уведомлять ли немедленно или отложенно
     * @return true - уведомление немедленно
     */
    public boolean isAwtInvokeAndWait() {
        return esupport.isAwtInvokeAndWait();
    }
    
    /**
     * Указывает уведомлять ли немедленно или отложенно
     * @param awtInvokeAndWait уведомление немедленно
     */
    public void setAwtInvokeAndWait(boolean awtInvokeAndWait) {
        esupport.setAwtInvokeAndWait(awtInvokeAndWait);
    }
    
    /**
     * Уведомляет о изменеии всех данных
     */
    public void fireAllChanged() {
        esupport.fireAllChanged();
    }
    
    /**
     * Уведомление о измении колонок таблицы
     */
    public void fireColumnsChanged() {
        esupport.fireColumnsChanged();
    }
    
    /**
     * Уведомление о измении строки таблицы
     * @param row строка таблицы
     */
    public void fireRowUpdated(int row) {
        esupport.fireRowUpdated(row);
    }
    
    /**
     * Уведомление о измении строк таблицы
     * @param rowIndexFrom начало диапазона строк
     * @param toIndexInclude конец диапазона строк
     */
    public void fireRowsUpdated(int rowIndexFrom, int toIndexInclude) {
        esupport.fireRowsUpdated(rowIndexFrom, toIndexInclude);
    }
    
    /**
     * Уведомление о изменении ячейки таблицы
     * @param rowIndex индекс строки
     * @param columnIndex индекс колонки
     */
    public void fireCellChanged(int rowIndex, int columnIndex) {
        esupport.fireCellChanged(rowIndex, columnIndex);
    }
    
    /**
     * Уведомление о добавлении строк в таблицу
     * @param rowIndexFrom  начало диапазона
     * @param toIndexInclude конец диапазона
     */
    public void fireRowsInserted(int rowIndexFrom, int toIndexInclude) {
        esupport.fireRowsInserted(rowIndexFrom, toIndexInclude);
    }
    
    /**
     * Уведомление о удалении строки из таблицу
     * @param rowIndexFrom начало диапазона
     * @param toIndexInclude конец диапазона
     */
    public void fireRowsDeleted(int rowIndexFrom, int toIndexInclude) {
        esupport.fireRowsDeleted(rowIndexFrom, toIndexInclude);
    }
    
    /**
     * Уведомление о событии таблицы
     * @param e событие таблицы
     */
    public void fireTableModelEvent(TableModelEvent e) {
        esupport.fireTableModelEvent(e);
    }
    
    /**
     * Возвращает подписчиков на события табличной модели
     * @return подписчики
     */
    public Collection<TableModelListener> getListenersCollection() {
        return esupport.getListenersCollection();
    }
    
    /**
     * Возвращает подписчиков на события табличной модели
     * @return подписчики
     */
    public TableModelListener[] getListeners() {
        return esupport.getListeners();
    }
    
    /**
     * Добавляет подписчика на события табличной модели
     * @param l подписчик
     */
    @Override
    public void addTableModelListener(TableModelListener l) {
        esupport.addTableModelListener(l);
    }
    
    /**
     * Удаляет подписчика на события табличной модели
     */
    @Override
    public void removeTableModelListener(TableModelListener l) {
        esupport.removeTableModelListener(l);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="deletedVisible">
    private boolean deletedVisible = false;
    
    /**
     * Указывает видны ли удаленные строки
     * @return true удааленные строки отображаются
     */
    public boolean isDeletedVisible() {
        synchronized(sync){
            return deletedVisible;
        }
    }
    
    /**
     * Указывает видны ли удаленные строки
     * @param deletedVisible true - отображать удаленные строки
     */
    public void setDeletedVisible(boolean deletedVisible) {
        Object old,cur;
        synchronized(sync){
            old = this.deletedVisible;
            this.deletedVisible = deletedVisible;
            rebuildDeletedCache();
            cur = this.deletedVisible;
        }
        firePropertyChange("deletedVisible", old, cur);
        fireAllChanged();
    }
    //</editor-fold>
    
    /**
     * Новые строки данных (Detached) для добавления
     */
    protected final EventList<DataRow> newRows;
    
    //<editor-fold defaultstate="collapsed" desc="newRow()">
    /**
     * Создание строки без добавление в таблицу (Detached)
     * @return Индекс созданной строки или null
     */
    public Pair<DataRow,Integer> newRow(){
        DataRow dr = null;
        int rowidx = -2;
        synchronized(sync){
            if( dataTable==null )throw new IllegalStateException("dataTable not set");
            if( readOnly ){
                if( readOnlyThrowException )throw new ReadOnlyError();
                return null;
            }
            
            dr = new DataRow(dataTable);
            newRows.add(dr);
            
            rowidx = indexOf(dr);
        }
        if( rowidx>=0 ){
            fireRowsInserted(rowidx, rowidx);
        }
        return new BasicPair<>(dr,rowidx);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="getNewRows()">
    /**
     * Возвращадет новые но не присоединенные строки (Detached)
     * @return строки данных
     */
    public List<Pair<DataRow,Integer>> getNewRows(){
        synchronized(sync){
            ArrayList<Pair<DataRow,Integer>> list = new ArrayList<>();
            for( DataRow dr : newRows ){
                if( dr!=null ){
                    list.add(new BasicPair<>(dr,indexOf(dr)));
                }
            }
            return list;
            //return newRows.toArray(new DataRow[]{});
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="insertNewRows()">
    /**
     * Вставляет новые строки в таблицу (Detached =&gt; Inserted)
     * @return вставленные строки
     */
    public List<Pair<DataRow,Integer>> insertNewRows(){
        synchronized(sync){
            ArrayList<Pair<DataRow,Integer>> inserted = new ArrayList<>();
            if(readOnly){
                if(readOnlyThrowException)throw new ReadOnlyError();
                return inserted;
            }
            
            List<Pair<DataRow,Integer>> created = getNewRows();
            for( Pair<DataRow,Integer> pdr : created ){
                int ri = insertRow(pdr.A());
                if( ri>=0 ){
                    inserted.add(new BasicPair<>(pdr.A(),ri));
                }
            }
            return inserted;
        }
        
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="clearNewRows()">
    /**
     * Удаление новых но не добавленных (Detached) строк из модели
     * @return Удаленные строки
     */
    public List<Pair<DataRow,Integer>> clearNewRows(){
        ArrayList<DataRow> removedRows = new ArrayList<>();
        ArrayList<Pair<DataRow,Integer>> removed = new ArrayList<>();
        
        int from = -1;
        int to = -1;
        synchronized(sync){
            int nrc = newRows.size();
            if( nrc>0 ){
                if( nrc==1 ){
                    from = indexOf(newRows.get(0));
                    to = from;
                }else{
                    from = indexOf(newRows.get(0));
                    to = indexOf(newRows.get(nrc-1));
                }
            }
            removedRows.addAll(newRows);
            newRows.clear();
        }
        if( from>=0 ){
            fireRowsDeleted(from, to);
        }
        
        int idxFrom = from;
        for( DataRow dr : removedRows ){
            removed.add(new BasicPair<>(dr,idxFrom));
            
            if( idxFrom>=0 ){
                idxFrom++;
            }
        }
        
        return removed;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="insertRow()">
    /**
     * Вставка созданной/новой строки
     * @param dr строка
     * @return Индекс строки или -1;
     */
    public int insertRow( DataRow dr ){
        if( dr==null )throw new IllegalArgumentException("dr == null");
        
        DataTable dt = null;
        int newRowIdx = -2;
        
        synchronized(sync){
            dt = dataTable;
            if( dt==null )throw new IllegalStateException("dataTable not set");
            
            if( readOnly ){
                if( readOnlyThrowException )throw new ReadOnlyError();
                return -1;
            }
            
            newRowIdx = newRows.indexOf(dr);
            if( newRowIdx>=0 ){
                newRows.remove(newRowIdx);
            }
        }
        if( newRowIdx>=0 ){
            fireRowsDeleted(newRowIdx, newRowIdx);
        }
        dt.insert(dr);
        
        return newRowIdx;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="dataTable">
    private DataTable dataTable;
    
    /**
     * Указывает таблицу
     * @return таблица с данными
     */
    public DataTable getDataTable() {
        synchronized(sync){
            return dataTable;
        }
    }
    
    /**
     * Указывает таблицу
     * @param dataTable таблица с данными
     */
    public void setDataTable(DataTable dataTable) {
        Object old, cur;
        synchronized(sync){
            old = this.dataTable;
            this.dataTable = dataTable;
            rebuildDeletedCache();
            newRows.clear();
            cur = this.dataTable;
        }
        firePropertyChange("dataTable", old, cur);
        listen(dataTable);
        //fireAllChanged();
        fireColumnsChanged();
        fireAllChanged();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="rebuildDeletedCache()">
    private final List<DataRow> deletedCache = new ArrayList<>();
    
    /**
     * Перестраивает кеш удалленых строк
     */
    private void rebuildDeletedCache(){
        synchronized(sync){
            deletedCache.clear();
            if( dataTable!=null && deletedVisible ){
                //deletedCache.addAll(dataTable.getDeletedRows());
                deletedCache.addAll(dataTable.rowsList(DataRowState.Deleted));
            }
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="listen DataTable">
    /**
     * Возвращает признак что данная таблица прослушивается
     * @return true - есть подписка на события таблицы
     */
    public boolean isListenDataTable(){
        synchronized(dataTableListeners){
            Object[] cls = dataTableListeners.getCloseables();
            return cls!=null && cls.length>0;
        }
    }
    
    /**
     * Устанавливает/снимает подписку на события таблицы
     * @param listen true - установить/false - снять подписку
     */
    public void setListenDataTable(boolean listen){
        synchronized(dataTableListeners){
            synchronized(sync){
                dataTableListeners.closeAll();
                if( listen ){
                    if( dataTable!=null ){
                        listen(dataTable);
                    }else{
                        throw new IllegalStateException("dataTable not set");
                    }
                }
            }
        }
    }
    
    protected final CloseableSet dataTableListeners;
    
    /**
     * Устанавливает подписчика на таблицу
     * @param tbl таблица данных
     */
    protected void listen( DataTable tbl ){
        synchronized( dataTableListeners ){
            dataTableListeners.closeAll();
            if( tbl==null )return;
            
            Closeable cl =
            tbl.addDataEventListener(new DataEventListener() {
                @Override
                public void dataEvent(DataEvent ev) {
                    if( ev instanceof DataRowInserted ){
                        DataRowInserted e = (DataRowInserted)ev;
                        onDataRowInserted(e);
                    }

                    if( ev instanceof DataRowDeleted ){
                        DataRowDeleted e = (DataRowDeleted)ev;
                        onDataRowDeleted(e);
                    }

                    if( ev instanceof DataRowUndeleted ){
                        DataRowUndeleted e = (DataRowUndeleted)ev;
                        onDataRowUndeleted(e);
                    }

                    if( ev instanceof DataRowErased ){
                        DataRowErased e = (DataRowErased)ev;
                        onDataRowErased(e);
                    }

                    if( ev instanceof DataRowStateChanged ){
                        DataRowStateChanged e = (DataRowStateChanged)ev;
                        onDataRowStateChanged(e);
                    }

                    if( ev instanceof DataCellUpdated ){
                        DataCellUpdated e = (DataCellUpdated)ev;
                        onDataCellUpdated(e);
                    }

                    if( ev instanceof DataTableDropped ){
                        DataTableDropped e = (DataTableDropped)ev;
                        onDataTableDropped(e);
                    }
                }
            });
            dataTableListeners.add(cl);
            
            /*cl =
            tbl.getDeletedRows().onAdded(new Reciver<DataRow>() {
                @Override
                public void recive(DataRow dr) {
                    if( dr!=null ){
                        onDeletedAdded(dr);
                    }
                }
            });
            dataTableListeners.add(cl);*/

            cl = tbl.onRowDeleted(new Reciver<DataRowDeleted>() {
                @Override
                public void recive(DataRowDeleted ev) {
                    onDeletedAdded(ev.getRow());
                }
            });
            dataTableListeners.add(cl);
            
            /*cl =
            tbl.getDeletedRows().onRemoved(new Reciver<DataRow>() {
                @Override
                public void recive(DataRow dr) {
                    if( dr!=null ){
                        onDeletedRemoved(dr);
                    }
                }
            });
            dataTableListeners.add(cl);*/
            
            cl = tbl.onRowErased(new Reciver<DataRowErased>() {
                @Override
                public void recive(DataRowErased ev) {
                    onDeletedRemoved(ev.getRow());
                }
            });
            dataTableListeners.add(cl);
            cl = tbl.onRowUndeleted(new Reciver<DataRowUndeleted>() {
                @Override
                public void recive(DataRowUndeleted ev) {
                    onDeletedRemoved(ev.getRow());
                }
            });
            dataTableListeners.add(cl);
            
            cl =
            tbl.onColumnAdded(new Reciver<DataColumnAdded>() {
                @Override
                public void recive(DataColumnAdded ev) {
                    onDataColumnAdded(ev.getColumn());
                }
            });
            dataTableListeners.add(cl);

            cl =
            tbl.onColumnRemoved(new Reciver<DataColumnRemoved>() {
                @Override
                public void recive(DataColumnRemoved ev) {
                    onDataColumnRemoved(ev.getColumn());
                }
            });
            dataTableListeners.add(cl);
        }
    }
    
    /**
     * Вызывается при добавлении колонки в таблицу
     * @param dc колонка таблицы
     */
    protected void onDataColumnAdded( DataColumn dc ){
        //System.out.println("col added: "+dc.getName());
        Runnable r = new Runnable() {
            @Override
            public void run() {
                fireColumnsChanged();
            }
        };
        swingRun(r);
    }
    
    /**
     * Вызывается при удалении колонки в таблицу
     * @param dc колонка таблицы
     */
    protected void onDataColumnRemoved( DataColumn dc ){
        //System.out.println("col removed: "+dc.getName());
        //fireColumnsChanged();
        Runnable r = new Runnable() {
            @Override
            public void run() {
                fireColumnsChanged();
            }
        };
        swingRun(r);
    }

    /**
     * Вызывается при измении состояния строки
     * @param e событие изменения строки
     */
    protected void onDataRowStateChanged(DataRowStateChanged e){
        if( dataTable==null )return;
        
        if( (e.getFromState() == DataRowState.Updated && e.getToState() == DataRowState.Fixed)
        ||  (e.getFromState() == DataRowState.Fixed && e.getToState() == DataRowState.Updated)
        ){
            final int ri = indexOf(e.getRow());
            if( ri>=0 ){
                Runnable r = new Runnable() {
                    @Override
                    public void run() {
                        fireRowUpdated(ri);
                    }
                };
                swingRun(r);
            }
        }
    }
    
    /**
     * Вызывается при добавлении строки
     * @param e событие
     */
    protected void onDataRowInserted(DataRowInserted e){
        final int ri = e.getRowIndex();
        if( ri>=0 ){            
            Runnable r = new Runnable() {
                @Override
                public void run() {
                    fireRowsInserted(ri, ri);
                }
            };
            swingRun(r);
        }
    }
    
    /**
     * Вызывается при возврате строки из корзины
     * @param e событие
     */
    protected void onDataRowUndeleted(DataRowUndeleted e){
        DataRow dr = e.getRow();
        final int ri = indexOf(dr);
        if( ri>=0 ){
            Runnable r = new Runnable() {
                @Override
                public void run() {
                    fireRowsInserted(ri, ri);
                }
            };
            swingRun(r);
        }
    }
    
    /**
     * Вызывается при удалении строки
     * @param e событие
     */
    protected void onDataRowDeleted(DataRowDeleted e){
        final int ri = e.getRowIndex();
        if( ri>=0 ){
            Runnable r = new Runnable() {
                @Override
                public void run() {
                    fireRowsDeleted(ri, ri);
                }
            };
            swingRun(r);
        }
    }
    
    /**
     * Вызывается при стирании строки из таблицы
     * @param e событие
     */
    protected void onDataRowErased(DataRowErased e){
        final int ri = e.getRowIndex();
        if( ri>=0 ){
            Runnable r = new Runnable() {
                @Override
                public void run() {
                    fireRowsDeleted(ri, ri);
                }
            };
            swingRun(r);
        }
    }
    
    /**
     * Вызывается при добавлении строки
     * @param dr строка
     */
    protected void onDeletedAdded( final DataRow dr ){
        Runnable r = new Runnable() {
            @Override
            public void run() {
                int ri = -1;
                boolean fire = false;
                synchronized (sync) {
                    if (deletedVisible && dataTable != null) {
                        deletedCache.add(dr);
                        ri = indexOf(dr);
                        fire = true;
                    }
                }
                if (ri >= 0 && fire) {
                    fireRowsInserted(ri, ri);
                }
            }
        };
        swingRun(r);
    }
    
    /**
     * Вызывается при удалении строки
     * @param dr строка
     */
    protected void onDeletedRemoved( final DataRow dr ){
        Runnable r = new Runnable() {
            @Override
            public void run() {
                int ri = -1;
                boolean fire = false;
                synchronized (sync) {
                    if (deletedVisible) {
                        int dri = deletedCache.indexOf(dr);
                        if (dri >= 0) {
                            deletedCache.remove(dri);
                        }
                        if (dri >= 0 && dataTable != null) {
                            int rc = dataTable.getRowsCount();
                            ri = rc + dri;
                            fire = true;
                        }
                    }
                }
                if (fire && ri >= 0) {
                    fireRowsDeleted(ri, ri);
                }
            }
        };
        swingRun(r);
    }
    
    /**
     * Вызывается при обновлении ячейки
     * @param e событие
     */
    protected void onDataCellUpdated( final DataCellUpdated e ){
        Runnable r = new Runnable() {
            @Override
            public void run() {
                DataRow dr = e.getRow();
                int ri = indexOf(dr);
                if (ri < 0) {
                    return;
                }
                
                int ci = e.getColumn();
                if (ci < 0) {
                    return;
                }
                
                int ecc = getExtraColumns().size();
                fireCellChanged(ri, ci + ecc);
            }
        };
        swingRun(r);
    }
    
    /**
     * Вызывается при полном удалении данных в таблице
     * @param e событие
     */
    protected void onDataTableDropped( final DataTableDropped e ){
        Runnable r = new Runnable() {
            @Override
            public void run() {
                deletedCache.clear();
                newRows.clear();
                
                fireColumnsChanged();
                fireAllChanged();
            }
        };
        swingRun(r);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="row mapping">
    //<editor-fold defaultstate="collapsed" desc="row(), rowmap()">
    /**
     * Возвращает отображение номера строк на строки данных
     * @param rows номера строк
     * @return пары номер строки / строка данных
     */
    public Map<Integer,DataRow> rowmap( Iterable<Integer> rows ){
        return rowmap(rows, true);
    }
    
    /**
     * Возвращает отображение номера строк на строки данных
     * @param rows номера строк
     * @param skipNull true пропускать пусты строки
     * @return карта отображения номер строки / строка данных
     */
    public Map<Integer,DataRow> rowmap( Iterable<Integer> rows, boolean skipNull ){
        Map<Integer,DataRow> map = new LinkedHashMap<>();
        synchronized(sync){
            if( dataTable==null )return map;
            if( rows!=null ){
                for( Integer r : rows ){
                    if( r==null )continue;
                    if( map.containsKey(r) )continue;
                    DataRow dr = row(r);
                    if( skipNull && dr==null )continue;
                    map.put(r, dr);
                }
            }
        }
        return map;
    }
    
    /**
     * Возвращает строку данных для указанной строки таблицы
     * @param row индекс строки
     * @return строка данных или null
     */
    public DataRow row( int row ){
        if( row<0 )return null;
        synchronized(sync){
            if( dataTable==null )return null;
            
            int dataRowCnt = dataTable.getRowsCount();
            int deletedRowCnt = deletedVisible ? deletedCache.size() : 0;
            int newRowCnt = newRows.size();
            
            if( row<dataRowCnt )return dataTable.getRow(row);
            if( row<dataRowCnt+deletedRowCnt && deletedRowCnt>0 )return deletedCache.get(row - dataRowCnt);
            if( row<dataRowCnt+dataRowCnt+newRowCnt && newRowCnt>0 )return newRows.get(row - dataRowCnt - deletedRowCnt);
            return null;
            
            /*if( row>=dataRowCnt ){
                if( deletedVisible ){
                    int drow = row-dataRowCnt;
                    if( drow<0 ){
                        return null;
                    }
                    if( drow>=deletedCache.size() ){
                        int nrow = drow - deletedCache.size();
                        if( nrow<0 )return null;
                        if( nrow>=newRows.size() )return null;
                        return newRows.get(nrow);
                    }
                    return deletedCache.get(drow);
                }else{
                    int nrow = row - dataRowCnt;
                    if( nrow>=0 && nrow<newRows.size() )return newRows.get(nrow);
                }
                
                return null;
            }
            return dataTable.row(row);
            */
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="indexOf(row) indexOfMap(rows)">
    /**
     * Возвращает отображение строк данных на индексы таблицы
     * @param rows строки данных
     * @return карта отображения строки данных на строки ui таблицы
     */
    public Map<DataRow,Integer> indexOfMap( Iterable<DataRow> rows ){
        return indexOfMap(rows, true);
    }
    
    /**
     * Возвращает отображение строк данных на индексы таблицы
     * @param rows строки данных
     * @param skipNegative пропускать строки которые не отображаются в ui таблице
     * @return карта отображения строки данных на строки ui таблицы
     */
    public Map<DataRow,Integer> indexOfMap( Iterable<DataRow> rows, boolean skipNegative ){
        Map<DataRow,Integer> map = new LinkedHashMap<>();
        synchronized(sync){
            if( dataTable==null )return map;
            for( DataRow dr : rows ){
                if( dr==null )continue;
                int ri = indexOf(dr);
                if( skipNegative && ri<0 ){
                    continue;
                }
                map.put(dr, ri);
            }
        }
        return map;
    }
    
    /**
     * Возвращает индекс строки ui таблицы
     * @param row строка данных
     * @return индекс строки или -1
     */
    public int indexOf( DataRow row ){
        if( row==null )return -1;
        synchronized(sync){
            if( dataTable==null )return -1;
            
            if( deletedVisible ){
                int dri = deletedCache.indexOf(row);
                if( dri>=0 ){
                    int rc = dataTable.getRowsCount();
                    return rc + dri;
                }
            }
            
            int dtIdx = dataTable.indexOf(row);
            if( dtIdx>=0 )return dtIdx;
            
            int nrIdx = newRows.indexOf(row);
            if( nrIdx>=0 ){
                int deletedRowCnt = deletedVisible ? deletedCache.size() : 0; 
                return dataTable.getRowsCount() + deletedRowCnt + nrIdx;
            }
            
            return -1;
        }
    }
    //</editor-fold>
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="rowCount">
    @Override
    public int getRowCount() {
        synchronized(sync){
            if( dataTable==null )return 0;
            int dataRCnt = dataTable.getRowsCount();
            int deletedRCnt = deletedCache.size();
            int newRCnt = newRows.size();
            if( deletedVisible ){
                return dataRCnt + deletedRCnt + newRCnt;
            }
            return dataRCnt+newRCnt;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="extraColumns">
    protected EventList<DataTableModelColumn> extraColumns;
    
    /**
     * Возвращает дополнительные колонки ui таблицы
     * @return расширенные колонки ui таблицы
     */
    public EventList<DataTableModelColumn> getExtraColumns(){
        synchronized(sync){
            if( extraColumns!=null )return extraColumns;
            extraColumns = new IndexEventList<>();
            extraColumns.onChanged(new Func3<Object, Integer, DataTableModelColumn, DataTableModelColumn>() {
                @Override
                public Object apply(Integer ci, DataTableModelColumn oldcol, DataTableModelColumn newcol) {
                    fireColumnsChanged();
                    fireAllChanged();
                    return true;
                }
            });
            return extraColumns;
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="columnCount">
    @Override
    public int getColumnCount() {
        synchronized(sync){
            DataTable dt = dataTable;
            if( dt==null )return getExtraColumns().size();
            synchronized(dt){
                int ecc = getExtraColumns().size();
                int dcc = dt.getColumnsCount();
                int cc = ecc + dcc;
                logFinest("getColumnCount()={0} ecc={1} dcc={2}", cc, ecc, dcc);
                return cc;
            }
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getColumnName(ci):String">
    @Override
    public String getColumnName(int columnIndex) {
        synchronized(sync){
            DataTable dataTable = this.dataTable;
            if( dataTable==null )return "?columnName";
            synchronized(dataTable){
                if( columnIndex<0 )throw new IllegalArgumentException("columnIndex("+columnIndex+")<0");
                if( columnIndex>=getColumnCount() )
                    throw new IllegalArgumentException(
                        "columnIndex("+columnIndex+")>=columnCount("+getColumnCount()+")");

                int ecc = getExtraColumns().size();
                if( columnIndex>=0 && columnIndex<ecc ){
                    return getExtraColumns().get(columnIndex).getName();
                }

                return dataTable.getColumn(columnIndex-ecc).getName();
            }
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="getDataColumn()">
    /**
     * Возвращает колонку данных для указанного индекса колонки
     * @param col индекс колонки
     * @return колонка данных или null
     */
    public DataColumn getDataColumn(int col){
        if( col<0 )return null;
        synchronized(sync){
            int extraCcnt = getExtraColumns().size();
            if( extraCcnt>0 && col>=0 && col<extraCcnt )return null;
            
            int tcol = col - extraCcnt;
            
            DataTable dt = getDataTable();
            if( dt==null ){
                logWarning("DataTable not set");
                return null;
            }
            
            int cc = dt.getColumnsCount();
            if( tcol>=cc )return null;
            
            return dt.getColumn(tcol);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="getExtraColumn()">
    /**
     * Возвращает расширенную колонку данных для указанного индекса колонки
     * @param col индекс колонки
     * @return расшриренная колонка
     */
    public DataTableModelColumn getExtraColumn(int col){
        if( col<0 )return null;
        synchronized(sync){
            if( col >= getExtraColumns().size() )return null;
            return getExtraColumns().get(col);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getColumnClass(ci):Class">
    @Override
    public Class<?> getColumnClass(int columnIndex) {
        synchronized(sync){
            DataTable dt = dataTable;
            if( dt==null )return String.class;
            synchronized( dt ){
                if( columnIndex<0 )throw new IllegalArgumentException("columnIndex("+columnIndex+")<0");

                if( columnIndex>=getColumnCount() )
                    throw new IllegalArgumentException(
                        "columnIndex("+columnIndex+")>=columnCount("+getColumnCount()+")");

                int extraCcnt = getExtraColumns().size();
                if( columnIndex>=0 && columnIndex<extraCcnt ){
                    return getExtraColumns().get(columnIndex).getDataType();
                }

                return dt.getColumn(columnIndex-extraCcnt).getDataType();
            }
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="readOnlyThrowException : boolean">
    protected boolean readOnlyThrowException = true;
    
    /**
     * Генерировать исключение при попытке изменить таблицу
     * @return true (по умолчанию) - генериовать исключение
     */
    public boolean isReadOnlyThrowException() {
        synchronized(sync){
            return readOnlyThrowException;
        }
    }
    
    /**
     * Генерировать исключение при попытке изменить таблицу
     * @param readOnlyThrowException true (по умолчанию) - генериовать исключение
     */
    public void setReadOnlyThrowException(boolean readOnlyThrowException) {
        Object old,cur;
        synchronized(sync){
            old = this.readOnlyThrowException;
            this.readOnlyThrowException = readOnlyThrowException;
            cur = this.readOnlyThrowException;
        }
        firePropertyChange("readOnlyThrowException", old, cur);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="readOnly : boolean">
    protected boolean readOnly = false;
    
    /**
     * Таблица доступна только для чтения
     * @return true - только для чтения
     */
    public boolean getReadOnly() {
        synchronized(sync){
            return readOnly;
        }
    }
    
    /**
     * Таблица доступна только для чтения
     * @param readOnly true - только для чтения
     */
    public void setReadOnly(boolean readOnly) {
        Object old,cur;
        synchronized(sync){
            old = this.readOnly;
            this.readOnly = readOnly;
            cur = this.readOnly;
        }
        firePropertyChange("readOnly", old, cur);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="isCellEditable(ri,ci):boolean">
    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        if( rowIndex<0 || columnIndex<0 )return false;
        synchronized(sync){
            DataTable dt = dataTable;
            if( dt==null )return false;
            synchronized( dt ){
                //int drc = dataTable.rowsCount();
                //int crc = deletedCache.size();
                //int rc = deletedVisible ? drc + crc : drc;

                //if( rowIndex>=rc )return false;
                if( rowIndex>=getRowCount() )return false;
                if( columnIndex>=getColumnCount() )return false;

                int ecc = getExtraColumns().size();
                if( columnIndex>=0 && columnIndex<ecc ){
                    return false;
                }
                
                if( readOnly ){
                    return false;
                }

                return true;
            }
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getValueAt(ri,ci):value">
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        if( rowIndex<0 || columnIndex<0 )return null;
        synchronized(sync){
            if( dataTable==null )return null;

            //int dataRcnt = dataTable.rowsCount();
            //int deletedRcnt = deletedCache.size();
            //int rowCnt = deletedVisible ? dataRcnt + deletedRcnt : dataRcnt;
            int rowCnt = getRowCount();
            
            if( rowIndex>=rowCnt )return null;
            if( columnIndex>=getColumnCount() )return null;
            
            //DataRow dr = rowIndex >= dataRcnt ? deletedCache.get(rowIndex - dataRcnt) : dataTable.row(rowIndex);
            DataRow dr = row(rowIndex);
            
            if( dr==null ){
                int dataRcnt = dataTable.getRowsCount();
                int deletedRcnt = deletedCache.size();
                logWarning(
                    "getValueAt({0},{1}) fetched null row (rc={2}, dataTable.rc={3}, cached.rc={4})",
                    rowIndex,columnIndex,
                    rowCnt,dataRcnt,deletedRcnt
                );
                return null;
            }

            int extraCcnt = getExtraColumns().size();
            if( columnIndex>=0 && columnIndex<extraCcnt ){
                DataTableModelColumn ec = getExtraColumns().get(columnIndex);
                Object v = ec.getValueFor(dr, rowIndex, this);
                
                logFinest("extra row={1} col={2} value={0}", v, rowIndex, columnIndex);
                return v;
            }
            
            int ti = columnIndex-extraCcnt;
            Object v = dr.get(ti);
            logFinest("data row={1} col={2} ti={3} value={0}", v, rowIndex, columnIndex, ti);
            return v;
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="setValueAt(v,ri,ci)">
    @Override
    public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
        if( rowIndex<0 || columnIndex<0 )return;
        synchronized(sync){
            if( dataTable==null )return;

            //int drc = dataTable.rowsCount();
            //int crc = deletedCache.size();
            //int rc = deletedVisible ? drc + crc : drc;
            
            //if( rowIndex>=rc )return;
            if( rowIndex>=getRowCount() )return;
            if( columnIndex>=getColumnCount() )return;
            if( readOnly ){
                if( readOnlyThrowException )throw new ReadOnlyError();
                return;
            }

            //DataRow dr = rowIndex >= drc ? deletedCache.get(rowIndex - drc) : dataTable.row(rowIndex);
            DataRow dr = row(rowIndex);
            if( dr==null ){
                int drc = dataTable.getRowsCount();
                int crc = deletedCache.size();
                int rc = deletedVisible ? drc + crc : drc;

                logWarning(
                    "getValueAt({0},{1}) fetched null row (rc={2}, dataTable.rc={3}, cached.rc={4})",
                    rowIndex,columnIndex,
                    rc,drc,crc
                );
            }
            
            int extraCcnt = getExtraColumns().size();
            /*if( columnIndex>=0 && columnIndex<ecc ){
                DataTableModelColumn ec = getExtraColumns().get(columnIndex);
                return ec.getValueFor(dr, rowIndex, this);
            }*/
            
            dr.set(columnIndex-extraCcnt,aValue);
        }
    }
    //</editor-fold>
}
