/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.data;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyEditor;
import java.net.URL;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JSeparator;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;
import xyz.cofe.collection.BasicPair;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Pair;
import xyz.cofe.common.Reciver;
import xyz.cofe.data.DataColumn;
import xyz.cofe.data.DataRow;
import xyz.cofe.data.DataTable;
import xyz.cofe.data.store.CSVDesc;
import xyz.cofe.data.store.CSVUtil;
import xyz.cofe.gui.swing.BasicAction;
import xyz.cofe.gui.swing.GuiUtil;
import xyz.cofe.gui.swing.SwingListener;
import xyz.cofe.gui.swing.cell.CellFormat;
import xyz.cofe.gui.swing.cell.DefaultLabelRender;
import xyz.cofe.gui.swing.cell.FlatTextRender;
import xyz.cofe.gui.swing.cell.LabelRender;
import xyz.cofe.gui.swing.cell.TCRenderer;
import xyz.cofe.gui.swing.cell.TableCellContext;
import xyz.cofe.gui.swing.properties.PropertyDB;
import xyz.cofe.gui.swing.properties.editor.DateEditor;
import xyz.cofe.gui.swing.properties.editor.TextFieldEditor;
import xyz.cofe.gui.swing.table.Table;
import xyz.cofe.gui.swing.table.TableEvent;
import xyz.cofe.gui.swing.tree.TreeTableNodeValueEditor;
import xyz.cofe.gui.swing.tree.TreeTableNodeValueEditorDef;
import xyz.cofe.perfomance.TimeLaps;
import xyz.cofe.text.EndLine;

/**
 * UI Таблица для отображения таблицы данных
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 * @see DataTable
 */
public class DataTableView extends Table {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(DataTableView.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(DataTableView.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(DataTableView.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(DataTableView.class.getName(), method, result);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="DataTableCellContext">
    /**
     * Описывает контекст отобрадения ячейки данных
     */
    public static class DataTableCellContext
        extends TableCellContext
    {
        //<editor-fold defaultstate="collapsed" desc="dataRow : DataRow">
        private DataRow dataRow;
        
        /**
         * Указывает отображаемую строку данных
         * @return сторка данных
         */
        public DataRow getDataRow() {
            return dataRow;
        }
        
        /**
         * Указывает отображаемую строку данных
         * @param dataRow сторка данных
         */
        public void setDataRow(DataRow dataRow) {
            this.dataRow = dataRow;
        }
        //</editor-fold>
        //<editor-fold defaultstate="collapsed" desc="dataTable : DataTable">
        protected DataTable dataTable;
        
        /**
         * Указывает таблицу данных
         * @return таблица данных
         */
        public DataTable getDataTable() {
            return dataTable;
        }
        
        /**
         * Указывает таблицу данных
         * @param dataTable таблица данных
         */
        public void setDataTable(DataTable dataTable) {
            this.dataTable = dataTable;
        }
        //</editor-fold>
        //<editor-fold defaultstate="collapsed" desc="dataTableModel : DataTableModel">
        protected DataTableModel dataTableModel;
        
        /**
         * Указывает модель таблицы данных
         * @return модель таблицы данных
         */
        public DataTableModel getDataTableModel() {
            return dataTableModel;
        }
        
        /**
         * Указывает модель таблицы данных
         * @param dataTableModel модель таблицы данных
         */
        public void setDataTableModel(DataTableModel dataTableModel) {
            this.dataTableModel = dataTableModel;
        }
        //</editor-fold>
        //<editor-fold defaultstate="collapsed" desc="tableColumn : TableColumn">
        protected TableColumn tableColumn;
        
        /**
         * Указывает колонку UI таблицы
         * @return колонка UI таблицы
         */
        public TableColumn getTableColumn() {
            return tableColumn;
        }
        
        /**
         * Указывает колонку UI таблицы
         * @param tableColumn колонка UI таблицы
         */
        public void setTableColumn(TableColumn tableColumn) {
            this.tableColumn = tableColumn;
        }
        //</editor-fold>
        //<editor-fold defaultstate="collapsed" desc="extraColumn : boolean">
        /**
         * Указывает что колонка является расширенной
         * @return расширенная колонка или null
         */
        public boolean isExtraColumn(){
            DataTableModel dtm = getDataTableModel();
            TableColumn tc = getTableColumn();
            if( dtm!=null && tc!=null && tc.getModelIndex()>=0 ){
                int mi = tc.getModelIndex();
                return dtm.getExtraColumn(mi) != null;
            }
            return false;
        }
        //</editor-fold>
        //<editor-fold defaultstate="collapsed" desc="dataColumn : DataColumn">
        /**
         * Указывает на колонку данных
         * @return колонка данных или null
         */
        public DataColumn getDataColumn(){
            DataTableModel dtm = getDataTableModel();
            TableColumn tc = getTableColumn();
            DataTable dt = getDataTable();
            if( tc==null || dtm==null || dt==null )return null;
            
            int mi = tc.getModelIndex();
            if( isExtraColumn() )return null;
            
            if( mi>=0 && mi<dt.getColumnsCount() )return dt.getColumn(mi);
            return null;
        }
        //</editor-fold>
    }
    //</editor-fold>
    
    /**
     * Конструктор по умолчанию
     */
    public DataTableView(){
        initDefaultCellRenders();
        setModel(getDataTableModel());
        initIcons();
        initStateCellFormat();
        initColumnsPopup();
        initInnerLines();
        initCopy();        
        initTypeEditors();
        
        //getLabelRender().getFormat().maxLinesCount(20);
        final FlatTextRender.TLRUCache cache = new FlatTextRender.TLRUCache(400*50, 1000L * 15L);
        getLabelRender().getTextRender().setCache(cache);
        getLabelRender().getTextRender().setMinCachedTextLength(0);
        //getLabelRender().getFormat().maxLinesCount(20);
        
        final TimeLaps tlPrepare = new TimeLaps();
        getLabelRender().getTextRender().setTimeLaps(tlPrepare);
    }
    
    //<editor-fold defaultstate="collapsed" desc="pdb : PropertyDB">
    private PropertyDB pdb;
    /**
     * Ссылка на базу свойств
     * @return база редакторов свойств
     */
    public PropertyDB getPDB(){
        if( pdb==null )pdb = new PropertyDB();
        return pdb;
    }
    /**
     * Указывает ссылку на базу свойств
     * @param pdb база редакторов свойств
     */
    public void setPDB(PropertyDB pdb){
        this.pdb = pdb;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="initTypeEditors()">
    private void initTypeEditors(){
        TreeTableNodeValueEditorDef edtr = new TreeTableNodeValueEditorDef();
        
        edtr.setEditorFinder(new TreeTableNodeValueEditor.EditorFinder() {
            @Override
            public TreeTableNodeValueEditor.Editor findEditor(JTable table, Object value, boolean isSelected, int row, int column) {
                DataTableModel dtm = getDataTableModel();
                if( dtm==null )throw new IllegalStateException("DataTableModel == null");
                
                DataColumn dc = dtm.getDataColumn(column);
                if( dc==null )return null;
                
                Class dataType = dc.getDataType();
                if( dataType==null )return null;
                
                if( Objects.equals(dataType, java.util.Date.class) ){
                    CellFormat cf = getCellFormat(row, column);
                    DateFormat df = cf==null ? null : cf.getDateFormat();
                    String dfPtrn = df instanceof SimpleDateFormat ?
                        ((SimpleDateFormat)df).toPattern() : null;
                    
                    return DateEditor.getDateEditor(dc.isAllowNull(), dfPtrn);
                }else if( Objects.equals(dataType, java.sql.Date.class) ){
                    CellFormat cf = getCellFormat(row, column);
                    DateFormat df = cf==null ? null : cf.getDateFormat();
                    String dfPtrn = df instanceof SimpleDateFormat ?
                        ((SimpleDateFormat)df).toPattern() : null;
                    
                    return DateEditor.getSQLDateEditor(dc.isAllowNull(), dfPtrn);
                }else if( Objects.equals(dataType, java.sql.Time.class) ){
                    CellFormat cf = getCellFormat(row, column);
                    DateFormat df = cf==null ? null : cf.getDateFormat();
                    String dfPtrn = df instanceof SimpleDateFormat ?
                        ((SimpleDateFormat)df).toPattern() : null;
                    
                    return DateEditor.getSQLTimeEditor(dc.isAllowNull(), dfPtrn);
                }else if( Objects.equals(dataType, java.sql.Timestamp.class) ){
                    CellFormat cf = getCellFormat(row, column);
                    DateFormat df = cf==null ? null : cf.getDateFormat();
                    String dfPtrn = df instanceof SimpleDateFormat ?
                        ((SimpleDateFormat)df).toPattern() : null;
                    
                    return DateEditor.getSQLTimestampEditor(dc.isAllowNull(), dfPtrn);
                }
                
                PropertyDB pdb = getPDB();
                if( pdb==null )return null;
                
                if( !dc.isAllowNull() ){
                    if( Objects.equals(dataType, Byte.class) )dataType = byte.class;
                    if( Objects.equals(dataType, Short.class) )dataType = short.class;
                    if( Objects.equals(dataType, Integer.class) )dataType = int.class;
                    if( Objects.equals(dataType, Long.class) )dataType = long.class;
                    if( Objects.equals(dataType, Float.class) )dataType = float.class;
                    if( Objects.equals(dataType, Double.class) )dataType = double.class;
                }
                
                PropertyEditor oEd = null;
                Set<PropertyEditor> editors = pdb.getTypeEditors().fetch(dataType);
                if( editors!=null && editors.size()>0 ){
                    List<PropertyEditor> editorsList = new ArrayList(editors);
                    while( editorsList.size()>1 ){
                        oEd = editorsList.get(0);
                        if( oEd instanceof TextFieldEditor ){
                            TextFieldEditor tfe = (TextFieldEditor)oEd;
                            if( !Objects.equals(tfe.isAllowNull(),dc.isAllowNull()) ){
                                editorsList.remove(0);
                                continue;
                            }
                        }
                        
                        PropertyEditor oEd2 = editorsList.get(1);
                        double w1 = pdb.getWeightOf(oEd);
                        double w2 = pdb.getWeightOf(oEd2);
                        if( w1>w2 ){
                            editorsList.remove(1);
                            continue;
                        }else if( w1<w2 ){
                            editorsList.remove(0);
                            continue;
                        }
                        
                        break;
                    }
                    oEd = editorsList.get(0);
                }
                
                if( !(oEd instanceof TreeTableNodeValueEditor.Editor) )return null;
                
                return oEd!=null ? (TreeTableNodeValueEditor.Editor)oEd : null;
            }
        });
        
        PropertyDB pdb = getPDB();
        if( pdb!=null ){
            for( Class edType : pdb.getTypeEditors().keySet() ){
                setDefaultEditor(edType, edtr);
            }
        }
    }
    //</editor-fold>
    
    @Override
    public void fireTableEvent(TableEvent event) {
        if( event instanceof TableEvent.FocusedRowChanged &&
            !( event instanceof DataRowFocusChanged ) 
        ){
            super.fireTableEvent(new DataRowFocusChanged((TableEvent.FocusedRowChanged)event));
            return;
        }
        super.fireTableEvent(event);
    }

    //<editor-fold defaultstate="collapsed" desc="initDefaultCellRenders()">
    /**
     * Инициализация рендера
     */
    protected void initDefaultCellRenders(){
        TCRenderer tcr = getTCRenderer();
        
        for( Class dataType : new Class[]{
            String.class,
            Number.class,
            Byte.class, byte.class,
            Short.class, short.class,
            Integer.class, int.class,
            Long.class, long.class,
            Float.class, float.class,
            Double.class, double.class,
            Boolean.class, boolean.class,
            Icon.class,            
            Object.class
        } ){
            setDefaultRenderer(dataType, tcr);
        }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="state icons">
    private Icon detachedIcon;
    private Icon deletedIcon;
    private Icon fixedIcon;
    private Icon modifiedIcon;
    private Icon insertedIcon;
    
    private int iconMaxWidth = 0;
    private int iconMaxHeight = 0;
    private Dimension iconMaxSize;
    
    //<editor-fold defaultstate="collapsed" desc="initIcons()">
    /**
     * Инициализация иконок
     */
    private void initIcons(){
        URL u1 = DataTableView.class.getResource("datarow-state-deleted.png");
        deletedIcon = u1==null ? null : new ImageIcon(u1);
        
        URL u2 = DataTableView.class.getResource("datarow-state-detached.png");
        detachedIcon = u2==null ? null : new ImageIcon(u2);
        
        URL u3 = DataTableView.class.getResource("datarow-state-fixed.png");
        fixedIcon = u3==null ? null : new ImageIcon(u3);
        
        URL u4 = DataTableView.class.getResource("datarow-state-inserted.png");
        insertedIcon = u4==null ? null : new ImageIcon(u4);
        
        URL u5 = DataTableView.class.getResource("datarow-state-modified.png");
        modifiedIcon = u5==null ? null : new ImageIcon(u5);
        
        for( Icon ic : new Icon[]{deletedIcon, detachedIcon, fixedIcon, insertedIcon, modifiedIcon} ){
            if(ic==null)continue;
            iconMaxWidth = Math.max(ic.getIconWidth(), iconMaxWidth);
            iconMaxHeight = Math.max(ic.getIconHeight(), iconMaxHeight);
        }
        
        iconMaxSize = new Dimension(iconMaxWidth, iconMaxHeight);
    }
    //</editor-fold>    
    //<editor-fold defaultstate="collapsed" desc="initStateCellFormat()">
    /**
     * Инициализация рендера состояния строки DataRow state
     */
    private void initStateCellFormat(){
        getTCRenderer().getLabelRender().addFormatter(
            DataTableCellContext.class,
            new Func2<CellFormat, DataTableCellContext, CellFormat>() {
                @Override
                public CellFormat apply(DataTableCellContext tcx, CellFormat cf) {
                    if( !tcx.isExtraColumn() )return cf;
                    
                    cf.setIconPlaceholder(iconMaxSize);
                    DataRow dr = tcx.getDataRow();
                    if( dr!=null ){
                        switch(dr.getState()){
                            case Deleted: cf.setIcon(deletedIcon); break;
                            case Detached: cf.setIcon(detachedIcon); break;
                            case Fixed: cf.setIcon(fixedIcon); break;
                            case Inserted: cf.setIcon(insertedIcon); break;
                            case Updated: cf.setIcon(modifiedIcon); break;
                            default:
                        }
                    }
                    
                    //cf.setIconPadRight(3);
                    cf.setTextPadLeft(3.0);
                    
                    return cf;
                }
            });
    }
    //</editor-fold>
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="initInnerLines()">
    /**
     * инициалзация внутренних линий
     */
    protected void initInnerLines(){
        getTCRenderer().getLabelRender().addFormatter(
            TableCellContext.class, new Func2<CellFormat, TableCellContext, CellFormat>() {
                @Override
                public CellFormat apply(TableCellContext tcx, CellFormat cf) {
                    if( tcx!=null ){
                        if( tcx.getColumn()>0 ){
                            cf.borderLeft(Color.gray, 1f, 3f, 2f);
                            cf.padLeft(2.0);
                        }
                    }
                    return cf;
                }
            });
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="labelRender : LabelRender">
    private LabelRender labelRender;
    /**
     * Рендер метки
     * @return рендер
     */
    public LabelRender getLabelRender(){
        if( labelRender!=null )return labelRender;
        labelRender = new DefaultLabelRender();
        return labelRender;
    }
    //</editor-fold>    
    //<editor-fold defaultstate="collapsed" desc="TCRenderer : TCRenderer">
    protected TCRenderer tcRender;
    /**
     * Рендер ячейки данных
     * @return рендер
     */
    public TCRenderer getTCRenderer(){
        if( tcRender!=null )return tcRender;
        
        LabelRender labelRender = getLabelRender();
        
        tcRender = new TCRenderer(labelRender){
            @Override
            public synchronized Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
                Component cmpt = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

                Object cc = getCellContext();
                if( cc instanceof DataTableCellContext ){
                    DataTableCellContext dtcc = (DataTableCellContext)cc;
                    dtcc.setDataTable(getDataTable());
                    dtcc.setDataRow(dataRow(row));
                    dtcc.setTableColumn(getColumnModel().getColumn(column));
                    dtcc.setDataTableModel(getDataTableModel());
                }
                
                return cmpt;
            }
        };
        
        tcRender.setCellContext(new DataTableCellContext());
        return tcRender;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="dataTableModel : DataTableModel">
    private DataTableModel dataTableModel;
    /**
     * Модель таблицы
     * @return модель таблицы
     */
    public DataTableModel getDataTableModel() {
        if( dataTableModel!=null )return dataTableModel;
        dataTableModel = new DataTableModel();
        return dataTableModel;
    }
    /**
     * Модель таблицы
     * @param dataTableModel модель таблицы
     */
    public void setDataTableModel(DataTableModel dataTableModel) {
        this.dataTableModel = dataTableModel;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="dataTable : DataTable">
    /**
     * Таблица с данными
     * @return Таблица с данными
     */
    public DataTable getDataTable(){
        return getDataTableModel().getDataTable();
    }   
    
    /**
     * Таблица с данными
     * @param dt Таблица с данными
     */
    public void setDataTable(DataTable dt){
        if (dt== null) {
            throw new IllegalArgumentException("dt==null");
        }
        
        DataTableModel dtm = getDataTableModel();
        dtm.setDataTable(dt);

        TableModel tm = getModel();
        if( tm!=null && Objects.equals(tm.getClass(), DefaultTableModel.class) ){
            setModel(dtm);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="deletedVisible : boolean">
    /**
     * Удаленные но не зафиксированные строки видны
     * @return true - видны DataRow.state = Deleted
     */
    public boolean isDeletedVisible() {
        return getDataTableModel().isDeletedVisible();
    }
    /**
     * Удаленные но не зафиксированные строки видны
     * @param deletedVisible true - видны DataRow.state = Deleted
     */
    public void setDeletedVisible(boolean deletedVisible) {
        getDataTableModel().setDeletedVisible(deletedVisible);
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="listenDataTable : boolean">
    /**
     * Принимать события от DataTable
     * @return true - принимать события
     */
    public boolean isListenDataTable() {
        if( dataTableModel==null )throw new IllegalStateException("dataTableModel == null");
        return dataTableModel.isListenDataTable();
    }
    
    /**
     * Принимать события от DataTable
     * @param listen true - принимать события
     */
    public void setListenDataTable(boolean listen) {
        if( dataTableModel==null )throw new IllegalStateException("dataTableModel == null");
        dataTableModel.setListenDataTable(listen);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="create new row">
    /**
     * Создание строки без добавление в таблицу (Detached)
     * @return Созданная строка или null
     */
    public Pair<DataRow,Integer> newRow() {
        Pair<DataRow,Integer> r = dataTableModel.newRow(); 
        return r;
        //return r>=0 ? new BasicPair<>( r>=0 ? dataRow(r) : null, r ) : null;
    }
    /**
     * Insert строки в таблицу
     * @param dr Строка
     * @return Индекс строки или null
     */
    public Pair<DataRow,Integer> insertRow(DataRow dr) {
        int r = dataTableModel.insertRow(dr);
        if( r<0 )return null;
        return new BasicPair<>(dr,r);
    }
    /**
     * Добавление (Insert) созданных строк (Detached)
     * @return Добавленные строки
     */
    public List<Pair<DataRow,Integer>> insertNewRows() {
        return dataTableModel.insertNewRows();
    }
    /**
     * Удаление новых но не добавленных (Detached) строк из модели
     * @return Удаленные строки
     */
    public List<Pair<DataRow,Integer>> clearNewRows() {
        return dataTableModel.clearNewRows();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="dataRow(row):DataRow">
    /**
     * Возвращение строки таблицы по ее индексу
     * @param row индекс строки
     * @return строка
     */
    public DataRow dataRow( int row ){
        return dataTableModel.row(row);
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="indexOf(DataRow):int">
    /**
     * Возвращает индекс строки
     * @param dr строка таблицы
     * @return индекс в модели или -1
     */
    public int indexOf( DataRow dr ){
        if( dr==null )return -1;
        DataTableModel dtm = getDataTableModel();
        if( dtm==null )return -1;
        return dtm.indexOf(dr);
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="getSelectedDataRows():List<Pair<DataRow,Integer>>">
    /**
     * Возвращает выбранные строки
     * @return выбранные строки
     */
    public List<Pair<DataRow,Integer>> getSelectedDataRows(){
        ArrayList<Pair<DataRow,Integer>> list = new ArrayList<>();
        int[] selectedRows = getSelectedRows();
        for( int ri : selectedRows ){
            DataRow dr = dataRow(ri);
            list.add(new BasicPair<>(dr,ri));
        }
        return list;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="focusedDataRow : Pair<DataRow,Integer>">
    /**
     * Возвращает строку таблицы содержащую фокус
     * @return строка или null
     */
    public Pair<DataRow,Integer> getFocusedDataRow(){
        int focRow = getFocusedRow();
        if( focRow<0 )return null;
        return new BasicPair<>( dataRow(focRow), focRow );
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getCellFormat()">
    /**
     * Возвращает форматирование для ячейки
     * @param row строка UI таблицы
     * @param col колонка UI таблицы
     * @return форматирование
     */
    public CellFormat getCellFormat( int row, int col ){
        LabelRender lrnd = getLabelRender();
        
        CellFormat cf = lrnd.getFormat().clone();
        if( row<0 || col<0 )return cf;
        
        DefaultLabelRender dlrnd = lrnd instanceof DefaultLabelRender ?
            (DefaultLabelRender)lrnd : null;
        
        if( dlrnd==null )return cf;
        
        TCRenderer tcRender = getTCRenderer();
        if( tcRender==null )return cf;
        
        TableModel tm = getModel();
        if( tm==null )return cf;
        
        int rowcnt = tm.getRowCount();
        int colcnt = tm.getColumnCount();
        if( row >= rowcnt )return cf;
        if( col >= colcnt )return cf;
        
        Object val = tm.getValueAt(row, col);
        
        tcRender.getTableCellRendererComponent(this, val, false, false, row, col);
        
        TableCellContext tcc = tcRender.getCellContext().clone();
        dlrnd.prepare(getMemGraphics2D(), tcc, cf);
        
        return cf;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="clipboard">
    //<editor-fold defaultstate="collapsed" desc="getCellText(row,col):String">
    /**
     * Возвращает текст ячейки UI таблицы
     * @param row строка UI таблицы
     * @param col колонка UI таблицы
     * @return текст или null
     */
    public String getCellText( int row, int col ){
        if( row<0 || col<0 )return null;
        
        TableModel tm = getModel();
        if( tm!=null ){
            int rowcnt = tm.getRowCount();
            int colcnt = tm.getColumnCount();
            if( row >= rowcnt )return null;
            if( col >= colcnt )return null;
            
            Object val = tm.getValueAt(row, col);
            
            TCRenderer tcRender = getTCRenderer();
            if( tcRender==null ){
                return val.toString();
            }
            
            LabelRender lrnd = getLabelRender();
            DefaultLabelRender dlrnd = lrnd instanceof DefaultLabelRender ?
                (DefaultLabelRender)lrnd : null;
            
            if( dlrnd!=null ){
                tcRender.getTableCellRendererComponent(this, val, false, false, row, col);
                
                CellFormat cf = dlrnd.getFormat().clone();
                TableCellContext tcc = tcRender.getCellContext().clone();
                dlrnd.prepare(getMemGraphics2D(), tcc, cf);
                
                Object oval = tcc.getValue();
                if( oval instanceof String ){
                    return (String)oval;
                }
                
                return val.toString();
            }else{
                if( val!=null ){
                    return val.toString();
                }
            }
        }
        return null;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="prepareCopy()">
    /**
     * Подготавливает данные для копирования в буфер обмена
     * @return данные для копирования в буфер обмена
     */
    public List<PreparedCopyCells> prepareCopy(){
        ArrayList<PreparedCopyCells> list = new ArrayList<>();
        
        int[] selectedCols = null;
        if( getColumnModel().getColumnSelectionAllowed() ){
            selectedCols = getColumnModel().getSelectedColumns();
        }else{
            selectedCols = new int[ getColumnModel().getColumnCount() ];
            for( int ci=0; ci<selectedCols.length; ci++ )selectedCols[ci] = ci;
        }
        
        if( selectedCols.length<1 )return list;
        
        TableColumnModel tcm = getColumnModel();
        TableModel tm = getModel();
        
        int[] selectedRows = getSelectedRows();
        for( int ri : selectedRows ){
            List<PreparedCopyCell> lpcc = new ArrayList<>();
            for( int ci : selectedCols ){
                PreparedCopyCell pcc = new PreparedCopyCell();
                
                pcc.setText( getCellText(ri, ci) );
                
                TableColumn tc = tcm.getColumn(ci);
                if( tc!=null && tc.getModelIndex()>=0 && tm!=null ){
                    pcc.setColumnName( tm.getColumnName(tc.getModelIndex()) );
                }
                
                if( tm instanceof DataTableModel && tc!=null && tc.getModelIndex()>=0 ){
                    DataTableModel dtm = (DataTableModel)tm;
                    DataTableModelColumn dtmc = dtm.getExtraColumn(tc.getModelIndex());
                    DataTable dt = dtm.getDataTable();
                    
                    pcc.setDataColumn( dtmc==null && dt!=null ? dt.getColumn(tc.getModelIndex()) : null );
                }
                pcc.setValue( getValueAt(ri, ci) );
                
                lpcc.add(pcc);
            }
            
            PreparedCopyCells pccRow = new PreparedCopyCells(ri, dataRow(ri), lpcc);
            list.add( pccRow );
        }
        
        return list;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="copy()">
    /**
     * Копирование данный в буфер обмена
     */
    public void copy(){
        List<PreparedCopyCells> preparedValues = prepareCopy();
        if( preparedValues==null || preparedValues.isEmpty() )return;
        
        CSVDesc csvDesc = new CSVDesc();
        csvDesc.setFixedWidth(false);
        csvDesc.setQuoteVariants(CSVDesc.QuoteVariants.Always);
        CSVUtil csvUtl = new CSVUtil();
        
        StringBuilder sb = new StringBuilder();
        
        int lineIdx = -1;
        for( PreparedCopyCells pccs : preparedValues ){
            lineIdx++;
            if( lineIdx>0 ){
                sb.append(EndLine.Default.get());
            }
            
            if( pccs.getCells().size()==1 ){
                String txt = pccs.getCells().get(0).getText();
                sb.append( txt==null ? "" : txt );
            }else{
                String[] txts = new String[pccs.getCells().size()];
                for( int i=0; i<txts.length; i++ ){
                    String txt = pccs.getCells().get(i).getText();
                    txts[i] = txt==null ? "" : txt;
                }
                String line = csvUtl.toString(txts, csvDesc);
                sb.append(line);
            }
        }
        
        StringSelection stringSelection = new StringSelection(sb.toString());
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        clipboard.setContents(stringSelection, stringSelection);
    }
    //</editor-fold>
    private final BasicAction copyAction = new BasicAction("copy", new Runnable(){
        @Override
        public void run() {
            copy();
        }
    });
    private void initCopy(){
        final KeyStroke ksCopy = KeyStroke.getKeyStroke("control pressed C");
        //getActionMap().put("copyAction", copyAction);
        //getInputMap().put(ksCopy, copyAction);
        SwingListener.onKeyPressed(this, new Reciver<KeyEvent>() {
            @Override
            public void recive(KeyEvent ke) {
                if( GuiUtil.match(ke, ksCopy) ){
                    copy();
                    ke.consume();
                }
            }
        });
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="is/setModelColumnVisible">
    /**
     * Указывает видна ли указанная колонка модели
     * @param mi индекс колонки модели
     * @return true - колонка отображается
     */
    public boolean isModelColumnVisible(int mi){
        TableColumnModel tcm = getColumnModel();
        for( int ci=0; ci<tcm.getColumnCount(); ci++ ){
            TableColumn tc = tcm.getColumn(ci);
            if( tc==null )continue;
            if( tc.getModelIndex()==mi )return true;
        }
        return false;
    }
    /**
     * Указывает отображать ли колонку модели
     * @param mi индекс колонки модели
     * @param visible true - отображать / false - скрыть
     */
    public void setModelColumnVisible(int mi, boolean visible){
        TableColumnModel tcm = getColumnModel();
        
        if( visible ){
            for( int ci=0; ci<tcm.getColumnCount(); ci++ ){
                TableColumn tc = tcm.getColumn(ci);
                if( tc==null )continue;
                if( tc.getModelIndex()==mi )return;
            }
            
            TableModel tm = getModel();
            if( tm!=null ){
                TableColumn tc = new TableColumn(mi);
                tc.setCellRenderer(getTCRenderer());
                tc.setHeaderValue(tm.getColumnName(mi));
                
                tcm.addColumn(tc);
            }
        }else{
            LinkedHashSet<TableColumn> removeSet = new LinkedHashSet<>();
            for( int ci=0; ci<tcm.getColumnCount(); ci++ ){
                TableColumn tc = tcm.getColumn(ci);
                if( tc==null )continue;
                if( tc.getModelIndex()==mi )removeSet.add( tc );
            }
            for( TableColumn tc : removeSet ){
                tcm.removeColumn(tc);
            }
        }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="column popup menu">
    /**
     * Создает меню для переключения видимости колонок данных
     * @return меню переключения колонок данных
     */
    protected JMenu createToggleModelColumnsVisibilityMenu(){
        JMenu visibleColumns = new JMenu("Model columns");
        TableModel tm = getModel();
        if( tm!=null ){
            int added = 0;
            for( int ci2=0; ci2<tm.getColumnCount(); ci2++ ){
                String cname = tm.getColumnName(ci2);
                if( cname==null )continue;
                
                final int mi = ci2;
                
                JCheckBoxMenuItem miVisible = new JCheckBoxMenuItem(cname);
                miVisible.addActionListener(new ActionListener() {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        boolean visible = isModelColumnVisible(mi);
                        setModelColumnVisible(mi, !visible);
                    }
                });
                miVisible.setSelected(isModelColumnVisible(mi));
                
                visibleColumns.add(miVisible);
                
                added++;
            }
            if( added>0 )return visibleColumns;
        }
        return null;
    }
    /**
     * Создает меню автоматического выравнивания колонок
     * @return меню автоматического выравнивания колонок
     */
    protected JMenu createAutoresizeModeMenu(){
        JMenu menuResize = new JMenu("Auto resize");
        
        int amode = getAutoResizeMode();
        
        JRadioButtonMenuItem mi1 = new JRadioButtonMenuItem("all columns");
        ButtonGroup bg = new ButtonGroup();
        
        mi1.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                setAutoResizeMode( JTable.AUTO_RESIZE_ALL_COLUMNS );
            }
        });
        mi1.setSelected(amode == JTable.AUTO_RESIZE_ALL_COLUMNS);
        
        JRadioButtonMenuItem mi2 = new JRadioButtonMenuItem("Subsequent columns");
        mi2.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                setAutoResizeMode( JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS );
            }
        });
        mi2.setSelected(amode == JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
        
        JRadioButtonMenuItem mi3 = new JRadioButtonMenuItem("Next column");
        mi3.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                setAutoResizeMode( JTable.AUTO_RESIZE_NEXT_COLUMN );
            }
        });
        mi3.setSelected(amode == JTable.AUTO_RESIZE_NEXT_COLUMN);
        
        JRadioButtonMenuItem mi4 = new JRadioButtonMenuItem("Last column");
        mi4.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                setAutoResizeMode( JTable.AUTO_RESIZE_LAST_COLUMN );
            }
        });
        mi4.setSelected(amode == JTable.AUTO_RESIZE_LAST_COLUMN);
        
        JRadioButtonMenuItem mi5 = new JRadioButtonMenuItem("Off");
        mi5.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                setAutoResizeMode( JTable.AUTO_RESIZE_OFF );
            }
        });
        mi5.setSelected(amode == JTable.AUTO_RESIZE_OFF);
        
        for( JRadioButtonMenuItem mi : new JRadioButtonMenuItem[]{
            mi1, mi2, mi3, mi4, mi5
        } ){
            bg.add(mi);
            menuResize.add(mi);
        }
        
        return menuResize;
    }
    /**
     * Меню переключения возможности вбора колонкок и строк / только строк
     * @return меню
     */
    protected JMenuItem createColumnSelectableMenu(){
        JCheckBoxMenuItem mi = new JCheckBoxMenuItem("Column selectable");
        mi.setSelected(
            getColumnModel().getColumnSelectionAllowed()
        );
        mi.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                boolean b = getColumnModel().getColumnSelectionAllowed();
                getColumnModel().setColumnSelectionAllowed(!b);
            }
        });
        return mi;
    }
    /**
     * Меню выбора клонок
     * @return меню выбора колонок
     */
    protected List<JMenuItem> createColumnsPopupItems(){
        ArrayList<JMenuItem> list = new ArrayList<>();
        
        JMenu visibleColumns = createToggleModelColumnsVisibilityMenu();
        if( visibleColumns!=null )list.add(visibleColumns);
        
        JMenu autoresizeColumns = createAutoresizeModeMenu();
        if( autoresizeColumns!=null )list.add(autoresizeColumns);
        
        JMenuItem miColSelectable = createColumnSelectableMenu();
        if( miColSelectable!=null )list.add(miColSelectable);
        
        return list;
    }
    /**
     * Создает контекстное меню
     * @return контекстное меню
     */
    protected JPopupMenu createColumnsPopupMenu(){
        JPopupMenu popup = new JPopupMenu();
        List itms = createColumnsPopupItems();
        if( itms!=null ){
            for( Object en : itms ){
                if( en==null )continue;
                if( en instanceof JSeparator ){
                    popup.addSeparator();
                }else if( en instanceof JMenuItem ){
                    popup.add((JMenuItem)en);
                }else if( en instanceof Action ){
                    popup.add((Action)en);
                }
            }
        }
        return popup;
    }
    /**
     * Отображает контекстное меню
     * @param me координаты мыши
     */
    public void showColumnsPopup( MouseEvent me ){
        if( me==null )return;
        showColumnsPopup(me.getComponent(), me.getX(), me.getY());
    }
    /**
     * Отображает контекстное меню
     * @param cmpt компонент события мыши
     * @param mouseX координаты мыши
     * @param mouseY координаты мыши
     */
    public void showColumnsPopup( Component cmpt, int mouseX, int mouseY ){
        if( cmpt==null )return;
        
        JPopupMenu menu = createColumnsPopupMenu();
        if( menu!=null ){
            menu.show(cmpt, mouseX, mouseY);
        }
    }
    /**
     * Инициализация контекстного меню при щелчке на заголовок таблицы
     */
    protected void initColumnsPopup(){
        final JTableHeader th = getTableHeader();
        if( th!=null ){
            th.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    if( e.getButton()==3 ){
                        showColumnsPopup(e);
                    }
                }
            });
        }
    }
    //</editor-fold>
}
