/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package xyz.cofe.gui.swing.font;

import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Locale;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func1;
import xyz.cofe.common.Reciver;
import xyz.cofe.gui.swing.SwingListener;

/**
 * Диалог выбора шрифта
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class FontChooser
    extends javax.swing.JDialog {

    /**
     * Creates new form FontChooser
     * @param parent Окно / null
     * @param modal Модальное окно
     */
    public FontChooser(java.awt.Frame parent, boolean modal) {
        super(parent, modal);
        initComponents();
        
        SwingListener.onActionPerformed(okButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                FontChooser.this.selectedFont = fontChooserPanel1.getUserFont();
                FontChooser.this.setVisible(false);                
            }
        });
        
        SwingListener.onActionPerformed(cancelButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                FontChooser.this.selectedFont = null;
                FontChooser.this.setVisible(false);                
            }
        });
        
        Locale loc = Locale.getDefault();
        URL url = null;
        for( String resname : new String[]{ 
            FontChooser.class.getSimpleName()+"_"+loc.getLanguage()+"_"+loc.getCountry()+".properties",
            FontChooser.class.getSimpleName()+"_"+loc.getLanguage()+".properties",
            FontChooser.class.getSimpleName()+".properties"
        } ){
            url = FontChooser.class.getResource(resname);
            if( url==null )continue;
            break;
        }
        
        if( url!=null ){
            final Properties props = new Properties();
            try {
                InputStream inp = url.openStream();
                props.load(inp);
                inp.close();
            } catch (IOException ex) {
                Logger.getLogger(FontChooser.class.getName()).log(Level.SEVERE, null, ex);
            }
            
            Func1<String,String> translFn = new Func1<String, String>() {
                @Override
                public String apply(String arg) {
                    if( arg==null )return null;
                    return props.getProperty(arg);
                }
            };
            
            translate(translFn);
        }
    }
    
    private String tr( String txt, Func1<String,String> fn ){
        if( txt==null )return null;
        if( fn==null )return txt;
        String t = fn.apply(txt);
        return t!=null ? t : txt;
    }
    
    /**
     * Указывает функцию перевода текстовых сообщений на местный язык
     * @param fn функция перевода
     */
    public void translate( Func1<String,String> fn ){
        if( fn==null )return;
        
        okButton.setText( tr(okButton.getText(), fn) );        
        cancelButton.setText( tr(cancelButton.getText(), fn) );      
        this.setTitle(tr(this.getTitle(), fn) );
        
        fontChooserPanel1.translate(fn);
    }
    
    protected Font selectedFont;
    
    /**
     * Возвращает выбранный шрифт
     * @return выбранный шрифт
     */
    public Font getSelectedFont(){
        return selectedFont;
    }
    
    /**
     * Указывает выбранный шрифт
     * @param fnt выбранный шрифт
     */
    public void setSelectedFont( Font fnt ){
        selectedFont = fnt;
    }
    
    /**
     * Указывает выбранный шрифт
     * @param font выбранный шрифт
     */
    public void select( Font font ){
        fontChooserPanel1.setUserFont(font);        
    }
    
    /**
     * Открывает диалог выбора шрифта
     * @param currentFont текущий шрифт, может быть null
     * @param parent родительское окно, может быть null
     * @param modal true - диалог модальный
     * @param translate функция перевода интерфеса, может быть null
     * @param consumer функция принимающая выбранный шрифт пользователем, может быть null
     * @return выбранный шрифт пользователем, модет быть null
     */
    public static Font choose( 
        Font currentFont, 
        java.awt.Frame parent, 
        boolean modal, 
        Func1<String,String> translate,
        final Reciver<Font> consumer
    ){
        final FontChooser fc = new FontChooser(parent, modal);
        if( translate!=null )fc.translate(translate);
        if( parent!=null ){
            fc.setLocationRelativeTo(parent);
        }
        fc.select(currentFont);
        fc.setVisible(true);
        fc.okButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                if( consumer!=null ){
                    Font f = fc.getSelectedFont();
                    if( f!=null ){
                        consumer.recive(f);
                    }
                }
            }
        });
        return fc.getSelectedFont();
    }
    
    /**
     * Открывает диалог выбора шрифта
     * @param currentFont текущий шрифт, может быть null
     * @param modal true - диалог модальный
     * @param consumer функция принимающая выбранный шрифт пользователем, может быть null
     * @return выбранный шрифт пользователем, модет быть null
     */
    public static Font choose( Font currentFont, boolean modal, Reciver<Font> consumer ){
        return choose(currentFont, null, modal, null, consumer);
    }

    /**
     * Открывает диалог выбора шрифта
     * @param currentFont текущий шрифт, может быть null
     * @param consumer функция принимающая выбранный шрифт пользователем, может быть null
     * @return выбранный шрифт пользователем, модет быть null 
     */
    public static Font choose( Font currentFont, Reciver<Font> consumer ){
        return choose(currentFont, null, false, null, consumer);
    }

    /**
     * Открывает диалог выбора шрифта
     * @param currentFont текущий шрифт, может быть null
     * @return выбранный шрифт пользователем, модет быть null
     */
    public static Font choose( Font currentFont ){
        return choose(currentFont, null, true, null, null);
    }
    /**
     * This method is called from within the constructor to initialize the form. WARNING: Do NOT modify this code. The
     * content of this method is always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        fontChooserPanel1 = new xyz.cofe.gui.swing.font.FontChooserPanel();
        okButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        setTitle("Select font");

        okButton.setText("ok");

        cancelButton.setText("cancel");

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup()
                        .addGap(0, 0, Short.MAX_VALUE)
                        .addComponent(cancelButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(okButton))
                    .addComponent(fontChooserPanel1, javax.swing.GroupLayout.DEFAULT_SIZE, 388, Short.MAX_VALUE))
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(fontChooserPanel1, javax.swing.GroupLayout.PREFERRED_SIZE, 255, Short.MAX_VALUE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(okButton)
                    .addComponent(cancelButton))
                .addContainerGap())
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    /**
     * @param args the command line arguments
     */
    public static void main(String args[]) {
        /* Set the Nimbus look and feel */
        //<editor-fold defaultstate="collapsed" desc=" Look and feel setting code (optional) ">
        /* If Nimbus (introduced in Java SE 6) is not available, stay with the default look and feel.
         * For details see http://download.oracle.com/javase/tutorial/uiswing/lookandfeel/plaf.html 
         */
        try {
            for (javax.swing.UIManager.LookAndFeelInfo info : javax.swing.UIManager.getInstalledLookAndFeels()) {
                if ("Nimbus".equals(info.getName())) {
                    javax.swing.UIManager.setLookAndFeel(info.getClassName());
                    break;
                }
            }
        } catch (ClassNotFoundException ex) {
            java.util.logging.Logger.getLogger(FontChooser.class.getName()).
                log(java.util.logging.Level.SEVERE, null, ex);
        } catch (InstantiationException ex) {
            java.util.logging.Logger.getLogger(FontChooser.class.getName()).
                log(java.util.logging.Level.SEVERE, null, ex);
        } catch (IllegalAccessException ex) {
            java.util.logging.Logger.getLogger(FontChooser.class.getName()).
                log(java.util.logging.Level.SEVERE, null, ex);
        } catch (javax.swing.UnsupportedLookAndFeelException ex) {
            java.util.logging.Logger.getLogger(FontChooser.class.getName()).
                log(java.util.logging.Level.SEVERE, null, ex);
        }
        //</editor-fold>

        /* Create and display the dialog */
        java.awt.EventQueue.invokeLater(new Runnable() {
            public void run() {
                FontChooser dialog = new FontChooser(new javax.swing.JFrame(), true);
                dialog.addWindowListener(new java.awt.event.WindowAdapter() {
                    @Override
                    public void windowClosing(java.awt.event.WindowEvent e) {
                        System.exit(0);
                    }
                });
                dialog.setVisible(true);
            }
        });
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton cancelButton;
    private xyz.cofe.gui.swing.font.FontChooserPanel fontChooserPanel1;
    private javax.swing.JButton okButton;
    // End of variables declaration//GEN-END:variables
}
