/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package xyz.cofe.gui.swing.font;

import java.awt.Font;
import java.awt.GraphicsEnvironment;
import java.awt.event.MouseWheelEvent;
import javax.swing.DefaultListModel;
import xyz.cofe.collection.Func1;
import xyz.cofe.common.Reciver;
import xyz.cofe.gui.swing.SwingListener;

/**
 * Панель выбора шрифта
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class FontChooserPanel
    extends javax.swing.JPanel {

    /**
     * Конструктор
     */
    public FontChooserPanel() {
        initComponents();
        initFontNames();
        setUserFontCalled = false;
        
        Reciver preview = new Reciver() {
            @Override
            public void recive(Object obj) {
                if( setUserFontCalled )return;
                preview(getUserFont());
            }
        };
        
        SwingListener.onValueChanged(fontNamesList, preview);
        SwingListener.onActionPerformed(plainRB, preview);
        SwingListener.onActionPerformed(boldRB, preview);
        SwingListener.onActionPerformed(italicRB, preview);
        SwingListener.onActionPerformed(italicBoldRB, preview);
        SwingListener.onStateChanged(sizeSpinner, preview);
        
        SwingListener.onMouseWheelMoved(sizeSpinner, new Reciver<MouseWheelEvent>() {
            @Override
            public void recive(MouseWheelEvent me) {
                if( me.getWheelRotation()>0 ){
                    Object nextv= sizeSpinner.getModel().getNextValue();
                    if( nextv!=null ){
                        sizeSpinner.getModel().setValue(nextv);
                    }
                }else if(me.getWheelRotation()<0) {
                    Object prevv= sizeSpinner.getModel().getPreviousValue();
                    if( prevv!=null ){
                        sizeSpinner.getModel().setValue(prevv);
                    }
                }
            }
        });
    }
    
    /**
     * Инициализация списка шрифтов
     */
    protected void initFontNames(){
        String fonts[] = 
            GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames();
        
        DefaultListModel<String> mdl = new DefaultListModel<String>();
        
        mdl.addElement(Font.DIALOG);
        mdl.addElement(Font.DIALOG_INPUT);
        mdl.addElement(Font.MONOSPACED);
        mdl.addElement(Font.SANS_SERIF);
        mdl.addElement(Font.SERIF);
        
        for( String fnt : fonts ){
            if( fnt==null || fnt.trim().length()<1 )continue;
            mdl.addElement(fnt);
        }
        
        fontNamesList.setModel(mdl);
    }
    
    private volatile boolean setUserFontCalled;
    
    /**
     * Указывает выбранный пользователем шрифт
     * @param fnt шрифт
     */
    public synchronized void setUserFont( Font fnt ){
        try{
            setUserFontCalled = true;
            
            if( fnt==null )return;

            fontNamesList.setSelectedValue(fnt.getName(), true);
            int stl = fnt.getStyle();

            boolean bld =  (stl & Font.BOLD) == Font.BOLD;
            boolean itl =  (stl & Font.ITALIC) == Font.ITALIC;

            if( stl==Font.PLAIN ){
                plainRB.setSelected(true);
            }else if( bld && !itl ){
                boldRB.setSelected(true);
            }else if( !bld && itl ){
                italicRB.setSelected(true);
            }else if( bld && itl ){
                italicBoldRB.setSelected(true);
            }

            sizeSpinner.setValue(fnt.getSize());
        }
        finally{
            setUserFontCalled = false;
        }
    }
    
    /**
     * Указывает выбранный пользователем шрифт
     * @return шрифт
     */
    public synchronized Font getUserFont(){
        String selfnt = fontNamesList.getSelectedValue();
        if( selfnt==null )selfnt = Font.DIALOG;
        
        int stl = 0;
        if( boldRB.isSelected() ){
            stl = Font.BOLD;
        }else if( italicRB.isSelected() ){
            stl = Font.ITALIC;
        }else if( italicBoldRB.isSelected() ){
            stl = Font.BOLD | Font.ITALIC;
        }else if( plainRB.isSelected() ){
            stl = 0;
        }
        
        int size = 10;
        Object osize = sizeSpinner.getValue();
        if( osize instanceof Number ){
            size = ((Number)osize).intValue();
        }
        if( size<1 )size = 1;
        
        return new Font(selfnt, stl, size);
    }
    
    private String tr( String txt, Func1<String,String> fn ){
        if( txt==null )return null;
        if( fn==null )return txt;
        String t = fn.apply(txt);
        return t!=null ? t : txt;
    }
    
    /**
     * Указывает функцию перевода пользовательского интерфейса на местный язык
     * @param fn функция перевода интерфейса
     */
    public void translate( Func1<String,String> fn ){
        if( fn==null )return;
        
        boldRB.setText( tr(boldRB.getText(), fn) );        
        italicBoldRB.setText( tr(italicBoldRB.getText(), fn) );        
        italicRB.setText( tr(italicRB.getText(), fn) );        
        plainRB.setText( tr(plainRB.getText(), fn) );
        previewTextField.setText( tr(previewTextField.getText(), fn) );
    }
    
    /**
     * Предпросмотр шрифта
     * @param fnt шрифт
     */
    public void preview( Font fnt ){
        if( fnt==null )return;
        previewTextField.setFont(fnt);
    }
    
    /**
     * This method is called from within the constructor to initialize the form. WARNING: Do NOT modify this code. The
     * content of this method is always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        buttonGroup1 = new javax.swing.ButtonGroup();
        jScrollPane1 = new javax.swing.JScrollPane();
        fontNamesList = new javax.swing.JList<>();
        plainRB = new javax.swing.JRadioButton();
        italicRB = new javax.swing.JRadioButton();
        boldRB = new javax.swing.JRadioButton();
        italicBoldRB = new javax.swing.JRadioButton();
        previewTextField = new javax.swing.JTextField();
        sizeSpinner = new javax.swing.JSpinner();

        setLayout(new java.awt.GridBagLayout());

        fontNamesList.setModel(new javax.swing.AbstractListModel<String>() {
            String[] strings = { "Item 1", "Item 2", "Item 3", "Item 4", "Item 5" };
            public int getSize() { return strings.length; }
            public String getElementAt(int i) { return strings[i]; }
        });
        fontNamesList.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
        jScrollPane1.setViewportView(fontNamesList);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridheight = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipadx = 200;
        gridBagConstraints.ipady = 150;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        add(jScrollPane1, gridBagConstraints);

        buttonGroup1.add(plainRB);
        plainRB.setText("Plain");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        add(plainRB, gridBagConstraints);

        buttonGroup1.add(italicRB);
        italicRB.setFont(italicRB.getFont().deriveFont((italicRB.getFont().getStyle() | java.awt.Font.ITALIC)));
        italicRB.setText("Italic");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.ipadx = 37;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        add(italicRB, gridBagConstraints);

        buttonGroup1.add(boldRB);
        boldRB.setFont(boldRB.getFont().deriveFont(boldRB.getFont().getStyle() | java.awt.Font.BOLD));
        boldRB.setText("Bold");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.ipadx = 36;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        add(boldRB, gridBagConstraints);

        buttonGroup1.add(italicBoldRB);
        italicBoldRB.setFont(italicBoldRB.getFont().deriveFont((italicBoldRB.getFont().getStyle() | java.awt.Font.ITALIC) | java.awt.Font.BOLD));
        italicBoldRB.setText("Italic Bold");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        add(italicBoldRB, gridBagConstraints);

        previewTextField.setText("Sample text");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.ipady = 73;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        add(previewTextField, gridBagConstraints);

        sizeSpinner.setModel(new javax.swing.SpinnerNumberModel(10, 1, null, 1));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.ipadx = 46;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        add(sizeSpinner, gridBagConstraints);
    }// </editor-fold>//GEN-END:initComponents


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JRadioButton boldRB;
    private javax.swing.ButtonGroup buttonGroup1;
    private javax.swing.JList<String> fontNamesList;
    private javax.swing.JRadioButton italicBoldRB;
    private javax.swing.JRadioButton italicRB;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JRadioButton plainRB;
    private javax.swing.JTextField previewTextField;
    private javax.swing.JSpinner sizeSpinner;
    // End of variables declaration//GEN-END:variables
}
