/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.font;

import java.awt.Font;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import xyz.cofe.common.Reciver;
import xyz.cofe.gui.swing.GuiUtil;
import xyz.cofe.gui.swing.SwingListener;
import xyz.cofe.gui.swing.properties.PropertyDB;
import xyz.cofe.gui.swing.properties.PropertyDBService;
import xyz.cofe.gui.swing.properties.editor.CustomEditor;

/**
 * Редактор шрифта для редактора свойств
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class FontPropertyEditor extends CustomEditor implements PropertyDBService
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(FontPropertyEditor.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(FontPropertyEditor.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(FontPropertyEditor.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(FontPropertyEditor.class.getName(), method, result);
    }
    //</editor-fold>
    
    public FontPropertyEditor(){
    }

    public FontPropertyEditor(FontPropertyEditor sample){
    }

    @Override
    public void register(PropertyDB pdb) {
        if( pdb!=null ){
            pdb.registerTypeEditor(Font.class, this);
        }
    }

    @Override
    public FontPropertyEditor clone() {
        return new FontPropertyEditor(this);
    }
    
    protected JPanel panel;
    protected synchronized JPanel getPanel(){
        if( panel!=null )return panel;
        panel = new JPanel(new GridBagLayout());
        
        GridBagConstraints gbc;
        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.weighty = 1;
        gbc.fill = GridBagConstraints.BOTH;
        panel.add( getLabel(), gbc );

        gbc = new GridBagConstraints();
        gbc.gridx = 2;
        gbc.gridy = 1;
        gbc.weightx = 0;
        gbc.weighty = 0;
        gbc.fill = GridBagConstraints.BOTH;
        panel.add( getEditButton(), gbc );
        
        return panel;
    }

    protected JLabel label;
    protected synchronized JLabel getLabel(){
        if( label!=null )return label;
        label = new JLabel();
        return label;
    }

    protected JButton editButton;
    protected synchronized JButton getEditButton(){
        if( editButton!=null )return editButton;
        editButton = new JButton("..");
        SwingListener.onActionPerformed(editButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                Frame frm = GuiUtil.getFrameOfComponent(editButton);
                Font nfnt = FontChooser.choose(font, frm, true, null, null);
                if( nfnt!=null ){
                    setFont(nfnt);
                    fireEditingStopped(FontPropertyEditor.this);
                }
            }
        });
        return editButton;
    }
    
    protected Font font;
    public Font getFont() {
        return font;
    }
    public void setFont(Font font) {
        this.font = font;
        if( font!=null ){
            getLabel().setText(font.getName()+" size="+font.getSize()+" style"+font.getStyle());
        }else{
            getLabel().setText("null");
        }
    }
    
    @Override
    protected JComponent createComponent() {
        return getPanel();
    }

    @Override
    public void setValue(Object value) {
        if( value instanceof Font ){
            setFont((Font)value);
        }else{
            setFont(null);
        }
    }

    @Override
    public Object getValue() {
        return getFont();
    }

    @Override
    public String getJavaInitializationString() {
        return "";
    }

    @Override
    public String getAsText() {
        if( font!=null ){
            return 
                "name="+font.getName()+"; size="+font.getSize()+"; style="+font.getStyle();
        }else{
            return "null";
        }
    }

    @Override
    public void setAsText(String text) throws IllegalArgumentException {
    }
}
