/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.log;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Filter;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

/**
 * Цепь фильтров
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class ChainFilters 
    implements Filter
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(ChainFilters.class.getName());

    private static Level logLevel(){ return logger.getLevel(); }
    
    private static boolean isLogSevere(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null ? true : logLevel.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level logLevel = logger.getLevel(); 
        return logLevel==null  ? true : logLevel.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINER.intValue();
    }
    
    private static boolean isLogFinest(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINEST.intValue();
    }

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(ChainFilters.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(ChainFilters.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(ChainFilters.class.getName(), method, result);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="action">
    private ChainAction defaultAction = ChainAction.Include;
    
    /**
     * Указвает действие по умолчанию
     * @return действие по умолчанию
     */
    public ChainAction getDefaultAction() {
        synchronized(this){
            return defaultAction;
        }
    }
    
    /**
     * Указвает действие по умолчанию
     * @param defaultAction действие по умолчанию
     */
    public void setDefaultAction(ChainAction defaultAction) {
        if( defaultAction==null )throw new IllegalArgumentException("defaultAction == null");
        synchronized(this){
            this.defaultAction = defaultAction;
        }
    }
    
    /**
     * Указывает действие по умолчанию
     * @param act действие по умолчанию
     * @return self ссылка
     */
    public ChainFilters defaultAction( ChainAction act ){
        setDefaultAction(act);
        return this;
    }
    //</editor-fold>
    
    protected final List<ChainElement> filters = new ArrayList<>();
    
    //<editor-fold defaultstate="collapsed" desc="onAdded/Removed">
    /**
     * Вызывается при добавлении в цепочку фильтров
     * @param member фильтр
     */
    protected void onAdded( ChainElement member ){
        if( member instanceof MaskFilter ){
            ((MaskFilter)member).setFiltersChain(this);
        }
    }
    
    /**
     * Вызывается при удалении из цепи фильтра
     * @param member фильтр
     */
    protected void onRemoved( ChainElement member ){
        if( member instanceof MaskFilter ){
            ((MaskFilter)member).setFiltersChain(null);
        }
    }
    //</editor-fold>
    
    /**
     * Добавляет в цепь фильтров еще один фильтр
     * @param member фильтр
     * @return индекс добавленного элемента
     */
    public int add( ChainElement member ){
        if( member==null )throw new IllegalArgumentException("member == null");
        synchronized(this){
            filters.add(member);
            onAdded(member);
            return filters.indexOf(member);
        }
    }
    
    /**
     * Добавляет фильтр в списк фильтров
     * @param idx индекс элемента/фильтра
     * @param member фильтр
     * @return индекст добавленного элемента
     */
    public int insert( int idx, ChainElement member ){
        if( member==null )throw new IllegalArgumentException("member == null");
        synchronized(this){
            filters.add(idx,member);
            onAdded(member);
            return filters.indexOf(member);
        }
    }
    
    /**
     * Удляет фильтр из списка фильтров
     * @param idx индекс удаляемого фильтра
     * @return Удаленный элемент
     */
    public ChainElement remove( int idx ){
        synchronized(this){
            ChainElement e = filters.remove(idx);
            onRemoved(e);
            return e;
        }
    }
    
    /**
     * Удаляеи фильтр из списка фильтров
     * @param member фильтр
     * @return индекс удаленного элемента
     */
    public int remove( ChainElement member ){
        if( member==null )throw new IllegalArgumentException("member == null");
        synchronized(this){
            int idx = filters.indexOf(member);
            remove(idx);
            return idx;
        }
    }
    
    /**
     * Удаляет все элементы списка фильтров
     * @return self ссылка
     */
    public ChainFilters clear(){
        synchronized(this){
//            for( ChainElement e : filter
            filters.clear();
            return this;
        }
    }
    
    /**
     * Возвращает элемент списка фильтров
     * @param idx индекс элемента
     * @return элемент
     */
    public ChainElement get( int idx ){
        synchronized(this){
            return filters.get(idx);
        }
    }
    
    /**
     * Указывает элемент списка фильтров
     * @param idx индекс заменяемого элемента
     * @param newMember новый элемент
     * @return предыдущий элемент
     */
    public ChainElement set( int idx, ChainElement newMember ){
        synchronized(this){
            return filters.set(idx, newMember);
        }
    }
    
    /**
     * Возвращает элементы списка фильтров
     * @return элементы списка
     */
    public ChainElement[] array(){
        synchronized(this){
            return filters.toArray(new ChainElement[]{});
        }
    }
    
    /**
     * Возвращает размер списка фильтров
     * @return размер списка
     */
    public int size(){
        synchronized(this){
            return filters.size();
        }
    }
    
    @Override
    public boolean isLoggable(LogRecord record) {
        synchronized(this){
            for( int i=0; i<size(); i++ ){
                ChainElement e = get(i);
                if( e==null )continue;
                if( !e.isEnabled() )continue;
                
                ChainAction act = e.isLoggable(record);
                if( act==null )continue;
                
                if( ChainAction.Include.equals(act) )return true;
                if( ChainAction.Exclude.equals(act) )return false;
            }
            
            ChainAction defAct = getDefaultAction();
            if( ChainAction.Include.equals(defAct) ){
                return true;
            }else{
                return false;
            }
        }
    }
}
