/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.log;

import java.awt.BorderLayout;
import java.awt.GraphicsEnvironment;
import java.awt.event.WindowEvent;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import xyz.cofe.common.Reciver;
import xyz.cofe.gui.swing.GuiUtil;
import xyz.cofe.gui.swing.SwingListener;

/**
 * Фрейм отображающий лог в текущий момент
 * @author Kamnev Georgiy
 */
public class LogHandlerFrame extends JFrame {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(LogHandlerFrame.class.getName());

    private static Level logLevel(){ return logger.getLevel(); }
    
    private static boolean isLogSevere(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null ? true : logLevel.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level logLevel = logger.getLevel(); 
        return logLevel==null  ? true : logLevel.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINER.intValue();
    }
    
    private static boolean isLogFinest(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINEST.intValue();
    }

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(LogHandlerFrame.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(LogHandlerFrame.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(LogHandlerFrame.class.getName(), method, result);
    }
    //</editor-fold>
    
    public LogHandlerFrame(){
        logHandlerPanel = new LogHandlerPanel();
        
        getContentPane().setLayout(new BorderLayout());
        getContentPane().add(logHandlerPanel);
        
        pack();
        setLocationRelativeTo(null);
        
        setTitle("Log handler");        
        setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
        SwingListener.onWindowClosing(this, new Reciver<WindowEvent>() {
            @Override
            public void recive(WindowEvent obj) {
                if( onClose().stopAll ){
                    logHandlerPanel.stop();
                }
            }
        });
        
        /*
        SwingListener.onWindowOpened(this, new Reciver<WindowEvent>() {
            @Override
            public void recive(WindowEvent obj) {
                if( onOpen().alignX!=null && onOpen().alignY!=null ){
                    GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
                    int screenX = ge.getCenterPoint().x * 2;
                    int screenY = ge.getCenterPoint().y * 2;
                    
                    int width = getWidth();
                    int height = getHeight();
                    
                    int maxX = Math.max(width, screenX);
                    int maxY = Math.max(height, screenY);
                    
                    int freeX = maxX - Math.min(width, screenX);
                    int freeY = maxY - Math.min(height, screenY);
                    
                    LogHandlerFrame.this.setLocation( 
                        (int)(onOpen().alignX * freeX) , 
                        (int)(onOpen().alignY * freeY)
                    );
                }
            }
        });
        */
    }
    
    //<editor-fold defaultstate="collapsed" desc="OnClose">
    public static class OnClose {
        public final LogHandlerFrame frame;
        public OnClose( LogHandlerFrame frame ){ this.frame=frame; }
        
        public boolean stopAll = true;
        public OnClose stopAll(boolean stop){ stopAll=stop; return this; }
        
        public Configure conf(){
            return new Configure(frame);
        }
    }
    
    protected OnClose onClose = new OnClose(this);
    public OnClose onClose(){ return onClose; }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="OnOpen">
    public static class OnOpen {
        public final LogHandlerFrame frame;
        public OnOpen( LogHandlerFrame frame ){ this.frame=frame; }
        
        public Double alignX = null;
        public Double alignY = null;
        public OnOpen align( Double x, Double y ){
            alignX = x; alignY = y;
            return this;
        }
        
        public Configure conf(){
            return new Configure(frame);
        }
    }
    
    protected OnOpen onOpen = new OnOpen(this);
    public OnOpen onOpen(){ return onOpen; }
    //</editor-fold>
    
    private final LogHandlerPanel logHandlerPanel;
    
    public LogHandlerPanel getLogHandlerPanel(){ return logHandlerPanel; }
    public Handler getLogHandler(){ return logHandlerPanel.getLogHandler(); }
    
    //<editor-fold defaultstate="collapsed" desc="Configure">
    public Configure configure(){ return new Configure(this); }
    public static class Configure
        extends LogHandlerPanel.Configure
    {
        protected final LogHandlerFrame frame;
        
        public Configure(LogHandlerFrame frame) {
            super(frame.logHandlerPanel);
            this.frame = frame;
        }
        
        @Override
        public Configure stop() {
            super.stop();
            return this;
        }
        
        @Override
        public Configure start() {
            super.start();
            return this;
        }
        
        @Override
        public Configure start(LoggerLevel level) {
            super.start(level);
            return this;
        }
        
        @Override
        public ConfigureFilters filters() {
            super.filters();
            return new ConfigureFilters(frame);
        }
        
        public void show(){
            Runnable r = new Runnable() {
                @Override
                public void run() {
                    frame.setVisible(true);
                }
            };
            if( SwingUtilities.isEventDispatchThread() ){
                r.run();
            }else{
                SwingUtilities.invokeLater(r);
            }
        }
        
        public OnOpen onOpen(){ return frame.onOpen(); }
        public OnClose onClose(){ return frame.onClose(); }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="ConfigureFilters">
    public static class ConfigureFilters
        extends LogHandlerPanel.ConfigureFilters
    {
        protected final LogHandlerFrame frame;
        
        public ConfigureFilters(LogHandlerFrame frame) {
            super(frame.logHandlerPanel);
            this.frame = frame;
        }
        
        @Override
        public Configure conf() {
            return new Configure(frame);
        }
        
        @Override
        public ConfigureFilters include(String loggerMaskMask, String messageMask) {
            super.include(loggerMaskMask, messageMask);
            return this;
        }
        
        @Override
        public ConfigureFilters exclude(String loggerMaskMask, String messageMask) {
            super.exclude(loggerMaskMask, messageMask);
            return this;
        }
        
        @Override
        public ConfigureFilters policy(ChainAction act) {
            super.policy(act);
            return this;
        }
        
        @Override
        public ConfigureFilters clear() {
            super.clear();
            return this;
        }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="ConfigureSources">
    public static class ConfigureSources
        extends LogHandlerPanel.ConfigureSources
    {
        protected final LogHandlerFrame frame;
        
        public ConfigureSources(LogHandlerFrame frame) {
            super(frame.logHandlerPanel);
            this.frame = frame;
        }
        
        @Override
        public Configure conf() {
            return new Configure(frame);
        }
        
        @Override
        public Configure listen(String... loggers) {
            super.listen(loggers);
            return new Configure(frame);
        }
        
        @Override
        public Configure listen(LoggerLevel lvl, String... loggers) {
            super.listen(lvl, loggers);
            return new Configure(frame);
        }
        
        @Override
        public ConfigureSources add(String loggerName, boolean start) {
            super.add(loggerName, start);
            return this;
        }
        
        @Override
        public ConfigureSources add(String loggerName, LoggerLevel lvl, boolean start) {
            super.add(loggerName, lvl, start);
            return this;
        }
        
        @Override
        public ConfigureSources add(String loggerName, LoggerLevel lvl) {
            super.add(loggerName, lvl);
            return this;
        }
        
        @Override
        public ConfigureSources add(String loggerName) {
            super.add(loggerName);
            return this;
        }
        
        @Override
        public ConfigureSources clear() {
            super.clear();
            return this;
        }
    }
    //</editor-fold>
}
