/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package xyz.cofe.gui.swing.log;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JTable;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.xml.stream.XMLStreamException;
import xyz.cofe.collection.Convertor;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.set.SyncEventSet;
import xyz.cofe.common.Reciver;
import xyz.cofe.gui.swing.AWTReciver;
import xyz.cofe.gui.swing.BasicAction;
import xyz.cofe.gui.swing.SwingListener;
import xyz.cofe.gui.swing.properties.PropertyValue;
import xyz.cofe.gui.swing.table.CachedTM;
import xyz.cofe.gui.swing.table.PropertyColumn;
import xyz.cofe.gui.swing.tree.TreeTableNodeFormat;
import xyz.cofe.gui.swing.tree.TreeTableNodeFormatBasic;
import xyz.cofe.gui.swing.tree.TreeTableNodeRender;
import xyz.cofe.gui.swing.tree.TreeTableNodeValueEditor;
import xyz.cofe.gui.swing.tree.TreeTableNodeValueEditorDef;

/**
 * Панель с таблицей логов
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class LogHandlerPanel extends javax.swing.JPanel {

    /**
     * Конструктор
     */
    public LogHandlerPanel() {
        initComponents();
        handlerSync = new Object();

        initPredefinedValues();
        initLimitFTextListeners();
        initLimitAndScroll();
        initLevelComboBox();        
        initClearButton();
        initSourcesTable();
        initFiltersTable();
        initConfMenu();
        
        logHandler.setFilter(filters);
    }

    //<editor-fold defaultstate="collapsed" desc="initClearButton()">
    private void initClearButton() {
        SwingListener.onActionPerformed(clearButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                clearMessages();
            }
        });
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="initLimitAndScroll()">
    private void initLimitAndScroll() {
        SwingListener.onActionPerformed(limitCheckBox, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                updateUnlimitedMessageCount();
            }
        });
        
        AWTReciver a = new AWTReciver<LogRecord>(){
            @Override
            public synchronized void recive(LogRecord obj) {
                if( !isUnlimitedMessageCount() ){
                    limitMessages();
                }
                if( isAutoScrollMessages() ){
                    scrollMessageToLast();
                }
            }
        };
        a.setAwtSync(false);
        logTable.getRecords().onAdded(a);
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="initPredefinedValues()">
    private void initPredefinedValues() {
        unlimitedMessageCount = false;
        limitMessageCount = 1000;
        uiRefreshLimitMessageCount();
        uiRefreshUnlimitedMessageCount();
        
        setAutoScrollMessages(true);
        
        logHandler.setLevel(Level.FINER);
        stoppedLevel = Level.FINER;
        uiRefreshLevel();
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="initLimitFTextListeners()">
    private void initLimitFTextListeners() {
        SwingListener.onFocusLost(limitFTextField, new Reciver<FocusEvent>() {
            @Override
            public void recive(FocusEvent obj) {
                updateLimitMessageCount();
            }
        });
        
        SwingListener.onKeyPressed(limitFTextField, new Reciver<KeyEvent>() {
            @Override
            public void recive(KeyEvent ke) {
                if( ke.getKeyCode()==KeyEvent.VK_ENTER ){
                    updateLimitMessageCount();
                }
            }
        });
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="ConfigureSources">
    /**
     * Конфигурация источников сообщений
     */
    public static class ConfigureSources {
        protected final LogHandlerPanel panel;
        
        public ConfigureSources( LogHandlerPanel panel ){
            if( panel==null )throw new IllegalArgumentException("panel == null");
            this.panel = panel;
        }

        //<editor-fold defaultstate="collapsed" desc="clear()">
        /**
         * Удаление всех источников сообщений
         * @return self ссылка
         */
        public ConfigureSources clear(){
            panel.getSources().clear();
            return this;
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="add()">
        /**
         * Добавляет логгера в качестве источника данных
         * @param loggerName логгер
         * @return self ссылка
         */
        public ConfigureSources add(String loggerName){
            if( loggerName==null )throw new IllegalArgumentException("loggerName == null");
            panel.getSources().add(new LogSource(loggerName, LoggerLevel.UNDEFINED));
            return this;
        }
        
        /**
         * Добавляет логгера в качестве источника данных
         * @param loggerName логгер
         * @param lvl уровенть логирования
         * @return self ссылка
         */
        public ConfigureSources add(String loggerName, LoggerLevel lvl){
            if( loggerName==null )throw new IllegalArgumentException("loggerName == null");
            panel.getSources().add(new LogSource(loggerName, lvl));
            return this;
        }
        
        /**
         * Добавляет логгера в качестве источника данных
         * @param loggerName имя логгера
         * @param lvl уровень логирования
         * @param start запустить прослушивание данных после добавления
         * @return self ссылка
         */
        public ConfigureSources add(String loggerName, LoggerLevel lvl, boolean start){
            if( loggerName==null )throw new IllegalArgumentException("loggerName == null");
            LogSource ls = new LogSource(loggerName, lvl);
            panel.getSources().add(ls);
            if( start )ls.start();
            return this;
        }
        
        /**
         * Добавляет логгера в качестве источника данных
         * @param loggerName имя логгера
         * @param start запустить прослушивание данных после добавления
         * @return self ссылка
         */
        public ConfigureSources add(String loggerName, boolean start){
            if( loggerName==null )throw new IllegalArgumentException("loggerName == null");
            LogSource ls = new LogSource(loggerName, LoggerLevel.UNDEFINED);
            panel.getSources().add(ls);
            if( start )ls.start();
            return this;
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="listen()">
        /**
         * Добавить указанный логеров и запустить прослушивание/получение данных
         * @param lvl уровень логирования
         * @param loggers имена логгеров
         * @return self ссылка
         */
        public Configure listen(LoggerLevel lvl, String ... loggers){
            //clear();
            for( String lgr : loggers ){
                add( lgr, lvl, true );
            }
            return new Configure(panel);
        }
        
        /**
         * Добавить указанный логеров и запустить прослушивание/получение данных
         * @param loggers имена логгеров
         * @return self ссылка
         */
        public Configure listen(String ... loggers){
            //clear();
            for( String lgr : loggers ){
                add( lgr, true );
            }
            return new Configure(panel);
        }
        //</editor-fold>
        
        /**
         * Конфигурация источников данных
         * @return конфигурация
         */
        public Configure conf(){
            return new Configure(panel);
        }
    }
    //</editor-fold>    
    //<editor-fold defaultstate="collapsed" desc="ConfigureFilters">
    /**
     * Конфигурация фильтров
     */
    public static class ConfigureFilters {
        protected final LogHandlerPanel panel;
        
        public ConfigureFilters( LogHandlerPanel panel ){
            if( panel==null )throw new IllegalArgumentException("panel == null");
            this.panel = panel;
        }

        /**
         * Удаление всех фильтров
         * @return self ссылка
         */
        public ConfigureFilters clear(){
            panel.getFilters().clear();
            return this;
        }
        
        /**
         * Указание поведения по умолчанию
         * @param act поведение по умолчанию
         * @return self ссылка
         */
        public ConfigureFilters policy( ChainAction act ){
            if( act==null )throw new IllegalArgumentException("act == null");
            panel.getFilters().setDefaultAction(act);
            return this;
        }
        
        /**
         * Добавление исключающего фильтра
         * @param loggerMaskMask маска (wildcard) имени логгера
         * @param messageMask маска (wildcard) текста сообщеия
         * @return self ссылка
         */
        public ConfigureFilters exclude(String loggerMaskMask, String messageMask){
            MaskFilter mf = new MaskFilter();
            mf.setAction(ChainAction.Exclude);
            mf.setEnabled(true);
            mf.setLoggerName(loggerMaskMask);
            mf.setMessage(messageMask);
            panel.getFilters().add(mf);
            return this;
        }
        
        /**
         * Добавление включающего фильтра
         * @param loggerMaskMask маска (wildcard) имени логгера
         * @param messageMask маска (wildcard) текста сообщеия
         * @return wildcard
         */
        public ConfigureFilters include(String loggerMaskMask, String messageMask){
            MaskFilter mf = new MaskFilter();
            mf.setAction(ChainAction.Include);
            mf.setEnabled(true);
            mf.setLoggerName(loggerMaskMask);
            mf.setMessage(messageMask);
            panel.getFilters().add(mf);
            return this;
        }
        
        /**
         * Конфигурация
         * @return конфигурация
         */
        public Configure conf(){
            return new Configure(panel);
        }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="Configure">
    /**
     * Конфигуоация
     */
    public static class Configure {
        protected final LogHandlerPanel panel;
        
        /**
         * Конструктор
         * @param panel панель обработчик
         */
        public Configure( LogHandlerPanel panel ){
            if( panel==null )throw new IllegalArgumentException("panel == null");
            this.panel = panel;
        }
        
        /**
         * Конфигурация источников данных
         * @return конфигурация
         */
        public ConfigureSources sources(){
            return new ConfigureSources(panel);
        }
        
        /**
         * Конфигурация фильтров
         * @return конфигурация
         */
        public ConfigureFilters filters(){
            return new ConfigureFilters(panel);
        }
        
        /**
         * Запуск прослушивания событий
         * @param level уровень обрабатываемых событий
         * @return self ссылка
         */
        public Configure start( LoggerLevel level ){
            if( level!=null ){
                panel.getLogHandler().setLevel( level.level() );
                panel.uiRefreshLevel();
            }
            return this;
        }
        
        /**
         * Запуск прослушивания событий
         * @return self ссылка
         */
        public Configure start(){
            panel.start();
            return this;
        }
        
        /**
         * Остановка полчения событий
         * @return self ссылка
         */
        public Configure stop(){
            return this;
        }
    }
    
    /**
     * Конфигурация
     * @return конфигурация
     */
    public Configure configure(){
        return new Configure(this);
    }
    //</editor-fold>
    
    protected final Object handlerSync;
    
    //<editor-fold defaultstate="collapsed" desc="unlimitedMessageCount">
    protected boolean unlimitedMessageCount = true;
    
    /**
     * Указывет, что не ограничивать кол-во хранимых сообщений
     * @return кол-во хранимых сообщений
     */
    public boolean isUnlimitedMessageCount(){
        synchronized( handlerSync ){
            return unlimitedMessageCount;
        }
    }
    
    /**
     * Указывет, что не ограничивать кол-во хранимых сообщений
     * @param unlim true - не ограничивать / false - ограничивать
     */
    public void setUnlimitedMessageCount( boolean unlim ){
        Object old,cur;
        synchronized( handlerSync ){
            old = unlimitedMessageCount;
            unlimitedMessageCount = unlim;
            cur = unlimitedMessageCount;
        }
        uiRefreshUnlimitedMessageCount();
        firePropertyChange("unlimitedMessageCount", old, cur);
    }
    
    /**
     * Обновление checkbox-а
     */
    protected void uiRefreshUnlimitedMessageCount(){
        final boolean unlim = isUnlimitedMessageCount();
        Runnable rn = new Runnable() {
            @Override
            public void run() {
                limitCheckBox.setSelected(!unlim);
            }
        };
        if( SwingUtilities.isEventDispatchThread() ){
            rn.run();
        }else{
            SwingUtilities.invokeLater(rn);
        }
    }
    
    /**
     * Обновляет свойство unlimitedMessageCount
     */
    protected void updateUnlimitedMessageCount(){
        boolean unlim = !limitCheckBox.isSelected();
        synchronized(handlerSync){
            unlimitedMessageCount = unlim;
        }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="limitMessageCount">
    protected int limitMessageCount = -1;
    
    /**
     * Указывает лимит кол-ва хранимых сообщений
     * @return лимит
     */
    public int getLimitMessageCount(){
        synchronized(handlerSync){
            return limitMessageCount;
        }
    }
    
    /**
     * Указывает лимит кол-ва хранимых сообщений
     * @param newLimit лимит
     */
    public void setLimitMessageCount( int newLimit ){
        Object old;
        Object cur;
        synchronized(handlerSync){
            old = limitMessageCount;
            limitMessageCount = newLimit;
            cur = limitMessageCount;
        }
        uiRefreshLimitMessageCount();
        firePropertyChange("limitMessageCount", old, cur);
    }
    
    protected Integer readLimitTextField(){
        String str = limitFTextField.getText();
        if( str==null || str.trim().length()<1 ){
            return null;
        }
        str = str.trim();
        if( !str.matches("^([\\-\\s]+)?\\d+") ){
            return null;
        }
        return Integer.parseInt(str);
    }
    
    protected void updateLimitMessageCount(){
        Integer v = readLimitTextField();
        if( v==0 )return;
        
        synchronized( handlerSync ){
            limitMessageCount = v;
        }
    }
    
    /**
     * Обновить поле limitFTextField. thread-safe.
     */
    protected void uiRefreshLimitMessageCount(){
        final int cnt = getLimitMessageCount();
        Runnable rn = new Runnable() {
            @Override
            public void run() {
                limitFTextField.setText(Integer.toString(cnt));
            }
        };
        if( SwingUtilities.isEventDispatchThread() ){
            rn.run();
        }else{
            SwingUtilities.invokeLater(rn);
        }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="autoScrollMessages">
    /**
     * Указывает автоскроллинг сообщений
     * @return true автоскроллинг установлен
     */
    public boolean isAutoScrollMessages(){ return scrollToggleButton.isSelected(); }
    /**
     * Указывает автоскроллинг сообщений
     * @param v true автоскроллинг установлен
     */
    public void setAutoScrollMessages(boolean v){ scrollToggleButton.setSelected(v); }
    
    private volatile long lastScrolled = System.currentTimeMillis();
    private volatile long scrollMessageToLastLast = System.currentTimeMillis();
    
    /**
     * Скролировать таблицу сообщений к последнему элементу.
     * <br>thread-safe
     */
    public void scrollMessageToLast(){
        final Runnable scroll = new Runnable() {
            @Override
            public void run() {
                int v = logScrollPane.getVerticalScrollBar().getValue();
                int max = logScrollPane.getVerticalScrollBar().getMaximum();
                if( v<max ){
                    logScrollPane.getVerticalScrollBar().setValue(max);
                }
                lastScrolled = System.currentTimeMillis();
            }
        };
        
        scrollMessageToLastLast = System.currentTimeMillis();
        
        Runnable run = new Runnable() {
            @Override
            public void run() {
                Timer timer = new Timer(50, new ActionListener() {
                    @Override
                    public void actionPerformed(ActionEvent e) {
                        if( scrollMessageToLastLast>lastScrolled )scroll.run();
                    }
                });
                timer.setRepeats(false);
                timer.setInitialDelay(50);
                timer.start();
            }
        };
        
        if( SwingUtilities.isEventDispatchThread() ){
            run.run();
        }else{
            SwingUtilities.invokeLater(run);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="clearMessages()">
    /**
     * Очистить накопившиеся сообщения.
     * <br>thread-safe
     */
    public void clearMessages(){
        Runnable run = new Runnable() {
            @Override
            public void run() {
                logTable.getRecords().clear();
            }
        };
        
        if( SwingUtilities.isEventDispatchThread() ){
            run.run();
        }else{
            SwingUtilities.invokeLater(run);
        }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="limitMessages()">
    /**
     * Ограничить кол-во накопившихся сообщений.
     * <br>thread-safe
     */
    public void limitMessages(){
        final int limitSize = getLimitMessageCount();
        
        Runnable run = new Runnable() {
            @Override
            public void run() {
                int size = logTable.getRecords().size();
                if( size>limitSize && limitSize>0 ){
                    int removeCount = size - limitSize;
                    for( int i=0; i<removeCount; i++ ){
                        logTable.getRecords().remove((int)0);
                    }
                }
            }
        };
        
        if( limitSize<1 )return;
        
        if( SwingUtilities.isEventDispatchThread() ){
            run.run();
        }else{
            SwingUtilities.invokeLater(run);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="logHandler">
    private final Handler logHandler = new Handler() {
        @Override
        public void publish(LogRecord record) {
            if( record==null )return;
            if( !this.isLoggable(record) )return;
            logTable.add(record);
        }
        
        @Override
        public void flush() {
        }
        
        @Override
        public void close() throws SecurityException {
        }
    };
    
    /**
     * Возвращает обрабочик лог записей.
     * Добавляет посупающие записи в таблицу.
     * @return обработчик лог записей
     */
    public Handler getLogHandler(){
        return logHandler;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="logLevel">
    protected void initLevelComboBox( JComboBox cbox ){
        cbox.setModel(
            new DefaultComboBoxModel<String>(
                new String[]{
                    Level.ALL.getName(),
                    Level.FINEST.getName(),
                    Level.FINER.getName(),
                    Level.FINE.getName(),
                    Level.CONFIG.getName(),
                    Level.INFO.getName(),
                    Level.WARNING.getName(),
                    Level.SEVERE.getName(),
                    Level.OFF.getName(),
                }));
    }
    
    protected void uiWriteLevel(JComboBox cbox,Level level){
        if( level==null ){
            cbox.setSelectedItem(Level.ALL.getName());
        }else{
            if( level.intValue() == Level.ALL.intValue() ){
                cbox.setSelectedItem(Level.ALL.getName());
            }else if( level.intValue() == Level.FINEST.intValue() ){
                cbox.setSelectedItem(Level.FINEST.getName());
            }else if( level.intValue() == Level.FINER.intValue() ){
                cbox.setSelectedItem(Level.FINER.getName());
            }else if( level.intValue() == Level.FINE.intValue() ){
                cbox.setSelectedItem(Level.FINE.getName());
            }else if( level.intValue() == Level.CONFIG.intValue() ){
                cbox.setSelectedItem(Level.CONFIG.getName());
            }else if( level.intValue() == Level.INFO.intValue() ){
                cbox.setSelectedItem(Level.INFO.getName());
            }else if( level.intValue() == Level.WARNING.intValue() ){
                cbox.setSelectedItem(Level.WARNING.getName());
            }else if( level.intValue() == Level.SEVERE.intValue() ){
                cbox.setSelectedItem(Level.SEVERE.getName());
            }else if( level.intValue() == Level.OFF.intValue() ){
                cbox.setSelectedItem(Level.OFF.getName());
            }
        }
    }
    
    protected Level readLevel(JComboBox cbox){
        Object sel = cbox.getSelectedItem();
        if( sel==null ){
            return null;
        }
        Level lvl = null;
        if( sel.toString().equalsIgnoreCase(Level.OFF.getName()) ){
            lvl = (Level.OFF);
        }else if( sel.toString().equalsIgnoreCase(Level.SEVERE.getName()) ){
            lvl = (Level.SEVERE);
        }else if( sel.toString().equalsIgnoreCase(Level.WARNING.getName()) ){
            lvl = (Level.WARNING);
        }else if( sel.toString().equalsIgnoreCase(Level.INFO.getName()) ){
            lvl = (Level.INFO);
        }else if( sel.toString().equalsIgnoreCase(Level.CONFIG.getName()) ){
            lvl = (Level.CONFIG);
        }else if( sel.toString().equalsIgnoreCase(Level.FINE.getName()) ){
            lvl = (Level.FINE);
        }else if( sel.toString().equalsIgnoreCase(Level.FINER.getName()) ){
            lvl = (Level.FINER);
        }else if( sel.toString().equalsIgnoreCase(Level.FINEST.getName()) ){
            lvl = (Level.FINEST);
        }else if( sel.toString().equalsIgnoreCase(Level.ALL.getName()) ){
            lvl = (Level.ALL);
        }
        return lvl;
    }
    
    protected void initLevelComboBox(){
        initLevelComboBox(levelComboBox);
        
        SwingListener.onActionPerformed(levelComboBox, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                updateLevel();
            }
        });
        levelComboBox.addItemListener(new ItemListener() {
            @Override
            public void itemStateChanged(ItemEvent e) {
                if( uiRefreshLevelCall )return;
                updateLevel();
            }
        });
        
        uiRefreshLevel();
        
        SwingListener.onActionPerformed(startButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                start();
            }
        });
        
        SwingListener.onActionPerformed(stopButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                stop();
            }
        });
    }

    /**
     * Запускает прослушивание поступающих лог записей
     */
    public void start() {
        if( stoppedLevel==null || stoppedLevel.intValue() == Level.OFF.intValue() ){
            getLogHandler().setLevel(Level.FINER);
        }else{
            getLogHandler().setLevel(stoppedLevel);
        }
        uiRefreshLevel();
    }

    /**
     * Запускает прослушивание поступающих лог записей
     * @param level уровень принимаемых записий
     */
    public void start(Level level) {
        if( level==null ){
            if( stoppedLevel==null || stoppedLevel.intValue() != Level.OFF.intValue() ){
                level = Level.FINER;
            }else{
                level = stoppedLevel;
            }
        }
        
        getLogHandler().setLevel(level);
        uiRefreshLevel();
    }
    
    /**
     * Останавливает прослушивание поступающих лог записей
     */
    public void stop() {
        Level lvl = getLogHandler().getLevel();
        if( lvl!=null && lvl.intValue() != Level.OFF.intValue() ){
            stoppedLevel = lvl;
            getLogHandler().setLevel(Level.OFF);
            uiRefreshLevel();
        }
    }
    
    private Level stoppedLevel = null;
    private boolean uiRefreshLevelCall = false;
    
    /**
     * Обновление ui уровня получаемых событий
     */
    public void uiRefreshLevel(){
        try{
            uiRefreshLevelCall = true;
            Level lvl = getLogHandler().getLevel();
            uiWriteLevel(levelComboBox, lvl);
        }finally{
            uiRefreshLevelCall = false;
        }
    }
    
    /**
     * Обновление уровня получаемых событий
     */
    public void updateLevel(){
        Level lvl = readLevel(levelComboBox);
        Object sel = levelComboBox.getSelectedItem();
        if( sel==null || lvl==null ){
            return;
        }
        
        Level clvl = getLogHandler().getLevel();
        if( clvl==null || clvl.intValue() != lvl.intValue() ){
            getLogHandler().setLevel(lvl);
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="components">
    public JToolBar getToolbar(){ return toolBar; }
    public JButton getClearButton(){ return clearButton; }
    public JToggleButton getScrollToggleButton(){ return scrollToggleButton; }
    public JCheckBox getLimitCheckBox(){ return limitCheckBox; }
    public xyz.cofe.gui.swing.text.ValidatedTextField getLimitTextField(){ return limitFTextField; }
    public LogTable getLogTable(){ return logTable; }
    public JTable getFiltersTable(){ return filtersTable; }
    public JTable getSourcesTable(){ return sourcesTable; }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="render / editor">
    //<editor-fold defaultstate="collapsed" desc="propertyRender">
    protected volatile TreeTableNodeRender propertyRender;
    protected TreeTableNodeRender getPropertyRender(){
        if( propertyRender!=null )return propertyRender;
        synchronized(this){
            if( propertyRender!=null )return propertyRender;
            propertyRender = new TreeTableNodeRender();
            return propertyRender;
        }
    }
    //</editor-fold>    
    //<editor-fold defaultstate="collapsed" desc="propertyEditor">
    protected volatile TreeTableNodeValueEditor propertyEditor;
    protected TreeTableNodeValueEditor getPropertyEditor(){
        if( propertyEditor!=null )return propertyEditor;
        synchronized(this){
            if( propertyEditor!=null )return propertyEditor;
            propertyEditor = new TreeTableNodeValueEditorDef();
            return propertyEditor;
        }
    }
    //</editor-fold>
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="sources">
    protected volatile SyncEventSet<LogSource> sources;
    /**
     * Указывает источники данных
     * @return источники данных
     */
    public SyncEventSet<LogSource> getSources(){
        if( sources!=null )return sources;
        synchronized(this){
            if( sources!=null )return sources;
            sources = new SyncEventSet<>(handlerSync);
            sources.onAdded(new Reciver<LogSource>() {
                @Override
                public void recive(LogSource ls) {
                    if( ls!=null ){
                        ls.setHandler(getLogHandler());
                    }
                }
            });
            sources.onRemoved(new Reciver<LogSource>() {
                @Override
                public void recive(LogSource ls) {
                    if( ls!=null ){
                        synchronized(ls){
                            if( ls.isListen() ){
                                ls.stop();
                            }
                            ls.setHandler(null);
                        }
                    }
                }
            });
            return sources;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="rootLoggerFormat">
    protected TreeTableNodeFormat rootLoggerFormat
        = new TreeTableNodeFormatBasic()
            .convertor(new Convertor<Object, String>() {
                @Override
                public String convert(Object from) {
                    if( from instanceof String ){
                        if( ((String)from).equals("") )return "root";
                        return from.toString();
                    }
                    return null;
                }
            }).foreground(Color.gray)
            .italic(true);
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="sourcesCacheTM">
    protected CachedTM<LogSource> sourcesCacheTM;
    
    /**
     * Возвращает модель таблицы источников данных
     * @return модель таблицы
     */
    public CachedTM<LogSource> getSourcesCacheTM(){
        if(sourcesCacheTM!=null)return sourcesCacheTM;
        sourcesCacheTM = new CachedTM<>();
        
        PropertyColumn lgrName = new PropertyColumn("loggerName");
        
        lgrName.addValueFormat(String.class, new Func1<TreeTableNodeFormat, String>() {
            @Override
            public TreeTableNodeFormat apply(String lgrName) {
                if( lgrName==null )return null;
                if( lgrName instanceof String ){
                    if( ((String)lgrName).equalsIgnoreCase("") ){
                        return rootLoggerFormat;
                    }
                }
                return null;
            }
        });
        sourcesCacheTM.getColumns().add(lgrName);
        
        PropertyColumn lgrLevel = new PropertyColumn("loggerLevel");
        sourcesCacheTM.getColumns().add(lgrLevel);
        
        PropertyColumn lgrListen = new PropertyColumn("listen");
        sourcesCacheTM.getColumns().add(lgrListen);
        
        sourcesCacheTM.setSource(getSources());
        sourcesCacheTM.fetch();
        
        return sourcesCacheTM;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="initSourcesTable()">
    /**
     * Инициализация таблицы источников данных
     */
    public void initSourcesTable(){
        sourcesTable.setDefaultRenderer(PropertyValue.class, getPropertyRender());
        sourcesTable.setDefaultEditor(PropertyValue.class, getPropertyEditor());
        sourcesTable.setModel(getSourcesCacheTM());
        //sourcesCacheTM.setSource(sources);
        
        initLevelComboBox(addSourceLevelComboBox);
        SwingListener.onActionPerformed(addSourceButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                addSource();
            }
        });
        SwingListener.onActionPerformed(deleteSourcesButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                deleteSources();
            }
        });
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="addSource()">
    /**
     * Добавляет пользовательский источник данных
     */
    protected void addSource(){
        String loggerName = addSourceTextField.getText();
        Level lvl = readLevel(addSourceLevelComboBox);
        
        if( loggerName==null )return;
        LoggerLevel llvl = LoggerLevel.level(lvl);
        
        getSources().add(new LogSource(loggerName, llvl));
        //getSourcesCacheTM().refresh();
    }
    
    /**
     * Добавляет пользовательский источник данных
     * @param loggerName имя логгера
     * @param level уровень логирования
     * @param start true - запустить получение данных
     * @return self ссылка
     */
    public LogHandlerPanel addSource( String loggerName, LoggerLevel level, boolean start ){
        if( loggerName==null )throw new IllegalArgumentException("loggerName");
        
        LogSource ls = new LogSource(loggerName, level);
        ls.setHandler(getLogHandler());
        
        getSources().add(ls);
        
        if( start ){
            ls.start();
        }
        
        return this;
    }
    
    /**
     * Добавляет пользовательский источник данных
     * @param loggerName имя логгера
     * @return self ссылка
     */
    public LogHandlerPanel addSource(String loggerName){
        if( loggerName==null )throw new IllegalArgumentException("loggerName");
        return addSource(loggerName, LoggerLevel.UNDEFINED, false);
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="deleteSources()">
    /**
     * Удаляет выбранные пользователем источники данных
     */
    protected void deleteSources(){
        for( int selrow : sourcesTable.getSelectedRows() ){
            Object itm = getSourcesCacheTM().getItemByIndex(selrow);
            if( itm instanceof LogSource ){
                getSources().remove((LogSource)itm);
            }
        }
        getSourcesCacheTM().fetch();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="filters">
    protected final ChainFilters filters = new ChainFilters().defaultAction(ChainAction.Include).clear();
    /**
     * Возвращает цепочку фльтров
     * @return цепь фильтров
     */
    public ChainFilters getFilters(){
        return filters;
    }
    private Iterable<ChainElement> filtersElements = new Iterable<ChainElement>() {
        @Override
        public Iterator<ChainElement> iterator() {
            final ChainElement[] arr = filters.array();
            
            Iterator<ChainElement> iter = new Iterator<ChainElement>() {
                int ptr = 0;
                
                @Override
                public boolean hasNext() {
                    return ptr<arr.length;
                }
                
                @Override
                public ChainElement next() {
                    if( ptr<arr.length ){
                        ChainElement e = arr[ptr];
                        ptr++;
                        return e;
                    }
                    return null;
                }
                
                @Override
                public void remove() {
                    throw new UnsupportedOperationException("Not supported yet.");
                }
            };
            
            return iter;
        }
    };
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="filtersCacheTM">
    private CachedTM<ChainElement> filtersCacheTM;
    /**
     * Возвращает модель фильтров
     * @return модель таблицы фильтров
     */
    protected CachedTM<ChainElement> getFiltersCacheTM(){
        if( filtersCacheTM!=null )return filtersCacheTM;
        filtersCacheTM = new CachedTM<>();
        filtersCacheTM.setSource(filtersElements);
        
        PropertyColumn lgrNameCol = new PropertyColumn("loggerName");
        filtersCacheTM.getColumns().add(lgrNameCol);
        
        PropertyColumn messageCol = new PropertyColumn("message");
        filtersCacheTM.getColumns().add(messageCol);
        
        PropertyColumn actionCol = new PropertyColumn("action");
        filtersCacheTM.getColumns().add(actionCol);
        
        PropertyColumn enabledCol = new PropertyColumn("enabled");
        filtersCacheTM.getColumns().add(enabledCol);
        
        return filtersCacheTM;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="initFiltersTable()">
    /**
     * Инициализация таблицы фильтров
     */
    protected void initFiltersTable(){
        filtersTable.setDefaultEditor(PropertyValue.class, getPropertyEditor());
        filtersTable.setDefaultRenderer(PropertyValue.class, getPropertyRender());
        filtersTable.setModel(getFiltersCacheTM());
        
        if( ChainAction.Include.equals(filters.getDefaultAction()) ){
            defIncActionRButton.setSelected(true);
        }
        if( ChainAction.Exclude.equals(filters.getDefaultAction()) ){
            defExcActionRButton.setSelected(true);
        }
        
        final Runnable updatePolicy = new Runnable(){
            @Override
            public void run() {
                if( defIncActionRButton.isSelected() ){
                    filters.setDefaultAction(ChainAction.Include);
                }
                if( defExcActionRButton.isSelected() ){
                    filters.setDefaultAction(ChainAction.Exclude);
                }
            }
        };
        
        Reciver<ActionEvent> rUpdatePolicy = new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                updatePolicy.run();
            }
        };
        
        SwingListener.onActionPerformed(defIncActionRButton, rUpdatePolicy);
        SwingListener.onActionPerformed(defExcActionRButton, rUpdatePolicy);
        
        SwingListener.onActionPerformed(addFilterButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                addFilter();
            }
        });
        
        SwingListener.onActionPerformed(removeFilterButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                removeSelectedFilters();
            }
        });
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="addFilter()">
    /**
     * Добавляет пользовательский фильтр
     */
    protected void addFilter(){
        String lgr = loggerNameTField.getText();
        String msg = messageTField.getText();
        
        ChainAction act = ChainAction.Exclude;
        if( includeRButton.isSelected() )act = ChainAction.Include;
        if( excludeRButton.isSelected() )act = ChainAction.Exclude;
        
        MaskFilter mf = new MaskFilter();
        mf.setAction(act);
        mf.setLoggerName(lgr);
        mf.setMessage(msg);
        
        filters.add(mf);
        getFiltersCacheTM().refresh();
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="removeSelectedFilters()">
    /**
     * Удаляет выбранные пользователем фильтры
     */
    protected void removeSelectedFilters(){
        for( int selrow : filtersTable.getSelectedRows() ){
            Object selitm = getFiltersCacheTM().getItemByIndex(selrow);
            if( selitm instanceof ChainElement ){
                filters.remove((ChainElement)selitm);
            }
        }
        getFiltersCacheTM().refresh();
    }
    //</editor-fold>
    
    protected XmlConfig xmlConfig;
    /**
     * Возвращает xml конфигурацию
     * @return xml конфигурация
     */
    public XmlConfig getXmlConfig(){
        synchronized(this){
            if( xmlConfig!=null )return xmlConfig;
            xmlConfig = new XmlConfig();
            return xmlConfig;
        }
    }
    
    /**
     * Указывает xml конфигурацию
     * @param xc xml конфигурация
     */
    public void setXmlConfig( XmlConfig xc ){
        synchronized(this){
            xmlConfig = xc;
        }
    }
    
    protected JPopupMenu confMenu;
    
    private void initConfMenu(){
        confMenu = new JPopupMenu();
        confMenu.add(new BasicAction("export/import xml settings", new Runnable() {
            @Override
            public void run() {
                exportImportXmlConfig();
            }
        }));
        confMenu.add(new JSeparator());
        confMenu.add(new BasicAction("save xml settings", new Runnable() {
            @Override
            public void run() {
                saveSettings();
            }
        }));
        confMenu.add(new BasicAction("save xml settings as", new Runnable() {
            @Override
            public void run() {
                saveSettingsAs();
            }
        }));        
        confMenu.add(new BasicAction("open xml settiongs", new Runnable() {
            @Override
            public void run() {
                openSettings();
            }
        }));
        
        SwingListener.onActionPerformed(confButton, new Reciver<ActionEvent>() {
            @Override
            public void recive(ActionEvent obj) {
                confMenu.show(confButton, confButton.getWidth()/2, confButton.getHeight()/2);
            }
        });
    }
    
    /**
     * Отоюражает окно настроек импорта/экспорта xml конфигурации
     */
    public void exportImportXmlConfig(){
        XmlConfigFrame xcframe = new XmlConfigFrame();
        xcframe.setXmlConfig(getXmlConfig());
        xcframe.setLogHandlerPanel(this);
        xcframe.setLocationRelativeTo(this);
        xcframe.setVisible(true);
    }
    
    private File settingsFile = null;
    
    /**
     * Загружает настройки из xml файла
     */
    public void openSettings(){
        JFileChooser fc = new JFileChooser();
        File curf = settingsFile;
        File curd = curf!=null ? curf.getParentFile() : null;
        if( curd!=null ){
            fc.setCurrentDirectory(curd);
        }
        fc.setFileSelectionMode(JFileChooser.FILES_ONLY);
        fc.setDialogTitle("load settings");
        fc.setMultiSelectionEnabled(false);
        int r = fc.showOpenDialog(this);
        if( r == JFileChooser.APPROVE_OPTION ){
            openSettings(fc.getSelectedFile());
        }
    }
    
    /**
     * Загружает настройки из xml файла
     * @param file файл xml
     */
    public void openSettings(File file){
        if( file==null )throw new IllegalArgumentException("file == null");
        XmlConfig cfg = getXmlConfig();
        try {
            cfg.read(this, file);
        } catch (XMLStreamException | IOException err) {
            JOptionPane.showMessageDialog(
                this, err.getMessage(), err.getClass().getSimpleName(), JOptionPane.ERROR_MESSAGE);
        }
    }
    
    /**
     * Сохраняет настройки в xml файл
     */
    public void saveSettings(){
        if( settingsFile!=null ){
            saveSettings(settingsFile);            
        }else{
            saveSettingsAs();
        }
    }
    
    /**
     * Сохраняет настройки в xml файл
     */
    public void saveSettingsAs(){
        JFileChooser fc = new JFileChooser();
        File curf = settingsFile;
        File curd = curf!=null ? curf.getParentFile() : null;
        if( curd!=null ){
            fc.setCurrentDirectory(curd);
        }
        fc.setFileSelectionMode(JFileChooser.FILES_ONLY);
        fc.setMultiSelectionEnabled(false);
        fc.setDialogTitle("save settings");
        int r = fc.showSaveDialog(this);
        if( r==JFileChooser.APPROVE_OPTION ){
            saveSettings(fc.getSelectedFile());
        }
    }
    
    /**
     * Сохраняет настройки в xml файл
     * @param file xml файл
     */
    public void saveSettings(File file){
        if( file==null )throw new IllegalArgumentException("file==null");
        try{
            XmlConfig cfg = getXmlConfig();
            cfg.write(file, this);
            settingsFile = file;
        }catch( Throwable err ){
            JOptionPane.showMessageDialog(
                this, err.getMessage(), err.getClass().getSimpleName(), JOptionPane.ERROR_MESSAGE);
        }
    }
    
    /**
     * This method is called from within the constructor to initialize the form. WARNING: Do NOT modify this code. The
     * content of this method is always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        addFilterButtonGroup = new javax.swing.ButtonGroup();
        defFilterButtonGroup = new javax.swing.ButtonGroup();
        toolBar = new javax.swing.JToolBar();
        clearButton = new javax.swing.JButton();
        scrollSeparator = new javax.swing.JToolBar.Separator();
        scrollToggleButton = new javax.swing.JToggleButton();
        jPanel1 = new javax.swing.JPanel();
        limitCheckBox = new javax.swing.JCheckBox();
        limitFTextField = new xyz.cofe.gui.swing.text.ValidatedTextField();
        jSeparator1 = new javax.swing.JToolBar.Separator();
        jPanel2 = new javax.swing.JPanel();
        startButton = new javax.swing.JButton();
        stopButton = new javax.swing.JButton();
        levelComboBox = new javax.swing.JComboBox<>();
        jSeparator2 = new javax.swing.JToolBar.Separator();
        confButton = new javax.swing.JButton();
        jTabbedPane1 = new javax.swing.JTabbedPane();
        logScrollPane = new javax.swing.JScrollPane();
        logTable = new xyz.cofe.gui.swing.log.LogTable();
        filtersPanel = new javax.swing.JPanel();
        jScrollPane2 = new javax.swing.JScrollPane();
        filtersTable = new javax.swing.JTable();
        editFilterPanel = new javax.swing.JPanel();
        loggerNameTField = new xyz.cofe.gui.swing.text.ValidatedTextField();
        messageTField = new xyz.cofe.gui.swing.text.ValidatedTextField();
        addFilterButton = new javax.swing.JButton();
        removeFilterButton = new javax.swing.JButton();
        includeRButton = new javax.swing.JRadioButton();
        excludeRButton = new javax.swing.JRadioButton();
        jLabel1 = new javax.swing.JLabel();
        defIncActionRButton = new javax.swing.JRadioButton();
        defExcActionRButton = new javax.swing.JRadioButton();
        sourcesPanel = new javax.swing.JPanel();
        jScrollPane1 = new javax.swing.JScrollPane();
        sourcesTable = new javax.swing.JTable();
        jPanel4 = new javax.swing.JPanel();
        addSourceLevelComboBox = new javax.swing.JComboBox<>();
        addSourceButton = new javax.swing.JButton();
        deleteSourcesButton = new javax.swing.JButton();
        addSourceTextField = new xyz.cofe.gui.swing.text.ValidatedTextField();

        setLayout(new java.awt.BorderLayout());

        toolBar.setFloatable(false);
        toolBar.setRollover(true);

        clearButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/xyz/cofe/gui/swing/ico/trash/user-trash-24.png"))); // NOI18N
        clearButton.setText("clear");
        clearButton.setToolTipText("clear messages");
        clearButton.setFocusable(false);
        clearButton.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        clearButton.setVerticalTextPosition(javax.swing.SwingConstants.BOTTOM);
        toolBar.add(clearButton);
        toolBar.add(scrollSeparator);

        scrollToggleButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/xyz/cofe/gui/swing/ico/autoscroll/autoscroll-24.png"))); // NOI18N
        scrollToggleButton.setText("scroll");
        scrollToggleButton.setToolTipText("auto scroll");
        scrollToggleButton.setFocusable(false);
        scrollToggleButton.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        scrollToggleButton.setVerticalTextPosition(javax.swing.SwingConstants.BOTTOM);
        toolBar.add(scrollToggleButton);

        jPanel1.setLayout(new java.awt.BorderLayout());

        limitCheckBox.setText("limit");
        jPanel1.add(limitCheckBox, java.awt.BorderLayout.PAGE_START);

        limitFTextField.setPlaceholder("1000");
        limitFTextField.setPreferredSize(new java.awt.Dimension(75, 27));
        jPanel1.add(limitFTextField, java.awt.BorderLayout.CENTER);

        toolBar.add(jPanel1);
        toolBar.add(jSeparator1);

        jPanel2.setLayout(new java.awt.GridBagLayout());

        startButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/xyz/cofe/gui/swing/ico/start/start-icon-16x16.png"))); // NOI18N
        startButton.setToolTipText("start");
        startButton.setBorder(javax.swing.BorderFactory.createEmptyBorder(1, 1, 1, 1));
        startButton.setBorderPainted(false);
        startButton.setContentAreaFilled(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        jPanel2.add(startButton, gridBagConstraints);

        stopButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/xyz/cofe/gui/swing/ico/stop/stop-icon-16.png"))); // NOI18N
        stopButton.setToolTipText("stop");
        stopButton.setBorder(javax.swing.BorderFactory.createEmptyBorder(1, 1, 1, 1));
        stopButton.setContentAreaFilled(false);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        jPanel2.add(stopButton, gridBagConstraints);

        levelComboBox.setModel(new javax.swing.DefaultComboBoxModel<>(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        jPanel2.add(levelComboBox, gridBagConstraints);

        toolBar.add(jPanel2);
        toolBar.add(jSeparator2);

        confButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/xyz/cofe/gui/swing/ico/settings/settings-32.png"))); // NOI18N
        confButton.setToolTipText("settings");
        confButton.setFocusable(false);
        confButton.setHorizontalTextPosition(javax.swing.SwingConstants.CENTER);
        confButton.setVerticalTextPosition(javax.swing.SwingConstants.BOTTOM);
        toolBar.add(confButton);

        add(toolBar, java.awt.BorderLayout.PAGE_START);

        jTabbedPane1.setTabPlacement(javax.swing.JTabbedPane.BOTTOM);

        logScrollPane.setViewportView(logTable);

        jTabbedPane1.addTab("messages", logScrollPane);

        filtersPanel.setLayout(new java.awt.BorderLayout());

        filtersTable.setModel(new javax.swing.table.DefaultTableModel(
            new Object [][] {
                {null, null, null, null},
                {null, null, null, null},
                {null, null, null, null},
                {null, null, null, null}
            },
            new String [] {
                "Title 1", "Title 2", "Title 3", "Title 4"
            }
        ));
        jScrollPane2.setViewportView(filtersTable);

        filtersPanel.add(jScrollPane2, java.awt.BorderLayout.CENTER);

        editFilterPanel.setLayout(new java.awt.GridBagLayout());

        loggerNameTField.setPlaceholder("logger name");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        editFilterPanel.add(loggerNameTField, gridBagConstraints);

        messageTField.setPlaceholder("message text");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        editFilterPanel.add(messageTField, gridBagConstraints);

        addFilterButton.setText("Add");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 10;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        editFilterPanel.add(addFilterButton, gridBagConstraints);

        removeFilterButton.setText("Del");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 11;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(0, 15, 0, 0);
        editFilterPanel.add(removeFilterButton, gridBagConstraints);

        addFilterButtonGroup.add(includeRButton);
        includeRButton.setText("include");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        editFilterPanel.add(includeRButton, gridBagConstraints);

        addFilterButtonGroup.add(excludeRButton);
        excludeRButton.setSelected(true);
        excludeRButton.setText("exclude");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 1;
        editFilterPanel.add(excludeRButton, gridBagConstraints);

        jLabel1.setText("Default policy");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 3);
        editFilterPanel.add(jLabel1, gridBagConstraints);

        defFilterButtonGroup.add(defIncActionRButton);
        defIncActionRButton.setSelected(true);
        defIncActionRButton.setText("Include");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        editFilterPanel.add(defIncActionRButton, gridBagConstraints);

        defFilterButtonGroup.add(defExcActionRButton);
        defExcActionRButton.setText("Exclude");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        editFilterPanel.add(defExcActionRButton, gridBagConstraints);

        filtersPanel.add(editFilterPanel, java.awt.BorderLayout.PAGE_END);

        jTabbedPane1.addTab("filters", filtersPanel);

        sourcesPanel.setLayout(new java.awt.BorderLayout());

        sourcesTable.setModel(new javax.swing.table.DefaultTableModel(
            new Object [][] {
                {null, null, null, null},
                {null, null, null, null},
                {null, null, null, null},
                {null, null, null, null}
            },
            new String [] {
                "Title 1", "Title 2", "Title 3", "Title 4"
            }
        ));
        jScrollPane1.setViewportView(sourcesTable);

        sourcesPanel.add(jScrollPane1, java.awt.BorderLayout.CENTER);

        jPanel4.setLayout(new java.awt.GridBagLayout());

        addSourceLevelComboBox.setModel(new javax.swing.DefaultComboBoxModel<>(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        jPanel4.add(addSourceLevelComboBox, gridBagConstraints);

        addSourceButton.setText("add");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        jPanel4.add(addSourceButton, gridBagConstraints);

        deleteSourcesButton.setText("delete");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 10;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(0, 15, 0, 0);
        jPanel4.add(deleteSourcesButton, gridBagConstraints);

        addSourceTextField.setPlaceholder("logger name");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        jPanel4.add(addSourceTextField, gridBagConstraints);

        sourcesPanel.add(jPanel4, java.awt.BorderLayout.PAGE_END);

        jTabbedPane1.addTab("sources", sourcesPanel);

        add(jTabbedPane1, java.awt.BorderLayout.CENTER);
    }// </editor-fold>//GEN-END:initComponents


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton addFilterButton;
    private javax.swing.ButtonGroup addFilterButtonGroup;
    private javax.swing.JButton addSourceButton;
    private javax.swing.JComboBox<String> addSourceLevelComboBox;
    private xyz.cofe.gui.swing.text.ValidatedTextField addSourceTextField;
    private javax.swing.JButton clearButton;
    private javax.swing.JButton confButton;
    private javax.swing.JRadioButton defExcActionRButton;
    private javax.swing.ButtonGroup defFilterButtonGroup;
    private javax.swing.JRadioButton defIncActionRButton;
    private javax.swing.JButton deleteSourcesButton;
    private javax.swing.JPanel editFilterPanel;
    private javax.swing.JRadioButton excludeRButton;
    private javax.swing.JPanel filtersPanel;
    private javax.swing.JTable filtersTable;
    private javax.swing.JRadioButton includeRButton;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel4;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JScrollPane jScrollPane2;
    private javax.swing.JToolBar.Separator jSeparator1;
    private javax.swing.JToolBar.Separator jSeparator2;
    private javax.swing.JTabbedPane jTabbedPane1;
    private javax.swing.JComboBox<String> levelComboBox;
    private javax.swing.JCheckBox limitCheckBox;
    private xyz.cofe.gui.swing.text.ValidatedTextField limitFTextField;
    private javax.swing.JScrollPane logScrollPane;
    private xyz.cofe.gui.swing.log.LogTable logTable;
    private xyz.cofe.gui.swing.text.ValidatedTextField loggerNameTField;
    private xyz.cofe.gui.swing.text.ValidatedTextField messageTField;
    private javax.swing.JButton removeFilterButton;
    private javax.swing.JToolBar.Separator scrollSeparator;
    private javax.swing.JToggleButton scrollToggleButton;
    private javax.swing.JPanel sourcesPanel;
    private javax.swing.JTable sourcesTable;
    private javax.swing.JButton startButton;
    private javax.swing.JButton stopButton;
    private javax.swing.JToolBar toolBar;
    // End of variables declaration//GEN-END:variables
}
