/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.log;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import xyz.cofe.collection.Convertor;
import xyz.cofe.collection.Func1;
import xyz.cofe.gui.swing.color.ColorModificator;
import xyz.cofe.gui.swing.table.Column;
import xyz.cofe.gui.swing.tree.FormattedValue;
import xyz.cofe.gui.swing.tree.TreeTableNodeFormatBasic;

/**
 * Описывает колонки таблицы логов
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class LogRecordColumn extends Column
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(LogRecordColumn.class.getName());

    private static Level logLevel(){ return logger.getLevel(); }
    
    private static boolean isLogSevere(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null ? true : logLevel.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level logLevel = logger.getLevel(); 
        return logLevel==null  ? true : logLevel.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINER.intValue();
    }
    
    private static boolean isLogFinest(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINEST.intValue();
    }

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(LogRecordColumn.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(LogRecordColumn.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(LogRecordColumn.class.getName(), method, result);
    }
    //</editor-fold>
    
    /**
     * Конструктор
     */
    public LogRecordColumn(){
    }
    
    /**
     * Конструктор копирования
     * @param column образец для копирования
     */
    public LogRecordColumn(LogRecordColumn column){
        super(column);
    }

    /**
     * Конструктор копирования
     * @param sync объект для синхронизации
     * @param src образец для копирования
     */
    public LogRecordColumn(Object sync, LogRecordColumn src) {
        super(sync, src);
    }
    
    //<editor-fold defaultstate="collapsed" desc="redef return type">
    @Override
    public LogRecordColumn type(Class type) {
        super.type(type);
        return this;
    }
    
    @Override
    public LogRecordColumn writer(Convertor<Cell, Boolean> writer) {
        super.writer(writer);
        return this;
    }
    
    @Override
    public LogRecordColumn rowReader(Func1<Convertor<Object, Object>, Integer> rowReader) {
        super.rowReader(rowReader);
        return this;
    }
    
    @Override
    public LogRecordColumn reader(Convertor<Object, Object> reader) {
        super.reader(reader);
        return this;
    }
    
    @Override
    public LogRecordColumn name(String name) {
        super.name(name);
        return this;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="clone">
    @Override
    public LogRecordColumn clone(){
        return new LogRecordColumn(this);
    }
    
    @Override
    public LogRecordColumn cloneWith(Object sync) {
        return new LogRecordColumn(sync, this);
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="formatMessage(lr)">
    /**
     * Форматирование текстогово сообщения
     * @param lr событие
     * @return текстовое сообщение
     */
    public static String formatMessage( LogRecord lr ){
        if( lr==null )return null;
        
        String msg = lr.getMessage();
        if( msg==null )return null;
        
        Object[] params = lr.getParameters();
        if( params==null || params.length<1 )return msg;
        
        for( int pi=0; pi<params.length; pi++ ){
            Object param = params[pi];
            String str = param == null ? "null" : param.toString();
            String fnd = "{"+pi+"}";
            msg = msg.replace(fnd, str);
        }
        
        return msg;
    }
    //</editor-fold>
    
    private static final float bgBright = 0.0f;
    private static final float bgSat = 0.5f;
    
    private static final TreeTableNodeFormatBasic severeFmt =
        new TreeTableNodeFormatBasic().base(
            new ColorModificator()
                .brighter(bgBright)
                .hue(0f/360f)
                .saturation(bgSat)
        );

    private static final TreeTableNodeFormatBasic warningFmt =
        new TreeTableNodeFormatBasic().base(
            new ColorModificator()
                .brighter(bgBright)
                .hue(30f/360f)
                .sate(bgSat)
        );

    private static final TreeTableNodeFormatBasic infoFmt =
        new TreeTableNodeFormatBasic().base(
            new ColorModificator()
                .brighter(bgBright)
                .hue(90f/360f)
                .sate(bgSat)
        );

    private static final TreeTableNodeFormatBasic fineFmt =
        new TreeTableNodeFormatBasic().base(
            new ColorModificator()
                .brighter(bgBright)
                .hue(200f/360f)
                .sate(bgSat)
        );

    private static final TreeTableNodeFormatBasic finerFmt =
        new TreeTableNodeFormatBasic().base(
            new ColorModificator()
                .brighter(bgBright)
                .hue(225f/360f)
                .sate(bgSat)
        );

    private static final TreeTableNodeFormatBasic finestFmt =
        new TreeTableNodeFormatBasic().base(
            new ColorModificator()
                .brighter(bgBright)
                .hue(240f/360f)
                .sate(bgSat)
        );

    //<editor-fold defaultstate="collapsed" desc="prebuild columns">
    //<editor-fold defaultstate="collapsed" desc="levelNameColumn">
    /**
     * Колонка <b>levelName</b>, тип String
     */
    public static final LogRecordColumn levelNameColumn = new LogRecordColumn()
        .name("levelName")
        .type(String.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    Level lvl = ((LogRecord)from).getLevel();
                    if( lvl!=null ){
                        TreeTableNodeFormatBasic fmt = null;
                        int ilvl = lvl.intValue();
                        if( ilvl>=Level.SEVERE.intValue() ){
                            fmt = severeFmt;
                        }else if( ilvl>=Level.WARNING.intValue() ){
                            fmt = warningFmt;
                        }else if( ilvl>=Level.INFO.intValue() ){
                            fmt = infoFmt;
                        }else if( ilvl>=Level.FINE.intValue() ){
                            fmt = fineFmt;
                        }else if( ilvl<Level.FINE.intValue() ){
                            fmt = finerFmt;
                        }
                        
                        FormattedValue fv = new FormattedValue(lvl.getName(),fmt);
                        return fv;
                        
                        //return lvl.getName();
                    }
                }
                return null;
            }
        });
    //</editor-fold>    
    //<editor-fold defaultstate="collapsed" desc="levelValueColumn">
    /**
     * Колонка <b>levelValue</b>, тип Integer
     */
    public static final LogRecordColumn levelValueColumn = new LogRecordColumn()
        .name("levelValue")
        .type(Integer.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    Level lvl = ((LogRecord)from).getLevel();
                    if( lvl!=null )return lvl.intValue();
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="loggerNameColumn">
    /**
     * Колонка <b>loggerName</b>, тип String
     */
    public static final LogRecordColumn loggerNameColumn = new LogRecordColumn()
        .name("loggerName")
        .type(String.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    return ((LogRecord)from).getLoggerName();
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="rawMessageColumn">
    /**
     * Колонка <b>rawMessage</b>, тип String
     */
    public static final LogRecordColumn rawMessageColumn = new LogRecordColumn()
        .name("rawMessage")
        .type(String.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    return ((LogRecord)from).getMessage();
                }
                return null;
            }
        });
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="messageColumn">
    /**
     * Колонка <b>message</b>, тип String
     */
    public static final LogRecordColumn messageColumn = new LogRecordColumn()
        .name("message")
        .type(String.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    return formatMessage(lr);
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="millisColumn">
    /**
     * Колонка <b>millis</b>, тип Long
     */
    public static final LogRecordColumn millisColumn  = new LogRecordColumn()
        .name("millis")
        .type(Long.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    return lr.getMillis();
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="dateColumn()">
    /**
     * Колонка даты, тип Date
     * @param name имя колонки
     * @param fmt Формат времени
     * @return Колонка
     */
    public static LogRecordColumn dateColumn(String name, final DateFormat fmt){
        if( name==null )throw new IllegalArgumentException("name==null");
        if( fmt==null )throw new IllegalArgumentException("fmt==null");
        return new LogRecordColumn().name(name).type(String.class).reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    long t = lr.getMillis();
                    Date date = new Date(t);
                    return fmt.format(date);
                }
                return null;
            }
        });
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="dateColumn">
    /**
     * Колонка <b>date</b>, тип Date, формат yyyy-MM-dd
     */
    public static LogRecordColumn dateColumn = dateColumn("date", new SimpleDateFormat("yyyy-MM-dd"));
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="timeColumn">
    /**
     * Колонка <b>time</b>, тип Date, формат HH:mm:ss
     */
    public static LogRecordColumn timeColumn = dateColumn("time", new SimpleDateFormat("HH:mm:ss"));
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="sequenceNumberColumn">
    /**
     * Колонка <b>scn</b>, тип Long
     */
    public static final LogRecordColumn sequenceNumberColumn  = new LogRecordColumn()
        .name("scn")
        .type(Long.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    return lr.getSequenceNumber();
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="threadIdColumn">
    /**
     * Колонка <b>threadId</b>, тип Integer
     */
    public static final LogRecordColumn threadIdColumn  = new LogRecordColumn()
        .name("threadId")
        .type(Integer.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    return lr.getThreadID();
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="sourceClassColumn">
    /**
     * Колонка <b>sourceClass</b>, тип String
     */
    public static final LogRecordColumn sourceClassColumn = new LogRecordColumn()
        .name("sourceClass")
        .type(String.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    return lr.getSourceClassName();
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="sourceMethodColumn">
    /**
     * Колонка <b>sourceMethod</b>, тип String
     */
    public static final LogRecordColumn sourceMethodColumn = new LogRecordColumn()
        .name("sourceMethod")
        .type(String.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    return lr.getSourceMethodName();
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="errorClassColumn">
    /**
     * Колонка <b>errorClass</b>, тип String
     */
    public static final LogRecordColumn errorClassColumn = new LogRecordColumn()
        .name("errorClass")
        .type(String.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    Throwable err = lr.getThrown();
                    return err!=null ? err.getClass().getName() : null;
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="errorMessageColumn">
    /**
     * Колонка <b>errorMessage</b>, тип String
     */
    public static final LogRecordColumn errorMessageColumn = new LogRecordColumn()
        .name("errorMessage")
        .type(String.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    Throwable err = lr.getThrown();
                    return err!=null ? err.getMessage(): null;
                }
                return null;
            }
        });
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="errorLocalizedMessageColumn">
    /**
     * Колонка <b>errorLocalizedMessage</b>, тип String
     */
    public static final LogRecordColumn errorLocalizedMessageColumn = new LogRecordColumn()
        .name("errorLocalizedMessage")
        .type(String.class)
        .reader(new Convertor<Object, Object>() {
            @Override
            public Object convert(Object from) {
                if( from instanceof LogRecord ){
                    LogRecord lr = (LogRecord)from;
                    Throwable err = lr.getThrown();
                    return err!=null ? err.getLocalizedMessage(): null;
                }
                return null;
            }
        });
    //</editor-fold>
    //</editor-fold>
}
