/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.log;

import java.beans.PropertyChangeSupport;
import java.util.Objects;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.gui.swing.bean.UiBean;

/**
 * Описывает источник сообщений (Logger)
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class LogSource 
    implements Comparable<LogSource>
{
    private final PropertyChangeSupport  psupport;
    private long scn = 0;
    
    /**
     * Конструктор
     */
    public LogSource(){
        psupport = new PropertyChangeSupport(this);
    }
    
    /**
     * Конструктор
     * @param loggerName имя источника
     * @param level уровень логирования
     */
    public LogSource(String loggerName,LoggerLevel level){
        psupport = new PropertyChangeSupport(this);
        if( loggerName!=null ){
            setLoggerName(loggerName);
        }
        if( level!=null ){
            setLoggerLevel(level);
        }
    }
    
    @Override
    @SuppressWarnings("FinalizeDeclaration")
    protected void finalize() throws Throwable {
        synchronized(this){
            if( isListen() )stop();
        }
        super.finalize();
    }
    
    /**
     * Возвращает номер изменений
     * @return номер изменений
     */
    public long getScn(){
        synchronized(this){ 
            return scn; 
        }
    }
    
    //<editor-fold defaultstate="collapsed" desc="logger">
    protected Logger logger;
    
    /**
     * Вовзаращет ссылку на логгер
     * @return логгер
     */
    public Logger getLogger() {
        synchronized(this){
            return logger;
        }
    }
    
    /**
     * Указывает логгер
     * @param logger логгер
     */
    protected void setLogger(Logger logger) {
        synchronized(this){
            this.logger = logger;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="handler">
    protected Handler handler;
    
    /**
     * Указывает обрабочик поступающий сообщений от логгера
     * @return обрабочик лог записей
     */
    public Handler getHandler() {
        synchronized(this){
            return handler;
        }
    }
    
    /**
     * Указывает обрабочик поступающий сообщений от логгера
     * @param handler обрабочик лог записей
     */
    public void setHandler(Handler handler) {
        synchronized(this){
            this.handler = handler;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="listen : boolean">
    /**
     * Указывает установленный ли обрабочик на заданный логгер
     * @return true - обрабочик установлен и принимает события
     */
    public boolean isListen(){
        synchronized(this){
            if( handler==null )return false;
            if( logger==null )return false;
            
            Handler[] hndlrs = logger.getHandlers();
            if( hndlrs==null || hndlrs.length<1 ){
                return false;
            }
            
            for( Handler hdlr : hndlrs ){
                if( Objects.equals(hdlr, handler) )return true;
            }
            return false;
        }
    }
    
    /**
     * Указывает установить или снять обрабочик
     * @param lstn true установить обрабочик / false снять
     */
    public void setListen( boolean lstn ){
        synchronized(this){
            if( lstn ){
                if( isListen() )return;
                start();
            }else{
                if( !isListen() )return;
                stop();
            }
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="start()">
    /**
     * Устанавливает обработчик лог записей на логгер
     */
    public void start(){
        synchronized(this){
            if( handler==null )throw new IllegalStateException("handler == null");
            if( logger==null )throw new IllegalStateException("logger == null");
            if( !isListen() ){
                logger.addHandler(handler);
            }
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="stop()">
    /**
     * Снимает обабочик лог записей с логгера
     */
    public void stop(){
        synchronized(this){
            if( handler==null )throw new IllegalStateException("handler == null");
            if( logger==null )throw new IllegalStateException("logger == null");
            if( isListen() ){
                logger.removeHandler(handler);
            }
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="loggerName">
    protected String loggerName;
    
    /**
     * Вощвращает имя логгера / источника лог записей
     * @return имя логгера
     */
    public String getLoggerName() {
        synchronized(this){
            return loggerName;
        }
    }
    
    /**
     * Указывает имя логгера / источник лог записей
     * @param loggerName имя логгера
     */
    public void setLoggerName(String loggerName) {
        Object old,cur;
        synchronized(this){
            boolean isLstn = isListen();
            if( isLstn ){
                stop();
            }
            
            old = this.loggerName;
            this.loggerName = loggerName;
            cur = this.loggerName;
            
            if( loggerName!=null ){
                logger = Logger.getLogger(loggerName);
            }else{
                logger = null;
            }
            
            if( handler!=null && logger!=null && isLstn ){
                start();
            }
            
            scn++;
        }
        psupport.firePropertyChange("loggerName", old, cur);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="loggerLevel">
    /**
     * Указывает уровень логирования
     * @return уровень логирования
     */
    public LoggerLevel getLoggerLevel() {
        synchronized(this){
            if( logger==null )return LoggerLevel.UNDEFINED;
            return LoggerLevel.level(logger.getLevel());
        }
    }
    
    /**
     * Указывает уровень логирования
     * @param loggerLevel уровень логирования
     */
    @UiBean(forceNotNull = true)
    public void setLoggerLevel(LoggerLevel loggerLevel) {
        Object old,cur=null;
        boolean chng = false;
        synchronized(this){
            old = getLoggerLevel();
            if( logger!=null ){
                Level lvl = loggerLevel!=null ? loggerLevel.level() : null;
                if( lvl!=null ){
                    logger.setLevel(lvl);
                    cur = getLoggerLevel();
                    chng = true;
                }
            }
        }
        if( chng ){
            psupport.firePropertyChange("loggerName", old, cur);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="hashCode()/equals()/compareTo()">
    @Override
    public int hashCode() {
        synchronized(this){
            int hash = 7;
            hash = 79 * hash + Objects.hashCode(this.loggerName);
            //hash = 79 * hash + Objects.hashCode(this.loggerLevel);
            return hash;
        }
    }
    
    @Override
    public boolean equals(Object obj) {
        synchronized(this){
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (getClass() != obj.getClass()) {
                return false;
            }
            final LogSource other = (LogSource) obj;
            
            if (!Objects.equals(this.loggerName, other.loggerName)) return false;
            //if (!Objects.equals(this.loggerLevel, other.loggerLevel)) return false;
            
            return true;
        }
    }
    
    @Override
    public int compareTo(LogSource o) {
        if( o==null )return -1;
        synchronized(this){
            String lgrn = this.loggerName;
            String slgrn = o.loggerName;
            
            if( Objects.equals(lgrn, slgrn) )return 0;
            
            int cmpLgrName =
                (lgrn==null && slgrn==null) ? 0 :
                ( lgrn!=null && slgrn==null ) ? -1 :
                ( lgrn==null && slgrn!=null ) ? 1 :
                lgrn.compareTo(slgrn)
                ;
            
            if( lgrn==null && slgrn==null )return 0;
            if( lgrn!=null && slgrn==null )return -1;
            if( lgrn==null && slgrn!=null )return 1;
            
            //return cmpLgrName!=0 ? cmpLgrName : cmpLgrLvl;
            return cmpLgrName;
        }
    }
//</editor-fold>
}
