/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.log;

import java.awt.Color;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.SortInsert;
import xyz.cofe.collection.list.IndexEventList;
import xyz.cofe.collection.set.BasicEventSet;
import xyz.cofe.common.Reciver;
import xyz.cofe.data.store.CSVDesc;
import xyz.cofe.data.store.CSVUtil;
import xyz.cofe.gui.swing.SwingListener;
import xyz.cofe.gui.swing.cell.CellFormat;
import xyz.cofe.gui.swing.cell.TCRenderer;
import xyz.cofe.gui.swing.cell.TableCellContext;
import xyz.cofe.gui.swing.color.ColorModificator;
import xyz.cofe.gui.swing.color.HSV;
import xyz.cofe.gui.swing.color.NColorModificator;
import xyz.cofe.gui.swing.properties.PropertyValue;
import xyz.cofe.gui.swing.table.Column;
import xyz.cofe.gui.swing.table.ListTM;
import xyz.cofe.gui.swing.table.Table;
import xyz.cofe.gui.swing.tree.FormattedValue;
import xyz.cofe.gui.swing.tree.TreeTableNode;
import xyz.cofe.gui.swing.tree.TreeTableNodeBasic;
import xyz.cofe.gui.swing.tree.TreeTableNodeRender;
import xyz.cofe.text.EndLine;
import xyz.cofe.typeconv.ExtendedCastGraph;
import xyz.cofe.typeconv.TypeCastGraph;

/**
 * Таблица лог сообщений/записей
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class LogTable extends Table
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(LogTable.class.getName());

    private static Level logLevel(){ return logger.getLevel(); }
    
    private static boolean isLogSevere(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null ? true : logLevel.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level logLevel = logger.getLevel(); 
        return logLevel==null  ? true : logLevel.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINER.intValue();
    }
    
    private static boolean isLogFinest(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINEST.intValue();
    }

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(LogTable.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(LogTable.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(LogTable.class.getName(), method, result);
    }
    //</editor-fold>
    
    /**
     * Конструктор
     */
    public LogTable(){
        records = new IndexEventList<>();
        queue = new LinkedBlockingQueue();
        listTM = new ListTM();
        initModel();
        
        setDefaultRenderer(FormattedValue.class, renderer);
        setDefaultRenderer(PropertyValue.class, renderer);
        setDefaultRenderer(TreeTableNode.class, renderer);
        setDefaultRenderer(TreeTableNodeBasic.class, renderer);
        
        //setRowHeight(20);
        initColumnConf();
        initFormatting();
        initSelectMode();
        initClipboard();
    }
    
    private void initSelectMode(){
        getColumnModel().setColumnSelectionAllowed(true);
    }
    
    private void initClipboard(){
        SwingListener.onKeyPressed(this, new Reciver<KeyEvent>() {
            @Override
            public void recive(KeyEvent ke) {
                if( !ke.isControlDown() )return;
                if( ke.getKeyCode()==KeyEvent.VK_C || ke.getKeyCode()==KeyEvent.VK_COPY ){
                    copy(ke);
                }
            }
        });
    }
    
    private void copy(KeyEvent ke){
        //ke.consume();
        int[] selectedRows = getSelectedRows();
        if( selectedRows==null || selectedRows.length<1 )return;
        
        if( getColumnModel().getColumnSelectionAllowed() ){
            int[] selectedColumns = getColumnModel().getSelectedColumns();
            if( selectedColumns==null || selectedColumns.length<1 )return;
            if( copyCells(selectedRows, selectedColumns) )ke.consume();
        }else{
            int[] selectedColumns = new int[getColumnCount()];
            for( int ci=0; ci<getColumnCount(); ci++ )selectedColumns[ci] = ci;
            if( copyCells(selectedRows, selectedColumns) )ke.consume();
        }
    }
    
    private boolean copyCells( int[] rows, int[] columns ){
        if( listTM==null )return false;
        if( columns.length<1 )return false;
        if( rows.length<1 )return false;
        
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSSXXX");
        TypeCastGraph tcast = new ExtendedCastGraph();
        
        CSVDesc csvDesc = new CSVDesc();
        csvDesc.setFixedWidth(false);
        csvDesc.setQuoteVariants(CSVDesc.QuoteVariants.Always);
        
        CSVUtil csvUtl = new CSVUtil();
        
        StringBuilder sb = new StringBuilder();
        int rcnt = 0;
        for( int row : rows ){
            rcnt++;
            if( rcnt>1 )sb.append(EndLine.Default.get());
            
            String[] txtCells = new String[columns.length];
            int ci = -1;
            for( int col : columns ){
                ci++;
                TableColumn tc = getColumnModel().getColumn(col);
                if( tc==null ){
                    txtCells[ci] = "";
                    continue;
                }
                
                int mi = tc.getModelIndex();
                Object val = listTM.getValueAt(row, mi);
                if( val instanceof FormattedValue ){
                    val = ((FormattedValue)val).getValue();
                }
                
                if( val==null ){
                    txtCells[ci] = "null";
                    continue;
                }else if( val instanceof Date ){
                    txtCells[ci] = sdf.format((Date)val);
                }else {
                    try{
                        String str = tcast.cast(val, String.class);
                        if( str!=null ){
                            txtCells[ci] = str;
                        }else{
                            txtCells[ci] = "null";
                        }
                    }catch( Throwable err ){
                        logException(err);
                        txtCells[ci] = val.toString();
                    }                    
                }
            }
            
            if( txtCells.length==1 ){
                sb.append(txtCells[0]);
            }else{
                String line = csvUtl.toString(txtCells, csvDesc).trim();
                sb.append(line);
            }
        }
        
        StringSelection stringSelection = new StringSelection(sb.toString());
    
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        clipboard.setContents(stringSelection, stringSelection);
        
        return true;
    }
    
    private TreeMap<Integer,Color> bgByLevelUpper;
    private List<NColorModificator> bgMod;
    
    private void initFormatting(){
        bgByLevelUpper = new TreeMap<>();
        bgByLevelUpper.put(Level.FINEST.intValue(), Color.white);
        
        bgByLevelUpper.put(Level.FINER.intValue(), 
            new Color(Color.HSBtoRGB(1f/360f*210f, 0.2f, 1f))
            //HSV.hsv2rgb(1f/360f*210f, 0.2f, 1f)
        );
        
        bgByLevelUpper.put(Level.FINE.intValue(), 
            //new ColorModificator().hue(105).sate(0.5f).apply(Color.red)
            new Color(Color.HSBtoRGB(1f/360f*105f, 0.2f, 1f))
        );
        
        bgByLevelUpper.put(Level.CONFIG.intValue(), 
            //new ColorModificator().hue(105).sate(0.5f).apply(Color.red)
            new Color(Color.HSBtoRGB(1f/360f*75f, 0.2f, 1f))
        );
        
        bgByLevelUpper.put(Level.INFO.intValue(), 
            //new ColorModificator().hue(60).sate(0.5f).apply(Color.red)
            new Color(Color.HSBtoRGB(1f/360f*60f, 0.2f, 1f))
        );
        
        bgByLevelUpper.put(Level.WARNING.intValue(), 
            //new ColorModificator().hue(30).sate(0.5f).apply(Color.red)
            new Color(Color.HSBtoRGB(1f/360f*30f, 0.2f, 1f))
        );
        
        bgByLevelUpper.put(Level.SEVERE.intValue(), 
            //new ColorModificator().hue(0).sate(0.5f).apply(Color.red)
            new Color(Color.HSBtoRGB(1f/360f*0f, 0.2f, 1f))
        );
        
        bgMod = new ArrayList<>();
        bgMod.add(new NColorModificator(1, 2, new ColorModificator().brighter(-0.15f)));
        
        renderer.getLabelRender().addFormatter(TableCellContext.class, new Func2<CellFormat, TableCellContext, CellFormat>() {
            @Override
            public CellFormat apply(TableCellContext cx, CellFormat cf) {                
                if( cf!=null && cx!=null ){
                    if( cx.isFocus() || cx.isSelected() )return cf;
                    
                    int row = cx.getRow();
                    Object orec = listTM.getItemByIndex(row);
                    if( orec instanceof LogRecord ){
                        LogRecord lr = (LogRecord)orec;
                        int lvl = lr.getLevel().intValue();
                        
                        Map.Entry<Integer,Color> colEn = bgByLevelUpper.floorEntry(lvl);
                        if( colEn!=null ){
                            Color bg = colEn.getValue();                            
                            if( bgMod!=null && bg!=null ){
                                for( NColorModificator ncMod : bgMod ){
                                    int c = ncMod.getCycle();
                                    int p = ncMod.getPhase();
                                    if( c<=1 || (row%c)==p ){
                                        bg = ncMod.apply(bg);
                                    }
                                }
                            }
                            if( bg!=null ){
                                cf.bgColor(bg);
                            }
                        }
                    }
                }
                return cf;
            }
        });
    }
    
    //<editor-fold defaultstate="collapsed" desc="initColumnConf">
    private void initColumnConf(){
        final JTableHeader header = getTableHeader();
        if( header!=null ){
            SwingListener.onMousePressed(header, new Reciver<MouseEvent>() {
                @Override
                public void recive(MouseEvent me) {
                    if( me.getButton()!=MouseEvent.BUTTON3 )return;
                    
                    JPopupMenu popup = new JPopupMenu();
                    List<ManagedColumn> mcList = new ArrayList<>();
                    for( ManagedColumn mc : getManagedColumns() ){
                        mcList.add(mc);
                    }
                    Collections.sort(mcList, new Comparator<ManagedColumn>() {
                        @Override
                        public int compare(ManagedColumn mc1, ManagedColumn mc2) {
                            return Objects.compare(mc1, mc2, new Comparator<ManagedColumn>(){
                                @Override
                                public int compare(ManagedColumn c1, ManagedColumn c2) {
                                    int di1 = c1.getDisplayIndex();
                                    int di2 = c2.getDisplayIndex();
                                    
                                    String n1 = c1.getName();
                                    String n2 = c2.getName();
                                    
                                    if( di1!=di2 )return di1 - di2;
                                    return n1.compareTo(n2);
                                }
                            });
                        }
                    });
                    for( final ManagedColumn mc : mcList ){
                        JCheckBoxMenuItem cbMenuItm = new JCheckBoxMenuItem(mc.getName());
                        cbMenuItm.setSelected(mc.isVisible());
                        SwingListener.onActionPerformed(cbMenuItm, new Reciver<ActionEvent>() {
                            @Override
                            public void recive(ActionEvent obj) {
                                mc.setVisible( !mc.isVisible() );
                            }
                        });
                        popup.add(cbMenuItm);
                    }
                    popup.show(header, me.getX(), me.getY());
                }
            });
        }
    }
    //</editor-fold>
    
    private final TCRenderer renderer = new TCRenderer();
    
    //<editor-fold defaultstate="collapsed" desc="avaliableColumnsNames">
    private String[] avaliableColumnsNames;
    public String[] getAvaliableColumnNames(){
        if( avaliableColumnsNames!=null )return avaliableColumnsNames;
        String[] colNames = new String[0];
        for( Column col : listTM.getColumns() ){
            colNames = Arrays.copyOf(colNames, colNames.length+1);
            colNames[colNames.length-1] = col.getName();
        }
        avaliableColumnsNames = colNames;
        return colNames;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="modelIndexOfColumn( name ):int">
    /**
     * Возвращает индекс колонки модели таблицы
     * @param name имя колонки в модели
     * @return индекс или -1
     */
    public int modelIndexOfColumn( String name ){
        if( name==null )return -1;
        String[] cols = getAvaliableColumnNames();
        for( int ci=0; ci<cols.length; ci++ ){
            if( name.equals(cols[ci]) )return ci;
        }
        return -1;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="getTableColumns( name ):TableColumn[]">
    /**
     * Возвращает колоки соответ колонки модели
     * @param columnName имя колонки в модели таблицы
     * @return отображаемые колонки, или пустой массив если колонка модели не отображается
     */
    public TableColumn[] getTableColumns( String columnName ){
        if( columnName==null )return new TableColumn[0];
        
        int mi = modelIndexOfColumn(columnName);
        if( mi<0 )return new TableColumn[0];
        
        TableColumnModel tcm = getColumnModel();
        TableColumn[] res = new TableColumn[0];
        
        for( int ci=0; ci<tcm.getColumnCount(); ci++ ){
            TableColumn tc = tcm.getColumn(ci);
            int m = tc.getModelIndex();
            if( m==mi ){
                res = Arrays.copyOf(res, res.length+1);
                res[res.length-1] = tc;
            }
        }
        
        return res;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="set/get column visible/displayIndex/width">
    /**
     * Возвращает видима ли колонка
     * @param columnName имя колонки
     * @return true - колонка отображается
     */
    public boolean isColumnVisible( String columnName ){
        if( columnName==null )return false;
        return getTableColumns(columnName).length>0;
    }
    
    /**
     * Отобразить/скрыть колонку модели
     * @param columnName имя колонки модели
     * @param visible true - отобразить / false - скрыть
     */
    public void setColumnVisible( String columnName, boolean visible ){
        int mi = modelIndexOfColumn(columnName);
        if( mi<0 )return;
        
        TableColumn[] tcs = getTableColumns(columnName);
        if( visible ){
            if( tcs.length>0 )return;
            
            TableColumn tc = createTableColumn(columnName);
            if( tc==null )return;
            
            this.getColumnModel().addColumn(tc);
        }else{
            for( TableColumn tc : tcs ){
                this.getColumnModel().removeColumn(tc);
            }
        }
    }
    
    /**
     * Возвразает видимый индекс колонки
     * @param columnName имя колонки в модели
     * @return индекс или -1
     */
    public int getColumnDisplayIndex( String columnName ){
        if( columnName==null )return -1;
        
        int mi = modelIndexOfColumn(columnName);
        if( mi<0 )return -1;
        
        TableColumnModel tcm = getColumnModel();
        
        for( int ci=0; ci<tcm.getColumnCount(); ci++ ){
            TableColumn tc = tcm.getColumn(ci);
            int m = tc.getModelIndex();
            if( m==mi ){
                return ci;
            }
        }
        
        return -1;
    }
    
    /**
     * Указывает какую колонку модели в каком порядке отображать
     * @param columnName имя колонки в модели
     * @param vi индекс или -1, если ее необходимо скрыть
     */
    public void setColumnDisplayIndex( String columnName, int vi ){
        if( columnName==null )return;
        
        if( vi<0 ){
            setColumnVisible(columnName, false);
            return;
        }
        
        int mi = modelIndexOfColumn(columnName);
        if( mi<0 )return;
        
        TableColumnModel tcm = getColumnModel();
        
        Map<TableColumn,Integer> cols = new LinkedHashMap<>();
        TableColumn firstColumn = null;
        for( int ci=0; ci<tcm.getColumnCount(); ci++ ){
            TableColumn tc = tcm.getColumn(ci);
            int m = tc.getModelIndex();
            if( m==mi ){
                cols.put(tc, ci);
                if( firstColumn==null )firstColumn = tc;
            }
        }
        
        // column already visible
        if( firstColumn!=null ){
            int idxFrom = cols.get(firstColumn);
            getColumnModel().moveColumn(idxFrom, vi);
        }else{
            TableColumn tc = createTableColumn(columnName);
            getColumnModel().addColumn(tc);
            int cnt = getColumnModel().getColumnCount();
            if( vi<(cnt-1) ){
                getColumnModel().moveColumn(cnt-1, vi);
            }
        }
    }
    
    /**
     * Возвращает ширину колонки
     * @param columnName имя колонки в модели
     * @return ширина или -1, если колонка невидима
     */
    public int getColumnWidth( String columnName ){
        if( columnName==null )return -1;
        TableColumn[] tcs = getTableColumns(columnName);
        return tcs.length>0 ? tcs[0].getWidth() : -1;
    }
    
    /**
     * Указывает ширину колонки
     * @param columnName имя колонки в модели
     * @param newWidth ширина колонки
     */
    public void setColumnWidth( String columnName, int newWidth ){
        if( columnName==null )throw new IllegalArgumentException("columnName");
        if( newWidth<1 )throw new IllegalArgumentException("neWidth < 1");
        
        TableColumn[] tcs = getTableColumns(columnName);
        if( tcs.length<1 ){
            TableColumn tc = createTableColumn(columnName);
            if( tc==null )throw new IllegalArgumentException("columnName("+columnName+") not exists");
            tc.setPreferredWidth(newWidth);
            addColumn(tc);
            return;
        }
        
        tcs[0].setPreferredWidth(newWidth);
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="createTableColumn">
    /**
     * Создает колонку для отображения
     * @param name имя колонки в модели
     * @return колонка отображения
     */
    public TableColumn createTableColumn( String name ){
        int mi = modelIndexOfColumn(name);
        if( mi<0 )return null;
        
        TableColumn tc = new TableColumn(mi);
        tc.setIdentifier(name);
        tc.setHeaderValue(name);
        tc.setModelIndex(mi);
        tc.setPreferredWidth(50);
        tc.setCellRenderer(renderer);
        return tc;
    }
    
    /**
     * Создает колонку для отображения
     * @param mi индекс колонки в модели
     * @return колонка отображения
     */
    public TableColumn createTableColumn( int mi ){
        if( mi<0 )return null;
        
        TableColumn tc = new TableColumn(mi);
        return tc;
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="managed columns">
    /**
     * Упралвение колонкой модели/колонкой таблицы
     */
    public class ManagedColumn {
        public ManagedColumn(LogRecordColumn lrc){
            if( lrc==null )throw new IllegalArgumentException("lrc==null");
            this.column = lrc;
        }
        
        //<editor-fold defaultstate="collapsed" desc="dataColumn">
        private LogRecordColumn column;
        
        /**
         * Возвращает колонку модели
         * @return колонка модели
         */
        public LogRecordColumn getDataColumn(){
            return column;
        }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="name">
        /**
         * Возвращает имя колонки в модели
         * @return имя колонки
         */
        public String getName(){ return column.getName(); }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="visible">
        /**
         * Отобразить колонку данных в таблице
         * @return self ссылка
         */
        public ManagedColumn visible(){
            setColumnVisible(column.getName(), true);
            return this;
        }
        
        /**
         * Указыает отображается ли колонка в таблице
         * @return true отображается
         */
        public boolean isVisible(){ return isColumnVisible(column.getName()); }
        
        /**
         * Указывает отобразаить колонку данных
         * @param visible true отобразить / false - скрыть
         */
        public void setVisible( boolean visible ){ setColumnVisible(column.getName(), visible); }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="displayIndex">
        /**
         * Возвращает индекс отображения колонки
         * @return индекс колонки в отображаемой таблице или -1, если колонка скрыта
         */
        public int getDisplayIndex(){ return getColumnDisplayIndex(getName()); }
        /**
         * Указывает индекс отображения колонки
         * @param idx индекс или -1, если необходимо скрыть
         */
        public void setDisplayIndex(int idx){ setColumnDisplayIndex(getName(),idx); }
        //</editor-fold>
        
        //<editor-fold defaultstate="collapsed" desc="width">
        /**
         * Возвращает ширину колонки
         * @return ширина или -1, если колонка скрыта
         */
        public int getWidth(){ return getColumnWidth(getName()); }
        /**
         * Указывает ширину колонки
         * @param w ширина колонки
         */
        public void setWidth( int w ){ setColumnWidth(getName(), w); }
        //</editor-fold>
    }
    
    //<editor-fold defaultstate="collapsed" desc="managedColumns : Set<ManagedColumn>">
    private Set<ManagedColumn> managedColumns;
    
    /**
     * Возвращает описание отображаемых колонок
     * @return отображаемые колонки
     */
    public Set<ManagedColumn> getManagedColumns(){
        if( managedColumns!=null )return managedColumns;
        managedColumns = new BasicEventSet<>();
        for( Column col : listTM.getColumns() ){
            if( col instanceof LogRecordColumn ){
                managedColumns.add(new ManagedColumn((LogRecordColumn)col));
            }
        }
        return managedColumns;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="add(col):managedColumn">
    /**
     * Добавляет колонку для отображения
     * @param col колонка
     * @return Управляемая/отображаемая колонка
     */
    public ManagedColumn add( LogRecordColumn col ){
        if( col==null )throw new IllegalArgumentException("col == null");
        int mi = listTM.getColumns().indexOf(col);
        if( mi<0 ){
            listTM.getColumns().add(col);
            mi = listTM.getColumns().size()-1;
            avaliableColumnsNames = null;
        }
        ManagedColumn ic = new ManagedColumn(col);
        return ic;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="set(col):managedColumn">
    /**
     * Добавляет, если еще не добавлена, колонку к модели
     * @param col колонка
     * @return Управляемая/отображаемая колонка
     */
    public ManagedColumn set( LogRecordColumn col ){
        if( col==null )throw new IllegalArgumentException("col == null");
        int mi = listTM.getColumns().indexOf(col);
        if( mi<0 ){
            listTM.getColumns().add(col);
            mi = listTM.getColumns().size()-1;
            avaliableColumnsNames = null;
        }
        
        ManagedColumn ic = new ManagedColumn(col);
        return ic;
    }
    //</editor-fold>
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="initModel()">
    protected void initModel(){
        setAutoCreateColumnsFromModel(false);
        
        add(LogRecordColumn.sequenceNumberColumn);
        add(LogRecordColumn.levelNameColumn);
        add(LogRecordColumn.loggerNameColumn);
        add(LogRecordColumn.messageColumn);
        
        add(LogRecordColumn.dateColumn);
        add(LogRecordColumn.timeColumn);
        add(LogRecordColumn.errorClassColumn);
        add(LogRecordColumn.errorLocalizedMessageColumn);
        add(LogRecordColumn.errorMessageColumn);
        add(LogRecordColumn.levelValueColumn);
        add(LogRecordColumn.millisColumn);
        add(LogRecordColumn.rawMessageColumn);
        add(LogRecordColumn.sourceClassColumn);
        add(LogRecordColumn.sourceMethodColumn);
        add(LogRecordColumn.threadIdColumn);
        
        setModel(listTM);
        listTM.setList((IndexEventList)records);
        
        set(LogRecordColumn.sequenceNumberColumn).visible();
        set(LogRecordColumn.levelNameColumn).visible();
        set(LogRecordColumn.loggerNameColumn).visible();
        set(LogRecordColumn.messageColumn).visible();
        
        listTM.getColumns().onChanged(new Func3<Object, Integer, Column, Column>() {
            @Override
            public Object apply(Integer idx, Column old, Column cur) {
                if( old!=null ){
                    Set<ManagedColumn> removeCols = new LinkedHashSet<>();                
                    for( ManagedColumn mc : getManagedColumns() ){
                        if( mc==null )continue;

                        LogRecordColumn lrc = mc.getDataColumn();
                        if( lrc==old ){
                            removeCols.add( mc );
                        }
                    }

                    getManagedColumns().removeAll(removeCols);
                }
                
                if( cur!=null && cur instanceof LogRecordColumn ){
                    getManagedColumns().add(new ManagedColumn((LogRecordColumn)cur));
                }
                
                return null;
            }
        });
    }
    //</editor-fold>
    
    protected final ListTM<LogRecord> listTM;
    
    //<editor-fold defaultstate="collapsed" desc="records : IndexEventList<LogRecord>">
    protected final IndexEventList<LogRecord> records;
    /**
     * Возвращает лог записи
     * @return лог записи
     */
    public IndexEventList<LogRecord> getRecords(){ return records; }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="queue">
    protected final Queue<LogRecord> queue;
    /**
     * Возвращает очередь лог задписи для отображения в таблице лог записей
     * @return очедерь лог записей
     */
    public Queue<LogRecord> getQueue(){ return queue; }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="add(lr)">
    /**
     * Потоко-безапастное добавление лог-сообщения
     * @param lr лог сообщение
     */
    public void add( LogRecord lr ){
        if( lr==null )return;
        
        int qs = queue.size();
        queue.add(lr);
        
        synchronized(releaseSync){
            scn++;
            
            if( qs==0 ){
                if( SwingUtilities.isEventDispatchThread() ){
                    releaseQueue.run();
                }else{
                    SwingUtilities.invokeLater(releaseQueue);
                }
            }
        }
    }
    //</editor-fold>
    //<editor-fold defaultstate="collapsed" desc="release queue">
    private final Object releaseSync = new Object();
    
    private long releaseStarted = 0;
    private long releaseFinished = 0;
    private long releaseCount = 0;
    private long addTime = 0;
    private long scn = 0;
    
    private long requestScn = 0;
    private long refreshStartScn = 0;
    private long refreshEndScn = 0;
    
    private final Runnable releaseQueue = new Runnable() {
        @Override
        public void run() {
            synchronized(releaseSync){
                releaseStarted = System.currentTimeMillis();
                releaseCount = 0;
                refreshStartScn = scn;
            }
            while(true){
                LogRecord lr = queue.poll();
                if( lr==null )break;
                
                records.add(lr);
                synchronized(releaseSync){
                    releaseCount++;
                }
            }
            synchronized(releaseSync){
                releaseFinished = System.currentTimeMillis();
                refreshEndScn = scn;
            }
        }
    };
    //</editor-fold>
}
