/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.log;

import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import xyz.cofe.text.Text;

/**
 * Фильтр-маска
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class MaskFilter 
    implements ChainElement
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(MaskFilter.class.getName());

    private static Level logLevel(){ return logger.getLevel(); }
    
    private static boolean isLogSevere(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null ? true : logLevel.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level logLevel = logger.getLevel(); 
        return logLevel==null  ? true : logLevel.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINER.intValue();
    }
    
    private static boolean isLogFinest(){ 
        Level logLevel = logger.getLevel();
        return logLevel==null  ? true : logLevel.intValue() <= Level.FINEST.intValue();
    }

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(MaskFilter.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(MaskFilter.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(MaskFilter.class.getName(), method, result);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="filtersChain">
    private volatile ChainFilters filtersChain;
    
    /**
     * Возвращает цепочку фильтров
     * @return цепочка фильтров
     */
    public ChainFilters getFiltersChain() {
        synchronized(this){
            return filtersChain;
        }
    }
    
    /**
     * Указывает цепочка фильтров
     * @param filtersChain цепочка фильтров
     */
    public void setFiltersChain(ChainFilters filtersChain) {
        synchronized(this){
            this.filtersChain = filtersChain;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="enabled">
    private boolean enabled = true;
    
    @Override
    public boolean isEnabled(){
        synchronized(this){
            return enabled;
        }
    }
    
    @Override
    public void setEnabled(boolean enabled){
        synchronized(this){
            if( filtersChain!=null ){
                synchronized(filtersChain){
                    this.enabled = enabled;
                }
            }else{
                this.enabled = enabled;
            }
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="loggerName : String">
    private volatile String loggerName;
    
    /**
     * Указывает шаблон (wildcard) имени логгера
     * @return шаблон (wildcard) имени логгера
     */
    public String getLoggerName() {
        synchronized(this){
            return loggerName;
        }
    }
    
    /**
     * Указывает шаблон (wildcard) имени логгера
     * @param loggerName шаблон (wildcard) имени логгера
     */
    public void setLoggerName(String loggerName) {
        synchronized(this){
            if( filtersChain!=null ){
                synchronized(filtersChain){
                    this.loggerName = loggerName;
                }
            }else{
                this.loggerName = loggerName;
            }
            loggerNamePattern = null;
        }
    }
    
    private volatile Pattern loggerNamePattern;
    protected Pattern getLoggerNamePattern(){
        synchronized(this){
            if( loggerNamePattern!=null )return loggerNamePattern;
            if( loggerName==null )return null;
            loggerNamePattern = Text.wildcard(loggerName, false, true);
            return loggerNamePattern;
        }
    }
    
    private boolean matchLoggerName(LogRecord rec){
        String lgrName = rec.getLoggerName();
        Pattern ptrn = getLoggerNamePattern();
        if( ptrn==null || lgrName==null )return false;
        return ptrn.matcher(lgrName).matches();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="message : String">
    private volatile String message;
    
    /**
     * Указывает шаблон текстового сообщения
     * @return шаблон (wildcard) сообщения
     */
    public String getMessage() {
        synchronized(this){ return message; }
    }
    
    /**
     * Указывает шаблон текстового сообщения
     * @param message шаблон (wildcard) сообщения
     */
    public void setMessage(String message) {
        synchronized(this){
            if( filtersChain!=null ){
                synchronized(filtersChain){
                    this.message = message;
                }
            }else{
                this.message = message;
            }
            messagePattern = null;
        }
    }
    
    private volatile Pattern messagePattern;
    protected Pattern getMessagePattern(){
        synchronized(this){
            if( messagePattern!=null )return messagePattern;
            if( message==null )return null;
            messagePattern = Text.wildcard(message, false, true);
            return messagePattern;
        }
    }
    
    private boolean matchMessage(LogRecord rec){
        String msg = rec.getMessage();
        if( msg==null )return false;
        msg = LogRecordColumn.formatMessage(rec);
        Pattern ptrn = getMessagePattern();
        if( ptrn==null || msg==null )return false;
        return ptrn.matcher(msg).matches();
    }
    //</editor-fold>
  
    //<editor-fold defaultstate="collapsed" desc="action : ChainAction">
    private volatile ChainAction action = ChainAction.Include;
    
    /**
     * Указывает поведение при совпадении правила
     * @return поведение при совпадении
     */
    public ChainAction getAction() {
        synchronized(this){
            return action;
        }
    }
    
    /**
     * Указывает поведение при совпадении правила
     * @param action поведение при совпадении правила
     */
    public void setAction(ChainAction action) {
        synchronized(this){
            if( filtersChain!=null ){
                synchronized(filtersChain){
                    this.action = action;
                }
            }else{
                this.action = action;
            }
        }
    }
    //</editor-fold>

    @Override
    public ChainAction isLoggable(LogRecord rec) {
        if( rec==null )return null;
        synchronized(this){
            int condCnt = 0;
            int mtchCnt = 0;
            
            boolean defLgr = getLoggerNamePattern()!=null;
            if( defLgr )condCnt++;
            
            boolean defMsg = getMessagePattern()!=null;
            if( defMsg )condCnt++;
            
            boolean mtcLgr = matchLoggerName(rec);
            if( mtcLgr )mtchCnt++;
            
            boolean mtcMsg = matchMessage(rec);
            if( mtcMsg )mtchCnt++;
            
            if( (mtchCnt == condCnt) && mtchCnt>0 )return getAction();
            return null;
        }
    }
}
