/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.log;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOError;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import xyz.cofe.xml.FormatXMLWriter;
import xyz.cofe.xml.stream.path.XEventPath;
import xyz.cofe.xml.stream.path.XVisitor;
import xyz.cofe.xml.stream.path.XVisitorAdapter;
import xyz.cofe.xml.stream.path.XmlReader;
import xyz.cofe.xml.stream.path.PathExpression;
import xyz.cofe.xml.stream.path.PathParser;

/**
 * XML Конфигурация
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class XmlConfig {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(XmlConfig.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(XmlConfig.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(XmlConfig.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(XmlConfig.class.getName(), method, result);
    }
    //</editor-fold>
    
    public XmlConfig(){
    }
    
    public void write( File out, LogHandlerPanel lp ){
        if( out==null )throw new IllegalArgumentException("out == null");
        if( lp==null )throw new IllegalArgumentException("lp == null");
        
        try {
            FileOutputStream fo = new FileOutputStream(out);
            FormatXMLWriter fxw = new FormatXMLWriter(fo,"utf-8");
            write(fxw, lp);
            fxw.flush();
            fo.close();
        } catch (XMLStreamException ex) {
            Logger.getLogger(XmlConfig.class.getName()).log(Level.SEVERE, null, ex);
            throw new IOError(ex);
        } catch (IOException ex) {
            Logger.getLogger(XmlConfig.class.getName()).log(Level.SEVERE, null, ex);
            throw new IOError(ex);
        }
    }
    
    public void write( OutputStream out, LogHandlerPanel lp ){
        if( out==null )throw new IllegalArgumentException("out == null");
        if( lp==null )throw new IllegalArgumentException("lp == null");
        
        try {
            FormatXMLWriter fxw = new FormatXMLWriter(out,"utf-8");
            write(fxw, lp);
            fxw.flush();
        } catch (XMLStreamException ex) {
            Logger.getLogger(XmlConfig.class.getName()).log(Level.SEVERE, null, ex);
            throw new IOError(ex);
        }
    }
    
    public void write( Writer out, LogHandlerPanel lp ){
        if( out==null )throw new IllegalArgumentException("out == null");
        if( lp==null )throw new IllegalArgumentException("lp == null");
        
        try {
            FormatXMLWriter fxw = new FormatXMLWriter(out);
            write(fxw, lp);
            fxw.flush();
        } catch (XMLStreamException ex) {
            Logger.getLogger(XmlConfig.class.getName()).log(Level.SEVERE, null, ex);
            throw new IOError(ex);
        }
    }
    
    public void write( XMLStreamWriter out, LogHandlerPanel lp ) throws XMLStreamException {
        if( out==null )throw new IllegalArgumentException("out == null");
        if( lp==null )throw new IllegalArgumentException("lp == null");
        
        out.writeStartElement("logHandlerPanel");
        writeSources(out, lp);
        writeFilters(out, lp.getFilters());
        writeColumns(out, lp.getLogTable().getManagedColumns() );
        out.writeEndElement();
    }
    
    private static final PathExpression sourcesExp;
    private static final PathExpression logSourceExp;
    
    private static final PathExpression filtersExp;
    private static final PathExpression maskFilterExp;
    
    private static final PathExpression columnsExp;
    private static final PathExpression columnExp;
    
    static {
        PathParser pp = new PathParser();
        
        sourcesExp = pp.parse("sources");
        logSourceExp = pp.parse("logSource");
        
        filtersExp = pp.parse("filters");
        maskFilterExp = pp.parse("maskFilter");
        
        columnsExp = pp.parse("columns");
        columnExp = pp.parse("column");
    }
    
    public void readXml( final LogHandlerPanel lp, String input )  throws XMLStreamException {
        if( lp==null )throw new IllegalArgumentException("lp == null");
        if( input==null )throw new IllegalArgumentException("input == null");
        
        new XmlReader(input, createXVisitor(lp));
    }
    
    public void readXml( final LogHandlerPanel lp, Reader input )  throws XMLStreamException {
        if( lp==null )throw new IllegalArgumentException("lp == null");
        if( input==null )throw new IllegalArgumentException("input == null");
        
        new XmlReader(input, createXVisitor(lp));
    }
    
    public void read( final LogHandlerPanel lp, File input )  throws XMLStreamException, IOException {
        if( lp==null )throw new IllegalArgumentException("lp == null");
        if( input==null )throw new IllegalArgumentException("input == null");
        
        new XmlReader(input, Charset.forName("utf-8"), createXVisitor(lp));
    }
    
    public void read( final LogHandlerPanel lp, XMLStreamReader input ) throws XMLStreamException {
        if( lp==null )throw new IllegalArgumentException("lp == null");
        if( input==null )throw new IllegalArgumentException("input == null");
        
        new XmlReader(input, createXVisitor(lp));
    }

    private XVisitor createXVisitor( final LogHandlerPanel lp ){
        if( lp==null )throw new IllegalArgumentException("lp == null");
        
        XVisitor visitor = new XVisitorAdapter(){
            @Override
            public void characters(XEventPath path, String text) {
            }

            @Override
            public void exit(XEventPath path) {
                if( filtersExp.validate(path) ){
                    lp.getFiltersCacheTM().refresh();
                }else if( sourcesExp.validate(path) ){
                    lp.getSourcesCacheTM().refresh();
                }else if( columnsExp.validate(path) ){
                    readColumnsFinish();
                }
            }

            @Override
            public void enter(XEventPath path) {
                if( sourcesExp.validate(path) ){
                    readLogSourcesStart(lp);
                }else if( logSourceExp.validate(path) ){
                    readLogSource(lp, path);
                }else if( filtersExp.validate(path) ){
                    readFiltersStart(lp, path);
                }else if( maskFilterExp.validate(path) ){
                    readMaskFilter(lp, path);
                }else if( columnExp.validate(path) ){
                    readColumnStart(lp, path);
                }else if( columnsExp.validate(path) ){
                    readColumnsStart();
                }
            }
        };
        
        return visitor;
    }
    
    //<editor-fold defaultstate="collapsed" desc="sources">
    private void write( XMLStreamWriter out, LogSource ls ) throws XMLStreamException {
        out.writeStartElement("logSource");
        
        String lgrName = ls.getLoggerName();
        if( lgrName!=null )out.writeAttribute("loggerName", lgrName);
        
        LoggerLevel ll = ls.getLoggerLevel();
        if( ll!=null )out.writeAttribute("loggerLevel", ll.name());
        
        boolean listen = ls.isListen();
        out.writeAttribute("listen", listen ? "true" : "false");
        
        out.writeEndElement();
    }
    
    private void writeSources( XMLStreamWriter out, LogHandlerPanel lp ) throws XMLStreamException {
        out.writeStartElement("sources");
        for( LogSource ls : lp.getSources() ){
            if( ls==null )continue;
            write(out, ls);
        }
        out.writeEndElement();
    }
    
    private void readLogSource( LogHandlerPanel lp, XEventPath xpath ){
        String lgrName = xpath.getAttributeValue("loggerName");
        String lgrLvl = xpath.getAttributeValue("loggerLevel");
        boolean listen = xpath.readAttributeAsBoolean("listen", false);
        
        LogSource ls = new LogSource();
        if( lgrName!=null ){ ls.setLoggerName(lgrName); }
        if( lgrLvl!=null ){ ls.setLoggerLevel(LoggerLevel.valueOf(lgrLvl)); }
        lp.getSources().add(ls);
        
        if( listen ){
            ls.start();
        }
    }
    
    private void readLogSourcesStart( LogHandlerPanel lp ){
        lp.getSources().clear();
    }
    //</editor-fold>
    
    private void writeMskFilter( XMLStreamWriter out, MaskFilter mf ) throws XMLStreamException {
        out.writeStartElement("maskFilter");
        
        String lgrName = mf.getLoggerName();
        if( lgrName!=null )out.writeAttribute("loggerName", lgrName);
        
        String msg = mf.getMessage();
        if( msg!=null )out.writeAttribute("message", lgrName);
        
        ChainAction ca = mf.getAction();
        if( ca!=null ){ out.writeAttribute("action", ca.name()); }
        
        boolean en = mf.isEnabled();
        out.writeAttribute("enabled", en ? "true" : "false");
        
        out.writeEndElement();
    }
    
    private void readMaskFilter( LogHandlerPanel lp, XEventPath xpath ){
        String lgrName = xpath.getAttributeValue("loggerName");
        String message = xpath.getAttributeValue("message");
        String action = xpath.getAttributeValue("action");
        boolean enabled = xpath.readAttributeAsBoolean("enabled", true);
        
        MaskFilter mf = new MaskFilter();
        if( lgrName!=null ) mf.setLoggerName(lgrName);
        if( message!=null ) mf.setMessage(message);
        if( action!=null ) mf.setAction(ChainAction.valueOf(action));
        mf.setEnabled(enabled);
        
        lp.getFilters().add(mf);        
    }
    
    private void writeFilters( XMLStreamWriter out, ChainFilters cf ) throws XMLStreamException {
        out.writeStartElement("filters");
        
        ChainAction ca = cf.getDefaultAction();
        if( ca!=null )out.writeAttribute("defaultAction", ca.name());
        
        for( Object o : cf.array() ){
            if( o instanceof MaskFilter ){
                writeMskFilter( out, (MaskFilter)o );
            }
        }
        
        out.writeEndElement();
    }
    
    private void readFiltersStart( LogHandlerPanel lp, XEventPath xpath ){
        lp.getFilters().clear();
        
        String action = xpath.getAttributeValue("action");
        if( action!=null ) lp.getFilters().setDefaultAction(ChainAction.valueOf(action));
    }
    
    private void writeColumn( XMLStreamWriter out, LogTable.ManagedColumn mc ) throws XMLStreamException {
        out.writeStartElement("column");
        
        out.writeAttribute("name", mc.getName());
        out.writeAttribute("displayIndex", Integer.toString(mc.getDisplayIndex()) );
        out.writeAttribute("width", Integer.toString(mc.getWidth()) );
        out.writeAttribute("visible", Boolean.toString(mc.isVisible()) );
        
        out.writeEndElement();
    }
    
    private void writeColumns( XMLStreamWriter out, Iterable<LogTable.ManagedColumn> mcs ) throws XMLStreamException {
        out.writeStartElement("columns");
        
        List<LogTable.ManagedColumn> mcList = new ArrayList<>();
        for( LogTable.ManagedColumn mc : mcs ){
            mcList.add(mc);
        }
        Collections.sort(mcList, new Comparator<LogTable.ManagedColumn>() {
            @Override
            public int compare(LogTable.ManagedColumn mc1, LogTable.ManagedColumn mc2) {
                return Objects.compare(mc1, mc2, new Comparator<LogTable.ManagedColumn>(){
                    @Override
                    public int compare(LogTable.ManagedColumn c1, LogTable.ManagedColumn c2) {
                        int di1 = c1.getDisplayIndex();
                        int di2 = c2.getDisplayIndex();

                        String n1 = c1.getName();
                        String n2 = c2.getName();

                        if( di1!=di2 )return di1 - di2;
                        return n1.compareTo(n2);
                    }
                });
            }
        });
        
        for( LogTable.ManagedColumn mc : mcList ){
            if( mc==null )continue;
            writeColumn(out, mc);
        }
        
        out.writeEndElement();
    }
    
    private Map<Integer,Set<LogTable.ManagedColumn>> columnsOrder = new TreeMap<>();
    private Map<LogTable.ManagedColumn,Integer> columnsWidth = new LinkedHashMap<>();
    
    private void readColumnStart( LogHandlerPanel lp, XEventPath xpath ){
        String name = xpath.getAttributeValue("name");
        if( name==null ){
            return;
        }
        
        LogTable.ManagedColumn mc = null;
        for( LogTable.ManagedColumn c : lp.getLogTable().getManagedColumns() ){
            if( c==null || c.getName()==null )continue;
            if( c.getName().equals(name) ){
                mc = c;
                break;
            }
        }
        if( mc==null )return;
        
        boolean visible = xpath.readAttributeAsBoolean("visible", false);
        if( !visible )return;
        mc.setVisible(true);

        int width = xpath.readAttributeAsInteger("width", -1);
        if( width>=1 ){
            //mc.setWidth(width);
            columnsWidth.put(mc, width);
        }
        
        int di = xpath.readAttributeAsInteger("displayIndex", -1);
        Set<LogTable.ManagedColumn> scols = columnsOrder.get(di);
        if( scols==null ){
            scols = new LinkedHashSet<>();
            columnsOrder.put(di, scols);
        }
        scols.add(mc);
    }
    
    private void readColumnsFinish(){
        int di = -1;
        for( Set<LogTable.ManagedColumn> mcset : columnsOrder.values() ){
            for( LogTable.ManagedColumn mc : mcset ){
                di++;
                mc.setDisplayIndex(di);
            }
        }
        
        for( LogTable.ManagedColumn mc : columnsWidth.keySet() ){
            mc.setWidth(columnsWidth.get(mc));
        }
    }
    
    private void readColumnsStart(){
        columnsOrder.clear();
        columnsWidth.clear();
    }
}
