/*
 * The MIT License
 *
 * Copyright 2016 nt.gocha@gmail.com.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.menu;


import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.Action;
import javax.swing.KeyStroke;
import xyz.cofe.collection.graph.GraphTools;
import xyz.cofe.collection.graph.Path;
import xyz.cofe.collection.graph.SimpleSDGraph;
import xyz.cofe.collection.graph.SingleDirectedGraph;
import xyz.cofe.gui.swing.KeyboardShortcut;

/**
 * Созданию меню через рефлексию
 * @author nt.gocha@gmail.com
 */
public class ReflectMenuBuilder {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(ReflectMenuBuilder.class.getName());
    private static final Level logLevel = logger.getLevel();
    private static final boolean isLogSevere =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();

    private static final boolean isLogWarning =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();

    private static final boolean isLogInfo =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();

    private static final boolean isLogFine =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();

    private static final boolean isLogFiner =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();

    private static final boolean isLogFinest =
        logLevel==null
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }

    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }

    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="menu">
    protected MenuItem menu;
    
    /**
     * Меню с котороым производятся манипуляции
     * @return меню
     */
    public MenuItem getMenu() {
        if( menu!=null )return menu;
        MenuContainer mc = new MenuContainer();
        menu = mc;
        return menu;
    }
    
    /**
     * Меню с котороым производятся манипуляции
     * @param menu меню
     */
    public void setMenu( MenuItem menu ) {
        this.menu = menu;
    }
    //</editor-fold>

    /**
     * Регистрация новых "действий"
     * @param action действие
     * @param act описение действия
     */
    protected void registerNewAction( Action action, MenuAction act ){
    }
    
    /**
     * Создание родительского меню
     * @param act описение действия
     * @return контейнер-меню
     */
    protected MenuContainer createMenuContainer( MenuAction act ){
        String menuId = act.menuContainerId();
        MenuContainer miFound = new MenuContainer(menuId);
        miFound.setId(menuId);
        return miFound;
    }
    
    /**
     * Создание пункта меню
     * @param action действие вызваемое пунктом меню
     * @param act Описание пункта меню
     * @return Действие вызываемое пунктом меню
     */
    protected MenuActionItem createMenuActionItem( Action action, MenuAction act  )
    {
        MenuActionItem ma = new MenuActionItem(action);
        String menuID = act.menuId();
        if( menuID!=null && menuID.length()>0 )ma.setId(menuID);
        return ma;
    }

    /**
     * Иследует объект на предмет действий Action. 
     * 
     * <p>
     * Для методов помеченных аннотацией MenuAction будут созданы обработчики меню
     * @param obj объект с методами меню
     * @see MenuAction
     * @see MethodCallAction
     */
    public void inspect( Object obj ){
        if( obj==null )return;
        
        Map<MenuItem,MenuItem> assignMenuItems = new LinkedHashMap<>();
        final Map<MenuItem,Integer> order = new LinkedHashMap<>();
        final Map<String,MenuItem> menuById = new LinkedHashMap();
        final Map<MenuItem,String> menuAfter = new LinkedHashMap();
        final int defOrder = 0;

        MenuItem miRoot = getMenu();
        
        Method[] methods = obj.getClass().getMethods();
        for( Method m : methods ){
            MenuAction act = m.getAnnotation(MenuAction.class);
            if( act==null )continue;

            MethodCallAction metCallAct = createAction(obj, m, act);
            if( metCallAct==null )continue;
            
            String menuContainerId = act.menuContainerId();
            if( menuContainerId!=null && menuContainerId.length()>0 
                && miRoot!=null 
            ){
                List<MenuItem> li = MenuItem.findMenuItemById(miRoot, menuContainerId);
                if( li==null || li.isEmpty() ){
                    for( MenuItem mi : assignMenuItems.keySet() ){
                        if( menuContainerId.equals( mi.getId() ) ){
                            li = new ArrayList<MenuItem>();
                            li.add(mi);
                            break;
                        }
                    }
                }
                
                if( li==null || li.isEmpty() ){
                    MenuItem miFound = createMenuContainer(act);

                    li = new ArrayList<MenuItem>();
                    li.add(miFound);

//                    MenuItem.addChild(miRoot, miFound);
                    assignMenuItems.put(miFound, miRoot);
                }
                
                for( MenuItem mic : li ){
                    MenuActionItem ma = createMenuActionItem( metCallAct, act );
//                    MenuItem.addChild(mic, ma);
                    assignMenuItems.put(ma, mic);
                    
                    order.put(ma, act.menuOrder());
                    
                    String mid = act.menuId();
                    if( mid!=null && mid.length()>0 )menuById.put(mid, ma);
                    
                    String after = act.menuOrderAfter();
                    if( after!=null && after.length()>0 )menuAfter.put(ma, after);
                }
            }
        }
        
        // build ordered map
        Map<MenuItem,List<MenuItem>> orderdMap = new LinkedHashMap<MenuItem, List<MenuItem>>();
        
        for( Map.Entry<MenuItem,MenuItem> me : assignMenuItems.entrySet() ){
            MenuItem child = me.getKey();
            MenuItem prnt = me.getValue();
            
            List<MenuItem> children = orderdMap.get(prnt);
            if( children==null ){
                children = new ArrayList<MenuItem>();
                orderdMap.put(prnt, children);
            }
            children.add(child);
        }
        
        SingleDirectedGraph<String,String> orderRefs = new SimpleSDGraph<String, String>();
        Map<Integer,Set<MenuItem>> orderRefsResolv = new TreeMap<Integer, Set<MenuItem>>();
        for( MenuItem mi : menuAfter.keySet() ){
            String mid = mi.getId();
            if( mid==null )continue;
            
            String refMid = menuAfter.get(mi);
            if( refMid==null )continue;
            
            orderRefs.set(mid, refMid, mid);
        }
        for( String mid : orderRefs.getNodes() ){
            MenuItem mi = menuById.get(mid);
            if( mi==null )continue;
            
            int len = GraphTools.getMaxPathLength(orderRefs, mid, Path.Direction.AB);
            if( len<0 )continue;
            
            Set<MenuItem> sm = orderRefsResolv.get(len);
            if( sm==null ){
                sm = new LinkedHashSet<MenuItem>();
                orderRefsResolv.put(len, sm);
            }
            sm.add(mi);
        }
        
        for( Integer ord : orderRefsResolv.keySet() ){
            Set<MenuItem> sm = orderRefsResolv.get(ord);
            for( MenuItem mi : sm ){
//        for( MenuItem mi : menuAfter.keySet() ){
                String refMI = menuAfter.get(mi);
                if( refMI!=null && menuById.containsKey(refMI) ){
                    int refOrder = order.get(menuById.get(refMI));
                    order.put(mi, refOrder+1);
                }
            }
        }
        
        for( Map.Entry<MenuItem,List<MenuItem>> me : orderdMap.entrySet() ){
            List<MenuItem> children = me.getValue();
            if( children.isEmpty() )continue;
            
            Collections.sort(children, new Comparator<MenuItem>() {
                @Override
                public int compare(MenuItem mi1, MenuItem mi2) {
                    int o1 = order.containsKey(mi1) ? order.get(mi1) : defOrder;
                    int o2 = order.containsKey(mi2) ? order.get(mi2) : defOrder;
                    return o1==o2 ? 0 : (o1 < o2 ? -1 : 1);
                }
            });
        }
        
        // attach children
        for( MenuItem prnt : orderdMap.keySet() ){
            for( MenuItem chld : orderdMap.get(prnt) ){
                MenuItem.addChild(prnt, chld);
            }
        }
    }

    /**
     * Создает MethodCallAction из метода.
     * 
     * <p>
     * Меод должен быть:
     * <ul>
     * <li>не статичиный и содержать ссылку на владельца метода
     * <li>может быть статичным, тогда допускается отсуствие ссылки на владельца метода
     * <li>Должен иметь 0 аргументов
     * <li>Может иметь 1 аргумент, он будет использоваться в качестве целевого параметра для фокуса ввода
     * </ul>
     * @param owner владелец метода
     * @param method метод
     * @param act аннотация меню
     * @return Действие для меню
     */
    public MethodCallAction createAction(
        final Object owner,
        final java.lang.reflect.Method method,
        final MenuAction act )
    {
        if( act==null )return null;
        if( method==null )return null;
        if( !Modifier.isStatic(method.getModifiers()) && owner==null )return null;

        String actName = act.name();
        if( actName==null || actName.length()==0 )
            actName = method.getName();

        Class cTarget = act.target();
        Class paramTarget = null;

        Class[] params = method.getParameterTypes();
        if( params.length!=0 ){
            if( params.length==1 ){
                paramTarget = params[0];
                cTarget = paramTarget;
            }else{
                return null;
            }
        }

        MethodCallAction ba = new MethodCallAction(actName, owner, method);
//        ba.setMenuAction(act);

        Shortcut[] shortcuts = act.shortcuts();
        if( shortcuts!=null ){
            if( shortcuts.length>1 ){
                for( Shortcut sc : shortcuts ){
                    if( sc.keys()!=null && sc.keys().length()>0 ){
                        KeyStroke ks = KeyStroke.getKeyStroke(sc.keys());
                        if( ks==null )continue;

                        KeyboardShortcut ksc = new KeyboardShortcut();
                        ksc.setKeyStroke(ks);

                        Class c = paramTarget!=null ? paramTarget : sc.target();
                        if( c!=null && !c.equals(Void.class) ){
                            ksc.setTarget(c);
                        }

                        ba.getKeyboardShortcuts().add(ksc);
                    }
                }
            }else if( shortcuts.length==1 && shortcuts[0]!=null ){
                Shortcut sc = shortcuts[0];
                if( sc.keys()!=null && sc.keys().length()>0 ){
                    KeyStroke ks = KeyStroke.getKeyStroke(sc.keys());
                    ba.setAccelerator(ks);

                    Class c = paramTarget!=null ? paramTarget : sc.target();
                    if( c!=null && !c.equals(Void.class) ){
                        ba.setTarget(c);
                    }
                }
            }
        }

        if(
            (cTarget!=null && !cTarget.equals(Void.class))
        ){
            ba.setTarget(cTarget);
        }

        registerNewAction(ba, act);
        return ba;
    }
}
