/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.properties.editor;


import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JOptionPane;
import javax.swing.border.EmptyBorder;
import javax.swing.text.JTextComponent;
import xyz.cofe.collection.Predicate;
import xyz.cofe.gui.swing.properties.PropertyDB;
import xyz.cofe.gui.swing.properties.PropertyDBService;
import xyz.cofe.gui.swing.text.ValidatedTextField;

/**
 * Редактор Date значений
 * @author user
 */
public class DateEditor extends TextFieldEditor implements PropertyDBService {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(DateEditor.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(DateEditor.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(DateEditor.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(DateEditor.class.getName(), method, result);
    }
    //</editor-fold>

    public DateEditor(boolean allowNull) {
        super(allowNull);
    }

    public DateEditor(boolean allowNull, List<DateFormat> dateFormats, DateFormat prefferedDateFormat) {
        super(allowNull);
        this.dateFormats = dateFormats;
        this.preffredDateFormat = prefferedDateFormat;
    }
    
    public DateEditor(boolean allowNull, DateFormat prefferedDateFormat) {
        super(allowNull);
        this.preffredDateFormat = prefferedDateFormat;
    }
    
    public DateEditor(DateEditor sample) {
        super(sample);
        if( sample!=null ){
            this.baloonText = sample.baloonText;
            this.dateFormats = new ArrayList<>();
            if( sample.dateFormats!=null ){
                for( DateFormat df : sample.dateFormats ){
                    DateFormat cdf = (DateFormat)df.clone();
                    this.dateFormats.add( cdf );
                }
            }
            if( sample.preffredDateFormat!=null ){
                this.preffredDateFormat = (DateFormat)sample.preffredDateFormat.clone();
            }
        }
    }
    
    public DateEditor(){
        super(true);
    }

    @Override
    public DateEditor clone(){
        return new DateEditor(this);
    }
    
    public static DateEditor getDateEditor(boolean nullable, String prefFormat){
        return new DateEditor(nullable,new SimpleDateFormat(
            prefFormat!=null ? prefFormat : "yyyy-MM-dd HH:mm:ss.SSSXXX"
        ));
    }

    public static DateEditor getSQLTimeEditor(boolean nullable, String prefFormat){
        return new DateEditor(nullable,new SimpleDateFormat(
            prefFormat!=null ? prefFormat : "yyyy-MM-dd HH:mm:ss.SSSXXX"
        )){
            @Override
            protected Date createDate(long t) {
                return new java.sql.Time(t);
            }
        };
    }

    public static DateEditor getSQLTimestampEditor(boolean nullable, String prefFormat){
        return new DateEditor(nullable,new SimpleDateFormat(
            prefFormat!=null ? prefFormat : "yyyy-MM-dd HH:mm:ss.SSSXXX")){
            @Override
            protected Date createDate(long t) {
                return new java.sql.Timestamp(t);
            }
        };
    }

    public static DateEditor getSQLDateEditor(boolean nullable, String prefFormat){
        return new DateEditor(nullable,new SimpleDateFormat(
            prefFormat!=null ? prefFormat : "yyyy-MM-dd")){
            @Override
            protected Date createDate(long t) {
                return new java.sql.Date(t);
            }
        };
    }

    @Override
    public void register(PropertyDB pdb) {
        if( pdb==null )return;
        
        pdb.registerTypeEditor(Date.class, this);
        pdb.registerTypeEditor(java.sql.Time.class, getSQLTimeEditor(true, null));
        pdb.registerTypeEditor(java.sql.Timestamp.class, getSQLTimestampEditor(true, null));
        pdb.registerTypeEditor(java.sql.Date.class, getSQLDateEditor(true, null));
    }
    
    protected Date createDate(long t){
        return new Date(t);
    }
    
    private List<DateFormat> dateFormats;
    public synchronized List<DateFormat> getDateFormats(){
        if( dateFormats!=null )return dateFormats;
        dateFormats = new ArrayList<>();
        dateFormats.add(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSSXXX"));
        dateFormats.add(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSSZ"));
        dateFormats.add(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
        dateFormats.add(new SimpleDateFormat("yyyy-MM-dd HH:mm"));
        dateFormats.add(new SimpleDateFormat("yyyy-MM-dd"));
        dateFormats.add(new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss"));
        dateFormats.add(new SimpleDateFormat("yyyyMMdd'T'HHmmss"));
        dateFormats.add(new SimpleDateFormat("yyyyMMdd'T'HHmm"));
        
        StringBuilder sb = new StringBuilder();
        sb.append("Supported date formats:\n");
        for( Object o : dateFormats ){
            if( o instanceof SimpleDateFormat ){
                SimpleDateFormat sdf = (SimpleDateFormat)o;
                sb.append(sdf.toPattern()).append("\n");
            }
        }
        baloonText = sb.toString();
        
        return dateFormats;
    }
    
    protected Date parse( String txt ){
        if( txt==null )return null;
        
        txt = txt.trim();
        if( txt.length()<1 )return null;
        
        DateFormat dfPref = getPreffredDateFormat();
        if( dfPref!=null ){
            try{
                Date d = dfPref.parse(txt);
                if( d!=null ){
                    return createDate(d.getTime());
                }
            }catch( ParseException ex ){
                logFiner("can't parse {0} with format {1}", txt, dfPref);
            }
        }
        
        List<DateFormat> dfs = getDateFormats();
        if( dfs.isEmpty() )return null;
        
        for( DateFormat df : dfs ){
            try{
                Date d = df.parse(txt);
                if( d!=null ){
                    return createDate(d.getTime());
                }
            }catch( ParseException ex ){
                logFiner("can't parse {0} with format {1}", txt, df);
            }
        }
        
        return null;
    }
    
    protected String baloonText;
    protected String getBaloonText(){
        if( baloonText!=null )return baloonText;
        baloonText = "Date formats";
        return baloonText;
    }
    
    protected DateFormat preffredDateFormat;
    public DateFormat getPreffredDateFormat() { 
        if( preffredDateFormat==null ){
            List<DateFormat> ldf = getDateFormats();
            if( ldf!=null && !ldf.isEmpty() ){
                return ldf.get(0);
            }else{
                return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSSXXX");
            }
        }
        return preffredDateFormat; 
    }
    public void setPreffredDateFormat(DateFormat preffredDateFormat) { this.preffredDateFormat = preffredDateFormat; }
    
    @Override
    protected JTextComponent getTextField() {
        if( textField!=null )return textField;
        
        final ValidatedTextField tf = new ValidatedTextField();
        
        Predicate<String> parseFilter = //(String txt) -> {
            new Predicate<String>() {
            @Override
            public boolean validate(String txt) {
                Date d = parse(txt);
                if( d==null ){
                    tf.setBalloonText(getBaloonText());
                    return false;
                }
                return true;
            }
        };
        
        tf.setFilter(parseFilter);
        tf.setPlaceholder("Enter date");
        tf.setBalloonText("");
        tf.setBorder(new EmptyBorder(0, 0, 0, 0));
        
        textField = tf;
        return textField;
    }
    
    @Override
    protected Object getTextFieldValue() {
        String txt = getTextField().getText();

        Date d = parse(txt);
        if( d==null ){
            JOptionPane.showMessageDialog( null,
                getBaloonText(),
                "Date format error",
                JOptionPane.ERROR_MESSAGE
            );
            
            return null;
        }
        
        return d;
    }

    @Override
    protected void setTextFieldValue(Object value) {
        if( value instanceof Date ){
            //super.setTextFieldValue(Double.toString((Double)value));
            DateFormat df = getPreffredDateFormat();
            if( df!=null ){
                super.setTextFieldValue(df.format((Date)value));
                return;
            }
        }
        super.setTextFieldValue(value);
    }
}
