/*
 * The MIT License
 *
 * Copyright 2018 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.properties.editor;


import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.border.EmptyBorder;
import javax.swing.text.JTextComponent;
import xyz.cofe.collection.Predicate;
import xyz.cofe.gui.swing.properties.PropertyDB;
import xyz.cofe.gui.swing.properties.PropertyDBService;
import xyz.cofe.gui.swing.text.ValidatedTextField;
import xyz.cofe.time.TimeDiff;

/**
 * Тексовый редактор для временных участков
 * @author user
 */
public class TimeDiffEditor extends TextFieldEditor implements PropertyDBService {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(TimeDiffEditor.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(TimeDiffEditor.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(TimeDiffEditor.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(TimeDiffEditor.class.getName(), method, result);
    }
    //</editor-fold>

    /**
     * Конструктор
     * @param allowNull true - допускается null значения
     */
    public TimeDiffEditor(boolean allowNull) {
        super(allowNull);
    }

    /**
     * Конструктор копирования
     * @param sample образец для копиования
     */
    public TimeDiffEditor(TimeDiffEditor sample) {
        super(sample);
    }

    /**
     * Конструктор
     */
    public TimeDiffEditor(){
        super(false);
    }

    @Override
    public void register(PropertyDB pdb) {
        if(pdb!=null){
            pdb.registerTypeEditor(TimeDiff.class, this);
        }
    }
    
    @Override
    public TimeDiffEditor clone(){
        return new TimeDiffEditor(this);
    }
    
    @Override
    protected JTextComponent getTextField() {
        if( textField!=null )return textField;
        
        final ValidatedTextField tf = new ValidatedTextField();
        
        Predicate<String> parseFilter = //(String txt) -> {
            new Predicate<String>() {
            @Override
            public boolean validate(String txt) {
                if( txt==null )return true;
                
                TimeDiff tdiff = TimeDiff.parse(txt);
                if( tdiff==null ){
                    if( isAllowNull() )return true;
                    tf.setBalloonText("format exception, enter like this: +2d 10:34:53.234");
                    return false;
                }
                
                return true;
            }};
        
        tf.setFilter(parseFilter);
        tf.setPlaceholder("Enter number time diff like: +2d 10:34:53.234");
        tf.setBalloonText("");
        tf.setBorder(new EmptyBorder(0, 0, 0, 0));
        
        textField = tf;
        return textField;
    }
    
    @Override
    protected Object getTextFieldValue() {
        String text = getTextField().getText();
        TimeDiff tdiff = TimeDiff.parse(text);
        return tdiff;
    }
    
    @Override
    protected void setTextFieldValue(Object value) {
        super.setTextFieldValue(value);
    }
}
