/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.gui.swing.table;

import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Convertor;
import xyz.cofe.collection.Func1;

/**
 * Колонка таблицы
 * @author gocha
 */
public class Column 
    implements GetReaderForRow
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(Column.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(Column.class.getName()).log(Level.FINER, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(Column.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(Column.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(Column.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(Column.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    protected final Object sync;
    
    /**
     * Конструктор
     */
    public Column(){
        sync = this;
    }
    
    /**
     * Конструктор
     * @param sync объект для синхронизации
     */
    public Column( Object sync ){
        this.sync = sync==null ? this : sync;
    }
    
    /**
     * Конструктор копирования
     * @param src образец для копирования
     */
    public Column( Column src ){
        sync = this;
        if( src!=null ){
            this.name = src.name;
            this.type = src.type;
            this.reader = src.reader;
            this.writer = src.writer;
        }
    }
    
    /**
     * Конструктор
     * @param sync объект для синхронизации, если null - то используется this
     * @param src образец для копирования
     */
    public Column( Object sync, Column src ){
        this.sync = sync==null ? this : sync;
        if( src!=null ){
            synchronized(src.sync){
                this.name = src.name;
                this.type = src.type;
                this.reader = src.reader;
                this.writer = src.writer;
            }
        }
    }
    
    @Override
    public Column clone(){
        return new Column(this);
    }
    
    /**
     * Клонирование с указаным объектом синхронизации
     * @param sync объект для синхронизации
     * @return клон
     */
    public Column cloneWith( Object sync ){
        return new Column(sync, this);
    }

    // <editor-fold defaultstate="collapsed" desc="name">
    /**
     * имя колонки
     */
    protected String name = "column";
    public static final String NAME = "name";

    /**
     * Имя колонки
     * @return имя колонки
     */
    public String getName() {
        synchronized(sync){ return name; }
    }

    /**
     * Имя колонки
     * @param name имя колонки
     */
    public void setName(String name) {
        Object old;
        synchronized(sync){
        old = this.name;
        this.name = name;
        }
        firePropertyChange(NAME, old, this.name);
    }
    
    /**
     * Имя колонки
     * @param name имя колонки
     * @return this ссылка
     */
    public Column name( String name ){
        setName(name);
        return this;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="reader">
    protected Convertor<Object,Object> reader = null;
    public static final String READER = "reader";

    /**
     * Чтение значения ячейки. <br>
     * Вызывается так: <code>column.getReader().convert( Элемент списка )</code> - должен вернуть значение элемента для соответ. колонки.
     * @return чтение значения ячейки
     */
    public Convertor<Object, Object> getReader() {
        synchronized(sync){
            return reader;
        }
    }
    
    /**
     * Чтение значения ячейки в зависимости от номера строки
     */
    protected Func1<Convertor<Object,Object>,Integer> rowReader;

    /**
     * Чтение значения ячейки. <br>
     * Вызывается так: <code>column.getReader( row ).convert( Элемент списка )</code> - должен вернуть значение элемента для соответ. колонки.
     * <br><br>
     * 
     * В реализации по умолчанию - исполюуется метод getReader(), если функция rowReader равна нулю или вернула null.
     * @return чтение значения ячейки
     * @see #getReader() 
     */
    @Override
    public Convertor<Object, Object> getReader(int row) {
        synchronized(sync){
            if( rowReader==null )return getReader();
            
            Convertor conv = rowReader.apply(row);
            if( conv==null )return getReader();
            
            return conv;
        }
    }
    
    /**
     * Чтение значения ячейки. <br>
     * Вызывается так: <code>column.getReader().convert( Элемент списка )</code> - должен вернуть значение элемента для соответ. колонки.
     * @param reader чтение значения ячейки
     */
    public void setReader(Convertor<Object, Object> reader) {
        Object old;
        synchronized(sync){
        old = this.reader;
        this.reader = reader;
        }
        firePropertyChange(READER, old, this.reader);
    }
    
    /**
     * Чтение значения ячейки. <br>
     * Вызывается так: <code>column.getReader().convert( Элемент списка )</code> - 
     * должен вернуть значение элемента для соответ. колонки.
     * @param reader чтение значения ячейки
     * @return this ссылка
     */
    public Column reader(Convertor<Object, Object> reader) {
        setReader(reader);
        return this;
    }
    
    /**
     * Чтение значения ячейки. <br>
     * Вызывается так: <code>column.getRowReader( row ).convert( Элемент списка )</code> - 
     * должен вернуть значение элемента для соответ. колонки.
     * @param rowReader Функция возвращ. функцию чтения
     * @return this ссылка
     */
    public Column rowReader(Func1<Convertor<Object,Object>,Integer> rowReader){
        synchronized(sync){
            this.rowReader = rowReader;
        }
        return this;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="writer">
    /**
     * Записываемое значение
     */
    public static class Cell{
        /**
         * Элемент списка
         */
        public Object object = null;
        
        /**
         * Записываемое значение
         */
        public Object newValue = null;

        /**
         * Конструктор по умолчанию
         */
        public Cell(){
        }
        
        public Cell(Object object,Object newValue){
            this.object = object;
            this.newValue = newValue;
        }
    }

    public static final String WRITER = "writer";
    
    /**
     * Функция записи
     */
    protected Convertor<Cell,Boolean> writer = null;

    /**
     * Запись значения ячейки. <br>
     * Вызывается так: <font style="font-family:monospaced"> getWriter().convert( new Cell( Элемент списка, Новое значение ячейки) ) </font>
     * @return true - записано, false - не записано
     */
    public Convertor<Cell, Boolean> getWriter() {
        synchronized(sync){ return writer; }
    }

    /**
     * Запись значения ячейки. <br>
     * Вызывается так: <font style="font-family:monospaced"> getWriter().convert( new Cell( Элемент списка, Новое значение ячейки) ) </font>
     * @param writer Функция записи
     */
    public void setWriter(Convertor<Cell, Boolean> writer) {
        Object old;
        synchronized(sync){
        old = this.writer;
        this.writer = writer;
        }
        firePropertyChange(WRITER, old, this.writer);
    }
    
    /**
     * Запись значения ячейки. <br>
     * Вызывается так: <font style="font-family:monospaced"> getWriter().convert( new Cell( Элемент списка, Новое значение ячейки) ) </font>
     * @param writer  Функция записи
     * @return this ссылка
     */
    public Column writer(Convertor<Cell, Boolean> writer){
        setWriter(writer);
        return this;
    }
    // </editor-fold>

    public static final String SOURCE_COLUMN="sourceColumn";

    // <editor-fold defaultstate="collapsed" desc="type">
    public static final String TYPE="type";
    protected Class type = String.class;

    /**
     * Тип данных колонки
     * @return Тип данных колонки
     */
    public Class getType() {
        synchronized(sync){ return type; }
    }

    /**
     * Тип данных колонки
     * @param type Тип данных колонки
     */
    public void setType(Class type) {
        Object old;
        synchronized(sync){
        old = this.type;
        this.type = type;
        }
        firePropertyChange(TYPE, old, type);
    }
    
    /**
     * Тип данных колонки
     * @param type Тип данных колонки
     * @return this ссылка
     */
    public Column type(Class type){
        setType(type);
        return this;
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="propertyChangeSupport">
    /**
        * Поддержка PropertyChangeEvent
        */
    private volatile transient java.beans.PropertyChangeSupport propertyChangeSupport = null;
    /**
    * Поддержка PropertyChangeEvent
     * @return Поддержка PropertyChangeEvent
    */
    protected java.beans.PropertyChangeSupport propertySupport(){
        if( propertyChangeSupport!=null )return propertyChangeSupport;
        synchronized(sync){
            if( propertyChangeSupport!=null )return propertyChangeSupport;
            propertyChangeSupport = new java.beans.PropertyChangeSupport(this);
        }
        return propertyChangeSupport;
    }

    /**
    * Уведомляет подписчиков о измении свойства
    * @param property Свойство
    * @param oldValue Старое значение
    * @param newValue Новое значение
    */
    protected void firePropertyChange(String property,Object oldValue, Object newValue){
        propertySupport().firePropertyChange(property, oldValue, newValue);
    }

    /**
    * Добавляет подписчика
    * @param listener Подписчик
    */
    public void addPropertyChangeListener(java.beans.PropertyChangeListener listener )
    {
        propertySupport().addPropertyChangeListener( listener );
    }

    /**
    * Удаляет подписчика
    * @param listener Подписчик
    */
    public void removePropertyChangeListener(java.beans.PropertyChangeListener listener )
    {
        propertySupport().removePropertyChangeListener( listener );
    }
    // </editor-fold>
}

