/*
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"),
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ,
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ,
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.gui.swing.table;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.event.SwingPropertyChangeSupport;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;
import xyz.cofe.collection.Convertor;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.Predicate;
import xyz.cofe.collection.list.EventList;
import xyz.cofe.collection.list.IndexEventList;
import xyz.cofe.collection.list.SimpleListAdapter;
import xyz.cofe.common.CloseableSet;

/**
 * Модель таблицы основанная на списке объектов
 * @author nt.gocha@gmail.com
 * @param <E> Тип элемента списка
 */
public class ListTM<E> implements TableModel {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(ListTM.class.getName()).log(Level.FINE, message, args);
    }

    private static void logFiner(String message,Object ... args){
        Logger.getLogger(ListTM.class.getName()).log(Level.FINER, message, args);
    }

    private static void logInfo(String message,Object ... args){
        Logger.getLogger(ListTM.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(ListTM.class.getName()).log(Level.WARNING, message, args);
    }

    private static void logSevere(String message,Object ... args){
        Logger.getLogger(ListTM.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(ListTM.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    protected final Lock lock;

    /**
     * Конструктор
     */
    public ListTM(){
        this.lock = new ReentrantLock();
        if( list!=null ){
            // list.addEventListListener(listAdapter);
            addSourceListeners();
        }
        if( columns!=null )columns.addEventListListener(columnsAdapter);
    }

    // <editor-fold defaultstate="collapsed" desc="propertyChangeSupport">
    /**
     * Поддержка PropertyChangeEvent
     */
    private volatile transient java.beans.PropertyChangeSupport propertyChangeSupport = null;
    
    /**
     * Поддержка PropertyChangeEvent
     * @return Поддержка PropertyChangeEvent
     */
    protected java.beans.PropertyChangeSupport propertySupport(){
        if( propertyChangeSupport!=null )return propertyChangeSupport;
        synchronized(this){
            if( propertyChangeSupport!=null )return propertyChangeSupport;
            
            propertyChangeSupport = //new java.beans.PropertyChangeSupport(this);
                new SwingPropertyChangeSupport(this);
            
            return propertyChangeSupport;
        }
    }

    /**
     * Уведомляет подписчиков о измении свойства
     * @param property Свойство
     * @param oldValue Старое значение
     * @param newValue Новое значение
     */
    protected void firePropertyChange(String property,Object oldValue, Object newValue){
        propertySupport().firePropertyChange(property, oldValue, newValue);
    }

    /**
     * Добавляет подписчика
     * @param listener Подписчик
     */
    public void addPropertyChangeListener(java.beans.PropertyChangeListener listener )
    {
        propertySupport().addPropertyChangeListener( listener );
    }

    /**
     * Удаляет подписчика
     * @param listener Подписчик
     */
    public void removePropertyChangeListener(java.beans.PropertyChangeListener listener )
    {
        propertySupport().removePropertyChangeListener( listener );
    }
    // </editor-fold>

    // <editor-fold defaultstate="collapsed" desc="evSupport">
    protected EventSupport evSupport = new EventSupport(this);
    public EventSupport getEventSupport(){ return evSupport; }

    /**
     * Удаляет подписчика, чтоб он не получал сообщения
     * @param l подписчик
     */
    @Override
    public void removeTableModelListener(TableModelListener l) {
        synchronized(this){ evSupport.removeTableModelListener(l); }
    }

    /**
     * Возвращает список подписчиков
     * @return список подписчиков
     */
    public Collection<TableModelListener> getListenersCollection() {
        synchronized(this){ return evSupport.getListenersCollection(); }
    }

    /**
     * Возвращает список подписчиков
     * @return список подписчиков
     */
    public TableModelListener[] getListeners() {
        synchronized(this){ return evSupport.getListeners(); }
    }

    /**
     * Рассылка уведомления подписчикам
     * @param e уведомление
     */
    public void fireTableModelEvent(TableModelEvent e) {
        synchronized(this){ evSupport.fireTableModelEvent(e); }
    }

    /**
     * Рассылка уведомления подписчикам о обновлении строк
     * @param rowIndexFrom начало диапазона строк
     * @param toIndexInclude конец (включительно) диапазона строк
     */
    public void fireRowsUpdated(int rowIndexFrom, int toIndexInclude) {
        synchronized(this){ evSupport.fireRowsUpdated(rowIndexFrom, toIndexInclude); }
    }

    /**
     * Рассылка уведомления подписчикам о добавлении строк в таблицу
     * @param rowIndexFrom начало диапазона строк
     * @param toIndexInclude конец (включительно) диапазона строк
     */
    public void fireRowsInserted(int rowIndexFrom, int toIndexInclude) {
        synchronized(this){ evSupport.fireRowsInserted(rowIndexFrom, toIndexInclude); }
    }

    /**
     * Рассылка уведомления подписчикам о удалении строк из таблицы
     * @param rowIndexFrom начало диапазона строк
     * @param toIndexInclude конец (включительно) диапазона строк
     */
    public void fireRowsDeleted(int rowIndexFrom, int toIndexInclude) {
        synchronized(this){ evSupport.fireRowsDeleted(rowIndexFrom, toIndexInclude); }
    }

    /**
     * Рассылка уведомления подписчикам о измении строки
     * @param row индекс строки
     */
    public void fireRowUpdated(int row) {
        synchronized(this){ evSupport.fireRowUpdated(row); }
    }

    /**
     * Рассылка уведомления подписчикам
     */
    public void fireColumnsChanged() {
        synchronized(this){ evSupport.fireColumnsChanged(); }
    }

    /**
     * Рассылка уведомления подписчикам о измении ячейки
     * @param rowIndex строка таблицы
     * @param columnIndex колонка таблицы
     */
    public void fireCellChanged(int rowIndex, int columnIndex) {
        synchronized(this){  evSupport.fireCellChanged(rowIndex, columnIndex); }
    }

    /**
     * Рассылка уведомления подписчикам о измении всех данных
     */
    public void fireAllChanged() {
        synchronized(this){ evSupport.fireAllChanged(); }
    }

    /**
     * Добавление подписчика на уведомлении о измении модели таблицы
     * @param l подписчик
     */
    @Override
    public void addTableModelListener(TableModelListener l) {
        synchronized(this){ evSupport.addTableModelListener(l); }
    }
    // </editor-fold>
    
    protected final CloseableSet listListeners = new CloseableSet();
    
    protected void addSourceListeners(){
        if( list==null )return;
        
        Closeable cl;
        
        cl = list.onInserted(new Func3<Object, Integer, E, E>() {
            @Override
            public Object apply(Integer idx, E oldv, E newv) {
                onInserted(newv, idx);
                return null;
            }
        });

        /*
        cl =
        list.onInserted( (idx,oldv,newv) -> {
            onInserted(newv, idx);
            return null;
        }, true); */

        listListeners.add( cl );

        /*cl =
        list.onUpdated( (idx,oldv,newv) -> {
            onUpdated(oldv, newv, idx);
            return null;
        }, true);*/
        
        cl = list.onUpdated(new Func3<Object, Integer, E, E>() {
            @Override
            public Object apply(Integer idx, E oldv, E newv) {
                onUpdated(oldv, newv, idx);
                return null;
            }
        });

        listListeners.add( cl );
        
        cl = list.onDeleted(new Func3<Object, Integer, E, E>() {
            @Override
            public Object apply(Integer idx, E oldv, E newv) {
                onDeleted(oldv, idx);
                return null;
            }
        });

        /*cl =
        list.onDeleted( (idx,oldv,newv) -> {
            onDeleted(oldv, idx);
            return null;
        }, true);*/

        listListeners.add( cl );
    }
    
    /**
     * Подписчик событий установлен на оригинал
     * @return true - в текущий момент подписчик на оригинал установлен
     */
    public boolean isSourceListen(){
        try{
            lock.lock();
            if( list==null ){
                return false;
            }
            //return list.containsEventListListener(listAdapter);
            return listListeners.getCloseables().length>0;
        }finally{
            lock.unlock();
        }
    }
    
    /**
     * Установить/сбросить подписчика на события оригинальной TableModel
     * @param listen true - установить/false - сбросить
     */
    public void setSourceListen( boolean listen ){
        boolean old = isSourceListen();
        
        if( listen!=old ){
            try{
                lock.lock();
//                if( listen ){
//                    list.addEventListListener(listAdapter);
//                }else{
//                    list.removeEventListListener(listAdapter);
//                }
                if( listen ){
                    addSourceListeners();
                }else{
                    listListeners.closeAll();
                }
            }finally{
                lock.unlock();
            }
            
            boolean now = isSourceListen();
            if( now!=old ){
                firePropertyChange("sourceListen", old, now);
            }
        }
    }
    
    // <editor-fold defaultstate="collapsed" desc="list">
    protected IndexEventList<E> list = new IndexEventList<E>();

    /**
     * Возвращает исходный список объектов
     * @return список объектов
     */
    public IndexEventList<E> getList() {
        try{
            lock.lock();
            if( list==null ){
                list = new IndexEventList<E>();
                //list.addEventListListener(listAdapter);
                addSourceListeners();
            }
            return list;
        }finally{
            lock.unlock();
        }
    }

    /**
     * Указывает исходный список объектов
     * @param newList список объектов
     */
    public void setList(IndexEventList<E> newList){
        Object oldl = null;
        Object newl = null;

        try{
            lock.lock();
//            if( this.list!=null ){
//                list.removeEventListListener(listAdapter);
//            }
            listListeners.closeAll();

            oldl = this.list;
            newl = newList;
            this.list = newList;

            if( this.list!=null ){
                //this.list.addEventListListener(listAdapter);
                addSourceListeners();
            }
        }finally{
            lock.unlock();
        }

        firePropertyChange("list", oldl, newl);
        fireAllChanged();
    }
    // </editor-fold>

    /**
     * Вызывается когда в исходный список добавляется элемент,
     * генериует событие вставки строки в таблицу
     * @param e Елемент исходного списка
     * @param position индекс списка в которой добавлен исходный элемент
     */
    protected void onInserted(E e, Integer position){
        if( position==null ){
            throw new IllegalArgumentException("position == null");
        }
        fireRowsInserted(position,position);
    }

    /**
     * Вызывается когда в исходном списоке обновляется элемент,
     * генерирует событие обновления строки таблицы
     * @param oldv предыдущий элемент списка
     * @param newv текущий элемент списка
     * @param position индекс списка
     */
    protected void onUpdated(E oldv, E newv, Integer position){
        if( position==null ){
            throw new IllegalArgumentException("position == null");
        }
        fireRowsUpdated(position,position);
    }

    /**
     * Вызывается когда из исходного списка удален элемент,
     * генериует собтиые удаления строки из таблицы
     * @param e удаленный элемент
     * @param position индекс удаленного элемента
     */
    protected void onDeleted(E e, Integer position){
        if( position==null ){
            throw new IllegalArgumentException("position == null");
        }
        fireRowsDeleted(position,position);
    }

    /*protected final SimpleListAdapter listAdapter = new SimpleListAdapter(){
        @Override
        protected void removed(Object e, EventList list, Integer position) {
            if( position==null )throw new IllegalStateException(
                "Не указан индекс удаленного элемента"
            );
            fireRowsDeleted(position,position);
        }

        @Override
        protected void added(Object e, EventList list, Integer position) {
            if( position==null ){
                position = list.indexOf(e);
            }
            fireRowsInserted(position,position);
        }
    };*/

    protected final SimpleListAdapter<Column> columnsAdapter = new SimpleListAdapter<Column>(){
        @Override
        protected void removed(Column e, EventList<Column> list, Integer position) {
            if( e!=null )e.removePropertyChangeListener(columnPropertiesListener);
            fireColumnsChanged();
        }

        @Override
        protected void added(Column e, EventList<Column> list, Integer position) {
            if( e!=null )e.addPropertyChangeListener(columnPropertiesListener);
            fireColumnsChanged();
        }
    };

    protected PropertyChangeListener columnPropertiesListener = new PropertyChangeListener() {
        @Override
        public void propertyChange(PropertyChangeEvent evt) {
            if( Column.NAME.equals( evt.getPropertyName() ) ){
                fireColumnsChanged();
            }else if( Column.TYPE.equals(evt.getPropertyName()) ){
                fireColumnsChanged();
            }else if( Column.SOURCE_COLUMN.equals(evt.getPropertyName()) ){
                fireAllChanged();
            }else if( Column.READER.equals(evt.getPropertyName()) ){
                fireAllChanged();
            }
        }
    };

    /**
     * Возвращает элемент списка по индексу строки
     * @param index индекс строки таблицы
     * @return элемент списка
     */
    public E getItemByIndex(int index){
        try{
            lock.lock();

            if( list==null )return null;
            if( index<0 )return null;
            if( index>=list.size() )return null;

            return list.get(index);
        }finally{
            lock.unlock();
        }
    }

    /**
     * Возвращает индекст строки таблицы соответ элементу
     * @param item искомый элемент
     * @return индекс строки или null
     */
    public int getIndexOfItem(E item){
        int res = -1;
        try{
            lock.lock();
            if( list==null )return -1;
            res = list.indexOf(item);
        }finally{
            lock.unlock();
        }
        return res;
    }

    /**
     * Возвращает индексы строк соответствующие фильтру элементов исходного списка
     * @param pred фильтр
     * @return индексы строк
     */
    public List<Integer> getIndexesOfItem(Predicate<E> pred){
        if( pred==null ) throw new IllegalArgumentException( "pred==null" );
        List<Integer> res = new ArrayList<Integer>();
        int idx = -1;

        try{
            lock.lock();
            if( list==null )return res;
            for( E o : list ){
                idx++;
                if( pred.validate(o) )
                    res.add( idx );
            }
        }finally{
            lock.unlock();
        }

        return res;
    }

    // <editor-fold defaultstate="collapsed" desc="columns">
    protected Columns columns = new Columns();

    /**
     * Возвращает колонки таблицы
     * @return колонки таблицы
     */
    public Columns getColumns() {
        return columns;
    }
    // </editor-fold>

    @Override
    public int getRowCount() {
        try{
            lock.lock();
            if( list==null )return 0;
            return list.size();
        }finally{
            lock.unlock();
        }
    }

    @Override
    public int getColumnCount() {
        if( columns==null )return 0;
        return columns.size();
    }

    @Override
    public String getColumnName(int columnIndex) {
        if( columns==null || columnIndex<0 || columnIndex>=columns.size() )return "?";
        return columns.get(columnIndex).getName();
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        if( columns==null || columnIndex<0 || columnIndex>=columns.size() )
            return Object.class;

        return columns.get(columnIndex).getType();
    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        try{
            lock.lock();
            if( list==null )return false;
            if( columns==null )return false;

            if( rowIndex<0 )return false;
            if( rowIndex>=list.size() )return false;

            if( columnIndex<0 )return false;
            if( columnIndex>=columns.size() )return false;

            Column col = columns.get(columnIndex);
            if( col.getWriter()!=null ){
                Object v = list.get(rowIndex);
                if( col instanceof IsRowEditable ){
                    return ((IsRowEditable)col).isRowEditable(v);
                }
                return true;
            }else{
                return false;
            }
        }finally{
            lock.unlock();
        }
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        Column c = null;
        Object v = null;

        try{
            lock.lock();

            if( list==null )return null;
            if( columns==null )return null;

            if( rowIndex<0 )return null;
            if( rowIndex>=list.size() )return null;

            if( columnIndex<0 )return null;
            if( columnIndex>=columns.size() )return null;

            v = list.get(rowIndex);
            c = columns.get(columnIndex);
        }finally{
            lock.unlock();
        }

        if( c==null )return null;
        
        Convertor<Object,Object> conv = 
            (c instanceof GetReaderForRow)
            ? ((GetReaderForRow)c).getReader(rowIndex)
            : c.getReader();
        
        if( conv==null )return null;

        return conv.convert(v);
    }

    @Override
    public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
        Column c = null;
        Object v = null;
        
        try{
            lock.lock();
            if( list==null )return;
            if( columns==null )return;

            if( rowIndex<0 )return;
            if( rowIndex>=list.size() )return;

            if( columnIndex<0 )return;
            if( columnIndex>=columns.size() )return;

            v = list.get(rowIndex);
            c = columns.get(columnIndex);
        }finally{
            lock.unlock();
        }

        if( c==null )return;

        Convertor<Column.Cell,Boolean> cellWriter = c.getWriter();
        if( cellWriter==null )return;

        Boolean succ = cellWriter.convert(new Column.Cell(v,aValue));
        if( succ )fireRowsUpdated(rowIndex, rowIndex);
    }
}
