package xyz.cofe.gui.swing.text;

/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

import java.io.Closeable;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import xyz.cofe.collection.BasicPair;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Pair;
import xyz.cofe.common.CloseableSet;
import xyz.cofe.common.ListenersHelper;
import xyz.cofe.common.Reciver;

/**
 * Отслеживание изменений в текстовом документе
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class DocumentChangeTracker 
    implements Closeable
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(DocumentChangeTracker.class.getName());

    private static Level logLevel(){ 
        return logger.getLevel() ;
    }
    
    private static boolean isLogSevere(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.SEVERE.intValue();
    }
    
    private static boolean isLogWarning(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.WARNING.intValue();
    }
    
    private static boolean isLogInfo(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.INFO.intValue();
    }
    
    private static boolean isLogFine(){
        Level level = logLevel();
        return level==null 
        ? true
        : level.intValue() <= Level.FINE.intValue();
    }
    
    private static boolean isLogFiner(){
        Level level = logLevel();
        return level==null 
        ? false
        : level.intValue() <= Level.FINER.intValue();
    }    
    
    private static boolean isLogFinest(){
        Level level = logLevel();
        return level==null 
        ? false
        : level.intValue() <= Level.FINEST.intValue();
    }    
    
    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    
    private static void logEntering(String method,Object ... params){
        logger.entering(DocumentChangeTracker.class.getName(),method,params);
    }

    private static void logExiting(String method,Object result){
        logger.exiting(DocumentChangeTracker.class.getName(),method,result);
    }

    private static void logExiting(String method){
        logger.exiting(DocumentChangeTracker.class.getName(),method);
    }
    //</editor-fold>
    
    protected DocumentChangeTracker( Document doc, boolean weak ){
        if( doc==null )throw new IllegalArgumentException( "doc==null" );
        this.text = readText(doc);
        this.doc = weak ? null : doc;
        this.docRef = weak ? new WeakReference<Document>(doc) : null;
        this.weak = weak;
    }
    
    //<editor-fold defaultstate="collapsed" desc="tracking / onChanges / ...">
    public static DocumentChangeTracker tracking( JTextComponent doc, boolean weak ){
        if( doc==null )throw new IllegalArgumentException( "doc==null" );
        return tracking(doc.getDocument(), weak);
    }
    
    public static DocumentChangeTracker tracking( JTextComponent doc ){
        if( doc==null )throw new IllegalArgumentException( "doc==null" );
        return tracking(doc.getDocument(), false);
    }
    
    public static DocumentChangeTracker tracking( Document doc, boolean weak ){
        if( doc==null )throw new IllegalArgumentException( "doc==null" );
        return new DocumentChangeTracker(doc, weak);
    }
    
    public static DocumentChangeTracker tracking( Document doc ){
        if( doc==null )throw new IllegalArgumentException( "doc==null" );
        return new DocumentChangeTracker(doc, false);
    }
    
    public DocumentChangeTracker onChanges( final Runnable run, boolean weak ){
        if( run==null )throw new IllegalArgumentException( "run==null" );
        addListener(
            new DocumentChangeListener() {
                @Override
                public void documentChangeEvent(DocumentChangeEvent e) {
                    run.run();
                }
            },
            weak);
        return this;
    }
    
    public DocumentChangeTracker onChanges( final Runnable run ){
        if( run==null )throw new IllegalArgumentException( "run==null" );
        addListener( new DocumentChangeListener() {
                @Override
                public void documentChangeEvent(DocumentChangeEvent e) {
                    run.run();
                    }
            }
            , weak );
        return this;
    }
    
    public DocumentChangeTracker onChanges( final Reciver<String> reciver ){
        if( reciver==null )throw new IllegalArgumentException( "reciver==null" );
        addListener( 
            new DocumentChangeListener() {
                @Override
                public void documentChangeEvent(DocumentChangeEvent e) {
                    reciver.recive(e.getCurrent());
                }}
            , weak );
        return this;
    }
    
    public DocumentChangeTracker onChangesPair( final Reciver<Pair<String,String>> reciver ){
        if( reciver==null )throw new IllegalArgumentException( "reciver==null" );
        addListener( 
            new DocumentChangeListener() {
                @Override
                public void documentChangeEvent(DocumentChangeEvent e) {
                    reciver.recive( new BasicPair<String,String>(e.getPrevious(), e.getCurrent()) );
                }}, weak );
        return this;
    }
    
    public DocumentChangeTracker onChangesEvent( final Reciver<DocumentChangeEvent> reciver ){
        if( reciver==null )throw new IllegalArgumentException( "reciver==null" );
        addListener( 
            new DocumentChangeListener() {
                @Override
                public void documentChangeEvent(DocumentChangeEvent e) {
                    reciver.recive(e);
                }},
        weak );
        return this;
    }
    
    public DocumentChangeTracker onInsertEvent( final Reciver<DocumentChangeEvent.InsertEvent> reciver ){
        if( reciver==null )throw new IllegalArgumentException( "reciver==null" );
        addListener( new DocumentChangeListener() {
                @Override
                public void documentChangeEvent(DocumentChangeEvent e) {
                    if( e instanceof DocumentChangeEvent.InsertEvent ){
                        reciver.recive((DocumentChangeEvent.InsertEvent)e);
                    }
                }
        }, weak );
        return this;
    }
    
    public DocumentChangeTracker onUpdateEvent( final Reciver<DocumentChangeEvent.UpdateEvent> reciver ){
        if( reciver==null )throw new IllegalArgumentException( "reciver==null" );
        addListener( new DocumentChangeListener() {
                @Override
                public void documentChangeEvent(DocumentChangeEvent e) {
                if( e instanceof DocumentChangeEvent.UpdateEvent ){
                    reciver.recive((DocumentChangeEvent.UpdateEvent)e);
                }
            }
        }, weak );
        return this;
    }
    
    public DocumentChangeTracker onDeleteEvent( final Reciver<DocumentChangeEvent.DeleteEvent> reciver ){
        if( reciver==null )throw new IllegalArgumentException( "reciver==null" );
        addListener( new DocumentChangeListener() {
                @Override
                public void documentChangeEvent(DocumentChangeEvent e) {
            if( e instanceof DocumentChangeEvent.DeleteEvent ){
                reciver.recive((DocumentChangeEvent.DeleteEvent)e);
            }
                }}, weak );
        return this;
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="weakDocRef">
    protected boolean weak = false;
    
    public synchronized boolean isWeakDocRef() {
        return weak;
    }
    
    public synchronized void setWeakDocRef(boolean weak) {
        this.weak = weak;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="start/stop/isRunning">
    protected volatile  Closeable listenCloser;
    
    public synchronized DocumentChangeTracker start(){
        if( listenCloser!=null )return this;
        
        Document d = getDocument();
        if( d==null )return this;
        
        text = readText(d);
        listenCloser = addDocumentListener(d, documentListener, weak);
        
        return this;
    }
    
    public synchronized DocumentChangeTracker stop(){
        if( listenCloser==null )return this;
        try {
            listenCloser.close();
            listenCloser = null;
        } catch( IOException ex ) {
            Logger.getLogger(DocumentChangeTracker.class.getName()).log(Level.SEVERE, null, ex);
        }
        return this;
    }
    
    public synchronized boolean isRunning(){
        return listenCloser != null;
    }
    //</editor-fold>
    
    protected String text;
    
    //<editor-fold defaultstate="collapsed" desc="readText">
    protected String readText( Document doc ){
        if( doc==null )throw new IllegalArgumentException( "doc==null" );
        try {
            int len = doc.getLength();
            String txt = doc.getText(0, len);
            return txt;
        } catch( BadLocationException ex ) {
            Logger.getLogger(DocumentChangeTracker.class.getName()).log(Level.SEVERE, null, ex);
        }
        return null;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="document">
    protected Document doc;
    protected WeakReference<Document> docRef;
    public synchronized Document getDocument(){
        if( doc!=null )return doc;
        if( docRef!=null )return docRef.get();
        return null;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="fixChanged( txt )">
    protected synchronized void fixChanged( String currentText ){
        text = currentText;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="documentListener">
    protected final DocumentListener documentListener = new DocumentListener(){
        @Override
        public void insertUpdate(DocumentEvent e){
            if( e==null )return;
            String oldtxt = text;
            Document d = e.getDocument();
            
            if( d!=null ){
                int off = e.getOffset();
                int len = e.getLength();
                
                String txt = readText(d);
                fixChanged(txt);
                
                if( len>0 && txt!=null ){
                    String insTxt;
                    try{
                        insTxt = txt.substring(off, off+len);
                    }catch( Throwable err ){
                        logException(err);
                        return;
                    }
                    textInsertedEvent(d, oldtxt, txt, off, insTxt);
                }
            }
        }
        
        @Override
        public void removeUpdate(DocumentEvent e){
            if( e==null )return;
            String oldtxt = text;
            Document d = e.getDocument();
            
            if( d!=null ){
                int off = e.getOffset();
                int len = e.getLength();
                
                String txt = readText(d);
                fixChanged(txt);
                
                if( len>0 && txt!=null && oldtxt!=null ){
                    String prvTxt;
                    String insTxt;
                    try{
                        prvTxt = oldtxt.substring(off, off+len);
                    }catch( Throwable err ){
                        logException(err);
                        return;
                    }
                    textDeletedEvent(d, oldtxt, txt, off, prvTxt);
                }
            }
        }
        
        @Override
        public void changedUpdate(DocumentEvent e){
            if( e==null )return;
            String oldtxt = text;
            Document d = e.getDocument();
            
            if( d!=null ){
                int off = e.getOffset();
                int len = e.getLength();
                
                String txt = readText(d);
                fixChanged(txt);
                
                if( len>0 && txt!=null && oldtxt!=null ){
                    String prvTxt;
                    String insTxt;
                    try{
                        insTxt = txt.substring(off, off+len);
                        prvTxt = oldtxt.substring(off, off+len);
                    }catch( Throwable err ){
                        logException(err);
                        return;
                    }
                    textUpdatedEvent(d, oldtxt, txt, off, prvTxt, insTxt);
                }
            }
        }
    };
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="addDocumentListener( doc, listener, weak )">
    protected Closeable addDocumentListener( Document doc, final DocumentListener listener, final boolean weak ){
        doc.addDocumentListener(listener);
        
        final Document fdoc = doc;
        Closeable cl = new Closeable()
        {
            protected Document doc
                = weak ? null : fdoc;
            
            protected WeakReference<Document> docRef
                = weak ? new WeakReference<Document>(fdoc) : null;
            
            protected Document getDocument(){
                if( doc!=null )return doc;
                if( docRef!=null )return docRef.get();
                return null;
            }
            
            @Override
            public void close()
                throws IOException {
                
                Document d = getDocument();
                if( d!=null ){
                    d.removeDocumentListener(listener);
                }
            }
        };
        
        return cl;
    }
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="listeners">
    protected final CloseableSet listenersCloser = new CloseableSet();
    
    protected final ListenersHelper<DocumentChangeListener,DocumentChangeEvent> listeners
        = new ListenersHelper<DocumentChangeListener, DocumentChangeEvent>(
            new Func2<Object, DocumentChangeListener, DocumentChangeEvent>(){
        @Override
                public Object apply(DocumentChangeListener listnr, DocumentChangeEvent ev) {
                    listnr.documentChangeEvent(ev); 
                    return null;
                }
        });
    
    public boolean hasListener(DocumentChangeListener listener)
    {
        return listeners.hasListener(listener);
    }
    
    public Set<DocumentChangeListener> getListeners()
    {
        return listeners.getListeners();
    }
    
    public Closeable addListener(DocumentChangeListener listener)
    {
        Closeable c = listeners.addListener(listener);
        listenersCloser.add(c, true);
        return c;
    }
    
    public Closeable addListener(DocumentChangeListener listener, boolean weakLink)
    {
        Closeable c = listeners.addListener(listener, weakLink);
        listenersCloser.add(c, true);
        return c;
    }
    
    public void removeListener(DocumentChangeListener listener)
    {
        listeners.removeListener(listener);
    }
    
    public void fireEvent(DocumentChangeEvent event)
    {
        listeners.fireEvent(event);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="textInsertedEvent(doc,prev,cur,off,ins)">
    protected DocumentChangeEvent createInsertedEvent(
        Document doc, String preview, String current, int offset, String inserted )
    {
        DocumentChangeEvent.InsertEvent ev = new DocumentChangeEvent.InsertEvent();
        ev.setDocument(doc);
        ev.setPrevious(preview);
        ev.setCurrent(current);
        ev.setOffset(offset);
        ev.setInserted(inserted);
        return ev;
    }
    
    protected void textInsertedEvent( Document doc, String preview, String current, int offset, String inserted ){
        fireEvent(createInsertedEvent(doc, preview, current, offset, inserted));
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="textUpdatedEvent(doc,prev,cur,off,from,to)">
    protected DocumentChangeEvent createUpdatedEvent(
        Document doc, String preview, String current, int offset, String from, String to )
    {
        DocumentChangeEvent.UpdateEvent ev = new DocumentChangeEvent.UpdateEvent();
        ev.setDocument(doc);
        ev.setPrevious(preview);
        ev.setCurrent(current);
        ev.setOffset(offset);
        ev.setFrom(from);
        ev.setTo(to);
        return ev;
    }
    
    protected void textUpdatedEvent(
        Document doc, String preview, String current, int offset, String from, String to ){
        fireEvent(createUpdatedEvent(doc, preview, current, offset, from, to));
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="textDeletedEvent(doc,prev,cur,off,del)">
    protected DocumentChangeEvent createDeletedEvent(
        Document doc, String preview, String current, int offset, String deleted )
    {
        DocumentChangeEvent.DeleteEvent ev = new DocumentChangeEvent.DeleteEvent();
        ev.setDocument(doc);
        ev.setPrevious(preview);
        ev.setCurrent(current);
        ev.setOffset(offset);
        ev.setDeleted(deleted);
        return ev;
    }
    
    protected void textDeletedEvent( Document doc, String preview, String current, int offset, String deleted ){
        fireEvent(createDeletedEvent(doc, preview, current, offset, deleted));
    }
//</editor-fold>
    
    @Override public synchronized void close() throws IOException {
        stop();
        text = null;
        doc = null;
        if( docRef!=null ){
            docRef.clear();
            docRef = null;
        }
        listenersCloser.close();
    }

    @Override protected void finalize() throws Throwable {
        try{
            close();
        }finally{
            super.finalize();
        }
    }
}
