/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.tree;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.event.TableModelEvent;
import xyz.cofe.collection.Predicate;
import xyz.cofe.collection.iterators.TreeWalk;
import xyz.cofe.collection.tree.TreeNode;
import xyz.cofe.common.CloseableSet;
import xyz.cofe.common.Reciver;
import xyz.cofe.gui.swing.table.FilterRowTM;
import xyz.cofe.gui.swing.table.RowData;

/**
 * Модель таблицы-дерева с поддержкой фильтрации узлов
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class TreeTableFilterModel extends FilterRowTM implements TreeTableModelInterface
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(TreeTableFilterModel.class.getName());
    private static final Level logLevel(){ return Logger.getLogger(TreeTableFilterModel.class.getName()).getLevel(); }
    
    private static final boolean isLogSevere(){ 
        Level logLevel = logLevel();
        return logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    }
    
    private static final boolean isLogWarning(){ 
        Level logLevel = logLevel();
        return logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue(); }
    
    private static final boolean isLogInfo(){ 
        Level logLevel = logLevel();
        return logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue(); }
    
    private static final boolean isLogFine(){ 
        Level logLevel = logLevel();
        return logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue(); }
    
    private static final boolean isLogFiner(){ 
        Level logLevel = logLevel();
        return logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue(); }
    
    private static final boolean isLogFinest(){ 
        Level logLevel = logLevel();
        return logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue(); }

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(TreeTableFilterModel.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(TreeTableFilterModel.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(TreeTableFilterModel.class.getName(), method, result);
    }
    //</editor-fold>
    
    /**
     * Конструктор
     * @param dm нтерфейс доступа к таблице-дереву
     */
    public TreeTableFilterModel( TreeTableModelInterface dm ){
        if (dm== null) {
            throw new IllegalArgumentException("dm==null");
        }
        
        setRowFilter( filter );
        
        setSourceListen(true);        
        setTableModel(dm);
        
        TreeTableNode root = dm.getRoot();
        if( root!=null ){
            setRoot(root);
        }
    }
    
    protected final Predicate<RowData> filter = new Predicate<RowData>() {
        @Override
        public boolean validate(RowData rowdata) {
            Object onode = rowdata.getValue(0);
            if( onode instanceof TreeTableNode ){            
                TreeTableNode node = (TreeTableNode)onode;
                boolean visible = isVisible(node);
                
//                if( isLo
                logFinest("filter.validate visible={0} ro={1} data={2}", 
                    visible, node.getRootOffset(), node.getData());
                
                return visible;
            }
            return true;
        }
    };
    
    /**
     * Возвращает виден-ли узел
     * @param node узел
     * @return true - узел должен быть виден
     */
    public synchronized boolean isVisible( TreeTableNode node ){
        if( node==null )return true;

        List npath = node.getNodePath();
        if( npath==null )return true;

        int level = npath.size();
        if( level==0 )return true;

        // root node
        if( level==1 )return true;

        // sub root
        if( !isRootVisible() && level==2 )return true;

        npath.remove(npath.size()-1);
        
        int ito = isRootVisible() ? 0 : 1;
        for( int i=npath.size()-1; i>=ito; i-- ){
            Object o_n = npath.get(i);
            if( o_n instanceof TreeTableNode ){
                TreeTableNode n = (TreeTableNode)o_n;
                boolean exp = n.isExpanded();
                if( !exp ){
                    return false;
                }
            }
        }
        
        return true;
    }
    
    @Override
    public synchronized TreeTableNode getNodeOf( int row ){
        int directRow = mapRowToInside(row);
        if( directRow<0 )return null;
        
        return ((TreeTableModelInterface)getTableModel()).getNodeOf(directRow);
    }
//    
    @Override
    public synchronized int getRowOf( TreeTableNode node ){
        if( node==null )return -1;
        if( root == null )return -1;
        
        int sortrow = ((TreeTableModelInterface)getTableModel()).getRowOf(node);
        if( sortrow<0 )return -1;
        
        int frow = mapRowToOutside(sortrow);
        return frow;
    }
    
    @Override
    public synchronized boolean isRootVisible(){
        return ((TreeTableModelInterface)getTableModel()).isRootVisible();
    }
    
    //<editor-fold defaultstate="collapsed" desc="root : TreeTableNode - древо таблицы">
    /**
     * Древо таблицы
     */
    protected TreeTableNode root;
    
    /**
     * Возвращает древо таблицы
     * @return древо таблицы
     */
    @Override
    public synchronized TreeTableNode getRoot() {
        if( root!=null )return root;
        root = new TreeTableNodeBasic();
        return root;
    }
    
    /**
     * Устанавливает древо таблицы
     * @param root древо
     */
    @Override
    public synchronized void setRoot(TreeTableNode root) {
        ((TreeTableModelInterface)getTableModel()).setRoot(root);

        TreeTableNode old = this.getRoot();
        this.root = root;
        TreeTableNode newv = this.getRoot();
        
        logFine("setRoot old={0} current={1}",old,newv);
        
        listenRoot();
        applyFilter();
    }
    //</editor-fold>
      
    //<editor-fold defaultstate="collapsed" desc="listenRoot()">
    protected final CloseableSet rootListeners = new CloseableSet();
    
    /**
     * Добавляет подписчика на событие корневого элемента дерева
     */
    protected synchronized void listenRoot(){
        logFine("listenRoot");
        
        logFinest("rootListeners.closeAll");
        rootListeners.closeAll();
        
        if( root==null )return;

        Closeable cl = null;
        
        cl = root.onTreeNodeEvent(TreeTableNodeCollapsed.class, 
            new Reciver() {
            @Override
            public void recive(Object e) {
                TreeTableNodeCollapsed ev = (TreeTableNodeCollapsed)e;
                onTreeNodeCollapsed(ev);
            }} );
        rootListeners.add( cl );

        cl = root.onTreeNodeEvent(TreeTableNodeExpanded.class, new Reciver() {
            @Override
            public void recive(Object e) {         
                TreeTableNodeExpanded ev = (TreeTableNodeExpanded)e;
                onTreeNodeExpanded(ev);
            }} 
        );
        rootListeners.add( cl );
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="onTreeNodeExpanded()">
    private void onTreeNodeExpanded(TreeTableNodeExpanded ev) {
        TreeNode ettn1 = ev==null ? null : ev.getSource();
        TreeTableNodeBasic ettn = (TreeTableNodeBasic)ettn1;
        Object ettnData = ettn==null ? null : ettn.getData();
        //Object ettnRo = ettn==null ? null : ettn.getRootOffset();
        
        logFine("onTreeNodeExpanded ev.source.data={0}", ettnData );
        
        /*
        TreeSet<Integer> srows = new TreeSet<Integer>();
        
        for( TreeTableNode node : ev.getSource().walk() ){
            int srow = ((TreeTableModelInterface)getTableModel()).getRowOf(node);
            if( srow>=0 ){
                srows.add(srow);
            }
        }
        
        for( Integer srow : srows.descendingSet() ){
            for( TableModelEvent tev : processRowUpdated(srow, srow) ){
                fireTableModelEvent(tev);
            }
        }*/
        
        if( source==null )return;
        
        TreeSet<Integer> includeRows = new TreeSet<>();
        
        for( TreeWalk<TreeTableNode> twnode : ev.getSource().tree()){
            TreeTableNode node = twnode.currentNode();
            
            int si = ((TreeTableModelInterface)getTableModel()).getRowOf(node);
            if( si<0 )continue;
            
            Integer di = source.indexOf(si);
            if( di==null || di<0 ){
                if( isVisible(node) ){
                    // необходимо отобразить строку
                    di = source.add(si);
                    includeRows.add(di);
                }
            }else{
                continue;
            }
        }
        
        if( includeRows.isEmpty() )return;
        if( includeRows.size()==1 ){
            int di = includeRows.first();
            fireRowsInserted(di, di);
            return;
        }
        
        List<int[]> ranges = new ArrayList<>();
        
        int rangeStart = -1;
        int rangeEnd = -1;
        int diPrev = -1;
        int i = -1;
        for( int di : includeRows ){
            i++;
            if( i==0 ){
                rangeStart = di;
            }else{
                int d = Math.abs(diPrev - di);
                if( d>1 ){
                    rangeEnd = diPrev;
                    ranges.add(new int[]{ rangeStart, rangeEnd });
                }
                rangeStart = di;
            }
            diPrev = di;
        }
        ranges.add(new int[]{ rangeStart, diPrev });
        
        for( int ri=0; ri<ranges.size(); ri++ ){
            int[] range = ranges.get(ri);
            int rStart = range[0];
            int rEnd = range[1];
            int rFrom = Math.min(rStart, rEnd);
            int rTo = Math.max(rStart, rEnd);
            fireRowsInserted(rFrom, rTo);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="onTreeNodeCollapsed()">
    private void onTreeNodeCollapsed(TreeTableNodeCollapsed ev) {
        TreeNode ettn1 = ev==null ? null : ev.getSource();
        
        TreeTableNodeBasic ettn = (TreeTableNodeBasic)ettn1;
        Object ettnData = ettn==null ? null : ettn.getData();
        //Object ettnRo = ettn==null ? null : ettn.getRootOffset();
        
        logFine("onTreeNodeCollapsed ev.source.data={0}", ettnData );
        
        /*TreeSet<Integer> srows = new TreeSet<Integer>();
        
        for( TreeTableNode node : ev.getSource().walk() ){
            int srow = ((TreeTableModelInterface)getTableModel()).getRowOf(node);
            if( srow>=0 ){
                srows.add(srow);
            }
        }
        
        for( Integer srow : srows.descendingSet() ){
            for( TableModelEvent tev : processRowUpdated(srow, srow) ){
                fireTableModelEvent(tev);
            }
        }*/
        
        if( source==null )return;
        
        TreeSet<Integer> removedRows = new TreeSet<>();
        
        for( TreeWalk<TreeTableNode> twnode : ev.getSource().tree()){
            TreeTableNode node = twnode.currentNode();
            
            int si = ((TreeTableModelInterface)getTableModel()).getRowOf(node);
            if( si<0 )continue;
            
            Integer di = source.indexOf(si);
            if( di==null )continue;            
            if( di<0 )continue;
            
            int lvl = Math.abs(twnode.currentLevel()-twnode.startLevel());
            if( lvl>0 ){
                // точно не видим
                source.removeByIndex(di);
                removedRows.add(di);
                continue;
            }
            
            if( !isVisible(node) ){
                source.removeByIndex(di);
            }
        }
        
        if( removedRows.isEmpty() )return;
        if( removedRows.size()==1 ){
            int di = removedRows.first();
            fireRowsDeleted(di, di);
            return;
        }
        
        List<int[]> ranges = new ArrayList<>();
        
        int rangeStart = -1;
        int rangeEnd = -1;
        int diPrev = -1;
        int i = -1;
        for( int di : removedRows ){
            i++;
            if( i==0 ){
                rangeStart = di;
            }else{
                int d = Math.abs(diPrev - di);
                if( d>1 ){
                    rangeEnd = diPrev;
                    ranges.add(new int[]{ rangeStart, rangeEnd });
                }
                rangeStart = di;
            }
            diPrev = di;
        }
        ranges.add(new int[]{ rangeStart, diPrev });
        
        for( int ri=ranges.size()-1; ri>=0; ri-- ){
            int[] range = ranges.get(ri);
            int rStart = range[0];
            int rEnd = range[1];
            int rFrom = Math.min(rStart, rEnd);
            int rTo = Math.max(rStart, rEnd);
            fireRowsDeleted(rFrom, rTo);
        }
    }
    //</editor-fold>
}
