/*
 * The MIT License
 *
 * Copyright 2017 Kamnev Georgiy <nt.gocha@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.gui.swing.tree;

import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.Func1;
import xyz.cofe.collection.NodesExtracter;
import xyz.cofe.collection.list.BulkInsertEvent;
import xyz.cofe.collection.list.BulkInsertListener;
import xyz.cofe.collection.list.BulkInsertedEvent;
import xyz.cofe.collection.list.EventList;
import xyz.cofe.collection.list.SimpleListAdapter;
import xyz.cofe.collection.tree.AbstractTreeNode;
import xyz.cofe.collection.tree.IndexTreeNode;
import xyz.cofe.collection.tree.TreeNode;
import xyz.cofe.collection.tree.TreeNodeAdded;
import xyz.cofe.collection.tree.TreeNodeBulkInserted;
import xyz.cofe.collection.tree.TreeNodeCacheDropped;
import xyz.cofe.collection.tree.TreeNodeEvent;
import xyz.cofe.collection.tree.TreeNodeFollowed;
import xyz.cofe.collection.tree.TreeNodeFollowing;
import xyz.cofe.collection.tree.TreeNodeRemoved;
import xyz.cofe.collection.tree.TreeNodeSetParent;
import xyz.cofe.collection.tree.TreeNodeUpdateParent;
import xyz.cofe.common.Reciver;
import xyz.cofe.gui.swing.tree.impl.TreeTableNodeBasicImpl;

/**
 * Базовая реализация TreeTableNode
 * @author nt.gocha@gmail.com
 */
public class TreeTableNodeBasic 
    extends IndexTreeNode<TreeTableNodeBasic>
    implements TreeTableNode<TreeTableNodeBasic>,
               TreeTableNodeGetText, TreeTableNodeGetFormat
{
    /*private static boolean eq( Object a, Object b ){
        if( a==null && b==null )return true;
        if( a==null && b!=null )return false;
        if( a!=null && b==null )return false;
        return a.equals(b);
    }*/

    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(TreeTableNodeBasic.class.getName());
    
    private static Level logLevel(){ return logger.getLevel(); }
    private static boolean isLogSevere(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.SEVERE.intValue();
    }
    private static boolean isLogWarning(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.WARNING.intValue();
    }
    private static boolean isLogInfo(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.INFO.intValue();
    }
    private static boolean isLogFine(){
        Level ll = logLevel();
        return ll == null
            ? true
            : ll.intValue() <= Level.FINE.intValue();
    }
    private static boolean isLogFiner(){
        Level ll = logLevel();
        return ll == null
            ? false
            : ll.intValue() <= Level.FINER.intValue();
    }
    private static boolean isLogFinest(){
        Level ll = logLevel();
        return ll == null
            ? false
            : ll.intValue() <= Level.FINEST.intValue();
    }
    
    private static void logEntering(String method,Object ... args){
        logger.entering(TreeTableNodeBasic.class.getName(), method, args);
    }
    private static void logExiting(String method,Object result){
        logger.exiting(TreeTableNodeBasic.class.getName(), method, result);
    }
    
    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }
    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }
    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }    
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getChildren()">
    @Override
    public TreeTableNodeBasic[] getChildren() {
        return (TreeTableNodeBasic[])
            syncrun( new Func0() {
            @Override
            public Object apply() {
                return getChildrenList().toArray(new TreeTableNodeBasic[]{});
            }}, "getChildren" );
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="TreeTableNodeBasic()">
    /**
     * Конструктор по умолчанию
     */
    public TreeTableNodeBasic(){
    }
    
    /**
     * Конструктор
     * @param data данные узла для отображения/редактирования
     */
    public TreeTableNodeBasic(Object data){
        setData(data);
    }
    
    /**
     * Конструктор копирования
     * @param sample образец для копирования
     * @param withChildren Копировать с дочерними узлами
     * @param preferred Копировать предпочтительные функции/данные (getPreffered...())
     */
    public TreeTableNodeBasic(TreeTableNodeBasic sample, boolean withChildren, boolean preferred){
        if( sample!=null ){
            setData( sample.getData() );
            
            setCacheLifeTime( preferred ? sample.getPreferredCacheLifeTime() : sample.getCacheLifeTime() );
            setDataFollowable( preferred ? sample.getPreferredDataFollowable() : sample.getDataFollowable() );
            setDataFollower( preferred ? sample.getPreferredDataFollower() : sample.getDataFollower() );
            setDataTextReader( sample.getDataTextReader() );
            setDataFormatter( preferred ? sample.getPreferredDataFormatter() : sample.getDataFormatter() );
            
            Map<TreeTableNode,Date> sampleCached = new LinkedHashMap<TreeTableNode,Date>();
            sampleCached.putAll(sample.getCachedNodes());
            
            setFollowStarted(sample.getFollowStarted());
            setFollowStarted(sample.getFollowFinished());
            
            if( withChildren ){
                List<TreeTableNodeBasic> sampleChildren = sample.getChildrenList();
                List<TreeTableNodeBasic> children = getChildrenList();
                
                if( sampleChildren!=null && !sampleChildren.isEmpty() ){
                    Map<TreeTableNode,TreeTableNode> clones = new LinkedHashMap<TreeTableNode,TreeTableNode>();
                    
                    for( TreeTableNodeBasic schild : sampleChildren ){
                        if( schild==null )continue;
                        
                        TreeTableNodeBasic child = schild.clone(withChildren, preferred);
                        clones.put(schild, child);
                        children.add(child);
                    }
                    
                    for( Map.Entry<TreeTableNode,Date> en : sampleCached.entrySet() ){
                        TreeTableNode schild = en.getKey();
                        Date d = en.getValue();
                        if( schild==null )continue;
                        
                        TreeTableNode child = clones.get(schild);
                        if( child==null )continue;
                        
                        getCachedNodes().put(child, d);
                    }
                }
            }
            
            setExpanded(sample.isExpanded());
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="clone()">
    /**
     * Создание клона узла с дочерними узлами.
     * @return клон
     */
    @Override
    @SuppressWarnings("CloneDeclaresCloneNotSupported")
    public TreeTableNodeBasic clone(){
        TreeTableNodeBasic cloned = (TreeTableNodeBasic)syncrun(new Func0() {
            @Override
            public Object apply() {
                return new TreeTableNodeBasic(TreeTableNodeBasic.this, true, false);
            }}, "clone");
        return cloned;
    }
    
    /**
     * Создание клона
     * @param withChildren клонировать так-же дочерние узлы
     * @param preferred Копировать предпочтительные функции/данные (getPreffered...())
     * @return Клон
     */
    public TreeTableNodeBasic clone( final boolean withChildren, final boolean preferred ){
        TreeTableNodeBasic cloned = (TreeTableNodeBasic)syncrun( new Func0() {
            @Override
            public Object apply() {
                return new TreeTableNodeBasic(TreeTableNodeBasic.this, withChildren, preferred);
            }}, "clone");
        return cloned;
    }
    //</editor-fold>
        
    //<editor-fold defaultstate="collapsed" desc="data : Object">
    protected Object data = null;
    
    @Override
    public Object getData(){
        //return TreeTableNodeDefs.getDataOf(this);
        return data;
    }
    
    @Override
    public void setData( Object v ){
        // Object old = TreeTableNodeDefs.setDataOf(this, v);
        // TreeTableDataChanged ev = new TreeTableDataChanged(this,old,v);
        // popup((TreeNodeEvent)ev);
        
        Object old = data;
        this.data = v;
        
        TreeTableDataChanged ev = new TreeTableDataChanged(this,old,v);
        popup((TreeNodeEvent)ev);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="dataPath">
    @Override
    public List<Object> getDataPath(){
        LinkedList ll = new LinkedList();
        for( TreeTableNodeBasic n : getNodePath() ){
            //getNodePath().forEach( n -> {
            ll.add(n.getData());
        } //);
        return ll;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="treeLevel : int">
    @Override
    public int getTreeLevel(){
        return getNodePath().size();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="toString()">
    private int toStringCall = 0;
    
    @Override
    public synchronized String toString() {
        try{
            toStringCall++;
            if( toStringCall>1 )return "";
            
            StringBuilder sb = new StringBuilder();
            
            Object d = getData();
            
            sb.append(this.getClass().getSimpleName()).append("{");
            //sb.append("rootOffset=").append(getRootOffset());
            sb.append("data=");
            
            if( d==null ){
                sb.append("null");
            }else{
                sb.append(d.toString());
            }
            
            sb.append("}");
            
            return sb.toString();
        }
        finally{
            toStringCall--;
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="dataTextReader">
    protected Func1<String,Object> dataTextReader;
    
    /**
     * Возвращает функцию преобразоавния данных узла в текстовое представление
     * @return функция (node.data):String
     */
    public Func1<String, Object> getDataTextReader() {
        // return (Func1)this.syncrun( ()->{
        return dataTextReader;
        // }, "getDataTextReader");
    }
    
    /**
     * Указывает функцию преобразоавния данных узла в текстовое представление
     * @param dataTextReader функция (node.data):String
     */
    public void setDataTextReader(Func1<String, Object> dataTextReader) {
        //        this.dataTextReader = dataTextReader;
        //        syncrun(()->{
        this.dataTextReader = dataTextReader;
        //            return null;
        //        }, "setDataTextReader", dataTextReader);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="getDataText()">
    /**
     * Возвращает отображаемый текст для данных узла, 
     * использует ближащую (по дереву вверх к корню) функцию dataTextReader
     * @return Отображаемый текст
     * @see #getDataTextReader() 
     */
    public String getDataText(){
        Func1<String,Object> dataTextResolver = dataTextReader;
        if( dataTextResolver==null ){
            List<TreeTableNodeBasic> path = getNodePath();
            if( path!=null && path.size()>0 )path.remove(path.size()-1);
            for( int i=path.size()-1; i>=0; i-- ){
                TreeTableNodeBasic n = path.get(i);
                dataTextResolver = n.getDataTextReader();
                if( dataTextResolver!=null )break;
            }
        }
        
        if( dataTextResolver!=null ){
            Object data = getData();
            String txt = dataTextResolver.apply(data);
            if( txt==null && data!=null )return data.toString();
            return txt;
        }
        
        Object data = getData();
        return data==null ? data.toString() : null;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="treeTableNodeGetText() : String">
    /**
     * Возвращает отображаемый текст для данных узла, 
     * использует ближащую (по дереву вверх к корню) функцию dataTextReader
     * @return Отображаемый текст
     * @see #getDataTextReader() 
     */
    @Override
    public String treeTableNodeGetText() {
        return getDataText();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Форматирование">
    /**
     * Форматирование вывода
     */
    protected TreeTableNodeGetFormatOf dataFormatter;
    
    /**
     * Возвращает форматтер данных
     * @return форматтер
     */
    public TreeTableNodeGetFormatOf getDataFormatter() {
        return dataFormatter;
    }
    
    /**
     * Указывает форматтер данных
     * @param dataFormatter форматтер
     */
    public void setDataFormatter(TreeTableNodeGetFormatOf dataFormatter) {
        this.dataFormatter = dataFormatter;
    }
    
    /**
     * Возвращает предпочтительное форматирование данных (ближайщее функция форматирования вверх по дереву)
     * @return предпочтительный форматтер
     * @see #getDataFormatter() 
     * @see #setDataFormatter(xyz.cofe.gui.swing.tree.TreeTableNodeGetFormatOf) 
     */
    public TreeTableNodeGetFormatOf getPreferredDataFormatter(){
        TreeTableNodeGetFormatOf formatter = dataFormatter;
        if( formatter==null ){
            List<TreeTableNodeBasic> path = getNodePath();
            if( path!=null && path.size()>0 )path.remove(path.size()-1);
            for( int i=path.size()-1; i>=0; i-- ){
                TreeTableNodeBasic n = path.get(i);
                formatter = n.getDataFormatter();
                if( formatter!=null )break;
            }
        }
        return formatter;
    }
    
    /**
     * Возвращает форматирование узла дерева
     * @return форматирование
     */
    @Override
    public TreeTableNodeFormat getTreeTableNodeFormat() {
        Object data = getData();
        if( data==null )return null;
        
        TreeTableNodeGetFormatOf getFmt = getPreferredDataFormatter();
        if( getFmt==null )return null;
        
        return getFmt.getTreeTableNodeFormatOf(data);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="dataFollower">
    protected NodesExtracter<Object,Object> dataFollower;
    
    /**
     * Возвращает функцию извлечения дочерних узлов поддерева
     * @return функция fn( node ) : [node]
     */
    public NodesExtracter<Object, Object> getDataFollower() {
        return dataFollower;
    }
    
    /**
     * Указывает функцию извлечения дочерних узлов поддерева
     * @param dataFollower функция fn( node ) : [node]
     */
    public void setDataFollower(NodesExtracter<Object, Object> dataFollower) {
        this.dataFollower = dataFollower;
    }

    /**
     * Возвращает ближайщую (вверх по дереву) функцию получения дочерних узлов
     * @return функция получения дочерних узлов или null
     */
    public NodesExtracter<Object,Object> getPreferredDataFollower(){
        NodesExtracter<Object,Object> extracter = dataFollower;
        if( extracter==null ){
            List<TreeTableNodeBasic> path = getNodePath();
            if( path!=null && path.size()>0 )path.remove(path.size()-1);
            for( int i=path.size()-1; i>=0; i-- ){
                TreeTableNodeBasic n = path.get(i);
                extracter = n.getDataFollower();
                if( extracter!=null )break;
            }
        }
        // if( extracter==null )return null;
        return extracter;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="followChildrenIterable">
    /**
     * Получение дочерних узлов/данных для данного узла
     * @return дочерние узлы/данные
     */
    public Iterable getFollowChildrenIterable(){
        NodesExtracter ne = getPreferredDataFollower();
        Object data = getData();
        
        if( ne==null || data==null )return null;
        if( ne instanceof TreeNodesExtracter ){
            return ((TreeNodesExtracter)ne).extract(this);
        }
        
        return ne.extract(data);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="dataFollower">
    protected Func1<Boolean,Object> dataFollowable;

    /**
     * Возвращает функцию проверки наличия дочерних улов для их дальшего извлечени.
     * @return функция fn( node.data ) : boolean
     * @see #getDataFollower() 
     * @see #getPreferredDataFollower() 
     */
    public Func1<Boolean, Object> getDataFollowable() {
        return dataFollowable;
    }

    /**
     * Указывает функцию проверки наличия дочерних улов для их дальшего извлечени
     * @param dataFollowable функция fn( node.data ) : boolean
     * @see #getDataFollower() 
     * @see #getPreferredDataFollower() 
     */
    public void setDataFollowable(Func1<Boolean, Object> dataFollowable) {
        this.dataFollowable = dataFollowable;
    }

    /**
     * Возвращает ближайшую (вверх по дереву) функцию проверки наличия дочерних улов
     * @return функция fn( node.data ) : boolean
     * @see #getDataFollowable() 
     * @see #getDataFollower() 
     * @see #getPreferredDataFollower() 
     */
    public Func1<Boolean, Object> getPreferredDataFollowable(){
        Func1<Boolean, Object> extractable = dataFollowable;
        if( extractable==null ){
            List<TreeTableNodeBasic> path = getNodePath();
            if( path!=null && path.size()>0 )path.remove(path.size()-1);
            for( int i=path.size()-1; i>=0; i-- ){
                TreeTableNodeBasic n = path.get(i);
                extractable = n.getDataFollowable();
                if( extractable!=null )break;
            }
        }
        // if( extracter==null )return null;
        return extractable;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="followStarted">
    /**
     * Указывает время начала follow функции
     * @see #follow() 
     * @see #getPreferredDataFollower() 
     */
    protected Date followStarted;
    
    /**
     * Указывает время начала follow функции
     * @return время начала или null, если еще не вызвана
     * @see #follow() 
     * @see #getPreferredDataFollower() 
     */
    public Date getFollowStarted() {
        return followStarted;
    }
    
    /**
     * Указывает время начала follow функции
     * @param followStarted время начала или null, если еще не вызвана
     * @see #follow() 
     * @see #getPreferredDataFollower() 
     */
    public void setFollowStarted(Date followStarted) {
        this.followStarted = followStarted;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="followFinished">
    /**
     * Указывает время завершения follow функции
     * @see #follow() 
     * @see #getPreferredDataFollower() 
     */
    protected Date followFinished;
    
    /**
     * Указывает время завершения follow функции
     * @return время завершения или null, если еще не завершена
     * @see #follow() 
     * @see #getPreferredDataFollower() 
     */
    public Date getFollowFinished() {
        return followFinished;
    }
    
    /**
     * Указывает время завершения follow функции
     * @param followFinished время завершения или null, если еще не завершена
     * @see #follow()
     * @see #getPreferredDataFollower() 
     */
    public void setFollowFinished(Date followFinished) {
        this.followFinished = followFinished;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="cacheLifeTime">
    protected Long cacheLifeTime = null;
    
    /**
     * Указывает время жизни извлеченных (follow) объектов в кэше
     * @return время жизни в кэше или null
     * @see #follow() 
     */
    public Long getCacheLifeTime() {
        return cacheLifeTime;
    }
    
    /**
     * Указывает время жизни извлеченных (follow) объектов в кэше
     * @param cacheLifeTime время жизни в кэше или null
     * @see #follow() 
     */
    public void setCacheLifeTime(Long cacheLifeTime) {
        this.cacheLifeTime = cacheLifeTime;
    }
    
    /**
     * Указывает предпочтительно время жизни извлеченных (follow) объектов в кэше. <p>
     * Если значение не установлено, то (рекурсивно) читается значение у родительского узла.
     * @return время жизни в кэше или null
     * @see #follow() 
     */
    public Long getPreferredCacheLifeTime(){
        Long lifetime = -1L;
        List<TreeTableNodeBasic> path = getNodePath();
        for( int i=path.size()-1; i>=0; i-- ){
            TreeTableNodeBasic ttnb = path.get(i);
            if( ttnb.cacheLifeTime!=null ){
                lifetime = ttnb.cacheLifeTime;
                break;
            }
        }
        return lifetime;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="cachedNodes">
    protected WeakHashMap<TreeTableNode,Date> cachedNodes = new WeakHashMap<TreeTableNode,Date>();
    
    //public 
    
    /**
     * Возвращает кэш извлеченных дочерних узлов
     * @return кеш
     * @see #follow() 
     * @see #getCacheLifeTime() 
     */
    public Map<TreeTableNode,Date> getCachedNodes(){
        //LinkedHashMap res = new LinkedHashMap();
        //res.putAll(cachedNodes);
        //return res;
        return cachedNodes;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="readFollowChildrenTo()">
    /**
     * Извлекает дочерние узлы и передает их в функцию приемник
     * @param childDataConsumer функция применик
     * @see #follow() 
     * @see #getPreferredDataFollower() 
     */
    protected void readFollowChildrenTo( final Reciver<Object> childDataConsumer ){
        Iterable children = getFollowChildrenIterable();
        if( children!=null ){
            int timeoutThreshold = TreeTableNodeBasicImpl.getUseExpanderThresholdTimeout();
            long tStart = System.currentTimeMillis();
            
            long tReadTotal = 0;
            long tConsumeTotal = 0;
            int readed = 0;
            
            Iterator iter = children.iterator();
            boolean createExpander = false;
            while( true ){
                long tRead0 = System.currentTimeMillis();
                boolean hnext = iter.hasNext();
                if( !hnext )break;
                
                long tCurrent = System.currentTimeMillis();
                long tDiff = Math.abs(tCurrent - tStart);
                if( timeoutThreshold>0 && tDiff>=timeoutThreshold ){
                    createExpander = true;
                    logFine( "terminate readFollowChildrenTo by timeoutThreshold={0} timeout={1}", timeoutThreshold, tDiff );
                    break;
                }
                
                Object childData = iter.next();
                readed++;
                long tRead1 = System.currentTimeMillis();
                
                long tConsume0 = System.currentTimeMillis();
                childDataConsumer.recive(childData);
                long tConsume1 = System.currentTimeMillis();
                
                tReadTotal += Math.abs(tRead0 - tRead1);
                tConsumeTotal += Math.abs(tConsume1-tConsume0);
                
                logFiner("node readed, total={0}, t.consume={1} {4} t.read1={2} {5}, t.*1={3} {6}",
                    readed, //0
                    Math.abs(tConsume1-tConsume0), //1
                    Math.abs(tRead0 - tRead1),  //2
                    Math.abs(tConsume1-tConsume0)+Math.abs(tRead0 - tRead1), //3
                    tConsumeTotal, //4
                    tReadTotal, //5
                    tConsumeTotal + tReadTotal
                );
            }
            if( createExpander ){
                logFine("create expander by timeoutThreshold={0}", timeoutThreshold);
                
                TreeTableNodeExpander expander = new TreeTableNodeExpander(iter);
                childDataConsumer.recive(expander);
            }
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="consumeChildData()">
    /**
     * Создает функцию применик для ивлеченных дочерних узлов,
     * кеширует принятые узлы
     * @return функция применик
     * @see #follow() 
     * @see #getCachedNodes() 
     * @see #getCacheLifeTime() 
     */
    protected Reciver<Object> consumeChildData(){
        //return (Object childData) -> {
        return new Reciver<Object>() {
            @Override
            public void recive(Object childData) {
                
                TreeTableNodeBasic ttnb = null;
                
                if( childData instanceof TreeTableNodeBasic ){
                    ttnb = (TreeTableNodeBasic)childData;
                }else{
                    ttnb = new TreeTableNodeBasic(childData);
                }
                
                appendChild(ttnb);
                
                cachedNodes.put(ttnb, new Date());
            }};
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="dropCache()">
    /**
     * Очистка кэша и удаление уешированных узлов (только прямые потомки данного узла)
     * @see #getCacheLifeTime() 
     * @see #getCachedNodes() 
     * @see #getPreferredDataFollower() 
     * @see #follow() 
     */
    public void dropCache(){
        TreeNodeCacheDropped ev = new TreeNodeCacheDropped(this);
        
        AtomicInteger cnt = new AtomicInteger(0);
        
        for( TreeTableNode node : cachedNodes.keySet() ){
            //cachedNodes.keySet().forEach( node -> {
            if( node!=null ){
                if( node instanceof TreeTableNodeBasic ){
                    removeChild((TreeTableNodeBasic)node);
                    cnt.addAndGet(((TreeTableNodeBasic) node).getNodesCount());
                    ev.getDropped().add(node);
                }
            }
        }// );
        
        cachedNodes.clear();
        
        followFinished = null;
        followStarted = null;
        
        popup(ev);
        
        logFine("dropped {0} child nodes", cnt.get());
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="follow()">
    /**
     * Извлекает дочерние объекты и добавляет их в дерево и кэш.
     * <p>
     *
     * Для извлечения использует:
     *
     * <ul>
     * <li>getPreferredDataFollowable() - Для проверки возможности извлечения данных
     * <li>getFollowChildrenIterable() - Для получения извлекающего итератора
     * </ul>
     *
     * Извлеченные объекты кэшируются, в дальшейшем кэш самостоятельно
     * (в зависимости от cacheLifTime) очищается. <p>
     *
     * Принудительно можно очистить вызвав dropCache() <p>
     *
     * Время начала и завершения извлечени содержаться в свойствах:
     * followStarted, followFinished.
     * 
     * @see #getPreferredDataFollowable() 
     * @see #readFollowChildrenTo(xyz.cofe.common.Reciver) 
     * @see TreeNodeFollowing
     * @see TreeNodeFollowed
     */
    public void follow(){
        followStarted = new Date();
        
        Func1<Boolean,Object> extrSuppTester = getPreferredDataFollowable();
        
        if( extrSuppTester!=null ){
            Object data = getData();
            if( extrSuppTester.apply(data) ){
                popup(new TreeNodeFollowing(this));
                readFollowChildrenTo( consumeChildData() );
                popup(new TreeNodeFollowed(this));
            }
        }else{
            popup(new TreeNodeFollowing(this));
            readFollowChildrenTo( consumeChildData() );
            popup(new TreeNodeFollowed(this));
        }
        
        followFinished = new Date();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="expand/collapse">
    /**
     * Раскрывает узел и при необходимости (followStarted==null) извлекает дочерние узлы
     * @see #getFollowStarted() 
     * @see #follow() 
     * @see TreeTableNodeExpanding
     * @see TreeTableNodeExpanded
     */
    @Override
    public void expand() {
        /* Тестирование 1
        if( followStarted==null ){
            follow();
        }
        */
        
        TreeTableNodeExpanding ev1 = null;
        
        // Генерация события Expanding
        Boolean old1 = TreeTableNodeDefs.getExpandedOf(this);
        if( !Objects.equals(old1, (Boolean)true) ){
            ev1 = new TreeTableNodeExpanding(this);
            popup( (TreeNodeEvent)ev1 );
        }
        
        // Раскрытие род. узла
        TreeTableNodeBasic prnt = getParent();
        if( prnt!=null ){
            prnt.expand();
        }
        
        // Тестирование 1
        if( followStarted==null ){
            follow();
        }
        
        // Генерация события Expanded
        boolean exp = true;
        Boolean old2 = TreeTableNodeDefs.setExpandedOf(this, exp);
        if( !Objects.equals(old2, (Boolean)exp) ){
            TreeTableNodeExpanded ev2 = new TreeTableNodeExpanded(this,ev1);
            popup( (TreeNodeEvent)ev2 );
        }
    }
    
    /**
     * Сворачивает узел (скрывает дочерние узлы) 
     * и удаляет из кеша ранее кешированные узлы при истечении времени
     * @see #getPreferredCacheLifeTime() 
     * @see #getFollowFinished() 
     * @see TreeTableNodeCollapsing
     * @see TreeTableNodeCollapsed
     */
    @Override
    public void collapse() {
        if( followFinished!=null ){
            Long lifet = getPreferredCacheLifeTime();
            if( lifet!=null && lifet>0 ){
                long tdiff = Math.abs(followFinished.getTime() - System.currentTimeMillis() );
                if( tdiff > lifet ){
                    dropCache();
                    followFinished = null;
                    followStarted = null;
                }
            }
        }
        
        TreeTableNodeCollapsing ev1 = null;

        Boolean old1 = TreeTableNodeDefs.getExpandedOf(this);
        
        if( !Objects.equals(old1, (Boolean)false) ){
            ev1 = new TreeTableNodeCollapsing(this);
            popup( (TreeNodeEvent)ev1 );
        }

        boolean exp = false;
        Boolean old2 = TreeTableNodeDefs.setExpandedOf(this, false);
        
        if(!Objects.equals(old2, (Boolean)exp)){
            TreeTableNodeCollapsed ev2 =new TreeTableNodeCollapsed(this,ev1);
            popup( (TreeNodeEvent)ev2 );
        }
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="expanded">
    @Override
    public boolean isExpanded(){
        Boolean ex = TreeTableNodeDefs.getExpandedOf(this);
        if( ex==null )return false;
        return (boolean)ex;
    }
    
    @Override
    public void setExpanded( boolean v ){
        Boolean old = TreeTableNodeDefs.setExpandedOf(this, v);
    }
    //</editor-fold>
    
    protected long rootScn = 0;
    
    /**
     * Возвращает номер изменения SCN корневого элемента дерева
     * @return номер изменения SCN
     */
    public long getRootScn(){
        TreeTableNodeBasic n = this;
        while(true){
            TreeTableNodeBasic np = n.parent;
            if( np==null ){
                return rootScn;
            }
            n = np;
        }
    }

    @Override
    public void popup(TreeNodeEvent<TreeTableNodeBasic> ev) {
        TreeTableNodeBasic parent = this.parent;
        if( parent==null ){
            if( ev instanceof TreeNodeAdded ||
                ev instanceof TreeNodeRemoved ||
                ev instanceof TreeNodeBulkInserted
            ){
                rootScn++;
            }
        }
            
        super.popup(ev);
    }
    
    //<editor-fold defaultstate="collapsed" desc="childrenSCN">
    protected long childrenSCN = 0;
    
    /**
     * Возвращает номер изменения SCN для дочерних узлов
     * @return номер SCN
     */
    public long getChildrenSCN(){
        return (Long)syncrun(new Func0() {
            @Override
            public Object apply() {
                return childrenSCN;
            }
        }, "getStructSCN");
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="collectionNotifier">
    protected class ChildrenCollectionListener
        extends SimpleListAdapter<TreeTableNodeBasic>
        implements BulkInsertListener<TreeTableNodeBasic>
    {
        @Override
        public void bulkInsertEvent(BulkInsertEvent<TreeTableNodeBasic> ev) {
            if( ev instanceof BulkInsertedEvent ){
                BulkInsertedEvent<TreeTableNodeBasic> bev = (BulkInsertedEvent)ev;
                
                List<TreeTableNodeBasic> itms = bev.getItems();
                int insIdx = bev.getInsertIndex();
                
                onTreeBulkInserted( insIdx, itms );
            }
        }
        
        @Override
        protected void added(TreeTableNodeBasic e, EventList<TreeTableNodeBasic> list, Integer position)
        {
            onTreeNodeAdded(position, e);
        }
        
        @Override
        protected void adding(TreeTableNodeBasic e, EventList<TreeTableNodeBasic> list, Integer position)
        {
            onTreeNodeAdding(position, e);
        }
        
        @Override
        protected void removed(TreeTableNodeBasic e, EventList<TreeTableNodeBasic> list, Integer position)
        {
            onTreeNodeRemoved(position, e);
        }
        
        @Override
        protected void removing(TreeTableNodeBasic e, EventList<TreeTableNodeBasic> list, Integer position)
        {
            onTreeNodeRemoving(position, e);
        }
    }
    
    protected final ChildrenCollectionListener collectionNotifier =
        new ChildrenCollectionListener();
    
    @Override
    protected void attachListeners( final EventList<TreeTableNodeBasic> elist ){
        syncrun( new Func0() {
            @Override
            public Object apply() {
                if( elist!=null ){
                    elist.addEventListListener(collectionNotifier, true);
                }
                return null;
            }}, "attachListeners", elist );
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="onTreeBulkInserted()">
    public void onTreeBulkInserted(final Integer index,final List<TreeTableNodeBasic> insertedChildren){
        syncrun(new Func0() {
            @Override
            public Object apply() {
                // Увеличение счетчика
                childrenSCN++;
                
                // Сброс кол-ва дочерних узлов
                resetNodesCount();
                
                // Установка parent
                for( TreeTableNodeBasic child : insertedChildren ){
                    if( child instanceof TreeNodeSetParent ){
                        TreeNodeSetParent<TreeNode> tn = (TreeNodeSetParent)child;
                        tn.setParent(TreeTableNodeBasic.this);
                    }
                }
                
                // Обновление ссылок prevSibling
                List<TreeTableNodeBasic> clist = childrenList;
                int ichild = -1;
                for( TreeTableNodeBasic child : insertedChildren ){
                    ichild++;
                    int childIndex = index + ichild;
                    
                    if( clist!=null && child!=null ){
                        child.updateIndex(childIndex, childrenSCN);
                        
                        TreeTableNodeBasic nextChild =
                            childIndex < (clist.size()-1)
                            ? clist.get(childIndex+1)
                            : null
                            ;
                        
                        TreeTableNodeBasic prevChild =
                            childIndex > 0
                            ? clist.get(childIndex-1)
                            : null
                            ;
                        
                        child.prevSibling = prevChild;
                        child.prevSiblingPSSCN = childrenSCN;
                        
                        if( nextChild!=null ){
                            nextChild.prevSibling = child;
                            nextChild.prevSiblingPSSCN = childrenSCN;
                        }
                    }
                }
                
                // Уведомление выше стоящих
                TreeNodeBulkInserted<TreeTableNodeBasic> ev =
                    new TreeNodeBulkInserted(
                        TreeTableNodeBasic.this,
                        TreeTableNodeBasic.this,
                        index,
                        insertedChildren
                    );
                
                ev.getPopupPath().add(TreeTableNodeBasic.this);
                
                popup( ev );
                
                return null;
            }
        }, "onTreeBulkInserted", index, insertedChildren );
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="onTreeNodeAdded()">
    @Override
    public void onTreeNodeAdded(final Integer index,final TreeTableNodeBasic child){
        syncrun(new Func0() {
            @Override
            public Object apply() {
                // Увеличение счетчика
                childrenSCN++;
                
                List<TreeTableNodeBasic> clist = childrenList;
                if( clist!=null && child!=null && index!=null ){
                    child.updateIndex(index, childrenSCN);
                    
                    TreeTableNodeBasic nextChild = 
                        index < (clist.size()-1)
                        ? clist.get(index+1)
                        : null
                        ;
                    
                    TreeTableNodeBasic prevChild = 
                        index > 0
                        ? clist.get(index-1)
                        : null
                        ;
                    
                    child.prevSibling = prevChild;
                    child.prevSiblingPSSCN = childrenSCN;
                    
                    if( nextChild!=null ){
                        nextChild.prevSibling = child;
                        nextChild.prevSiblingPSSCN = childrenSCN;
                    }
                }
                
                // Сброс кол-ва дочерних узлов
                resetNodesCount();
                
                // Установка parent
                if( child instanceof TreeNodeSetParent ){
                    TreeNodeSetParent<TreeNode> tn = (TreeNodeSetParent)child;
                    tn.setParent(TreeTableNodeBasic.this);
                }
                
                // Уведомление выше стоящих
                TreeNodeAdded<TreeTableNodeBasic> ev = new TreeNodeAdded(TreeTableNodeBasic.this,child,index);
                ev.getPopupPath().add(child);
                
                popup( ev );
                
                return null;
            }}, "onTreeNodeAdded", index, child );
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="onTreeNodeRemoved()">
    @Override
    public void onTreeNodeRemoved(final Integer index,final TreeTableNodeBasic child){
        syncrun(new Func0() {
            @Override
            public Object apply() {
                childrenSCN++;
                
                List<TreeTableNodeBasic> clist = childrenList;
                if( clist!=null && child!=null && index!=null ){
                    TreeTableNodeBasic nextChild = 
                        ((index+1) < clist.size()) && ((index+1) >= 0)
                        ? clist.get(index+1)
                        : null
                        ;
                    
                    TreeTableNodeBasic prevChild = 
                        index >= 0 && index< clist.size()
                        ? clist.get(index)
                        : null
                        ;
                    
                    child.prevSibling = null;

                    if( nextChild!=null ){
                        nextChild.prevSibling = prevChild;
                        nextChild.prevSiblingPSSCN = childrenSCN;
                    }
                    
                    /*child.prevSibling = prevChild;
                    child.prevSiblingPSSCN = childrenSCN;
                    
                    if( nextChild!=null ){
                        nextChild.prevSibling = child;
                        nextChild.prevSiblingPSSCN = childrenSCN;
                    }*/
                }

                resetNodesCount();
                
                // Обновление parent
                if( child instanceof TreeNodeUpdateParent ){
                    ((TreeNodeUpdateParent)child).updateParent(TreeTableNodeBasic.this, null);
                    
                    child.updateIndex(-1, childrenSCN);
                }
                
                // Уведомление выше стоящих
                TreeNodeRemoved<TreeTableNodeBasic> ev = new TreeNodeRemoved(TreeTableNodeBasic.this,child,index);
                ev.getPopupPath().add(child);
                popup( ev );
                
                return null;
            }}, "onTreeNodeRemoved", index, child );
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="stat">
    private static final Map<String,Number> stat = new LinkedHashMap<>();
    public static Map<String,Number> stat(){
        stat.clear();
        return stat;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="index">
    protected int index;
    protected long indexParentStuctSCN;
    
    protected void updateIndex( int idx, long psscn ){
        index = idx;
        indexParentStuctSCN = psscn;
    }
    
    @Override
    public int getIndex() {
        //return super.getIndex();
        /*
        return (Integer)syncrun(
            new Func0() {
                @Override
                public Object apply() {
                    return getIndex(TreeTableNodeBasic.this);
                }
            },
            "getIndex"
        );
        */
        
        TreeTableNodeBasic prnt = getParent();
        if( prnt!=null ){
            long pscn = prnt.getChildrenSCN();
            long cscn = indexParentStuctSCN;
            int idx = index;
            if( pscn == cscn && idx>=0 ){
                return idx;
            }else{
                index = getIndex(TreeTableNodeBasic.this);
                indexParentStuctSCN = pscn;
                return index;
            }
        }
        
        //return getIndex(TreeTableNodeBasic.this);
        return -1;
    }
    
    /**
     * Возвращает собственный дочерний индекс
     * @param thisNode Узел для которого возвращает индекс
     * @return Индекс или -1, если нет родительского элемента
     */
    public static int getIndex(TreeTableNodeBasic thisNode){
        if (thisNode== null) {
            throw new IllegalArgumentException("thisNode==null");
        }
        
        Object parent = thisNode.getParent();
        if( parent==null )return -1;
        if( !(parent instanceof TreeNode) )return -1;
        
        TreeNode parentTN = (TreeNode)parent;
        Object[] siblings = parentTN.getChildren();
        if( siblings==null )return -1;
        
        int idx = -1;
        for( idx=0; idx<siblings.length; idx++ ){
            Object sibling = siblings[idx];
            if( sibling!=null && sibling==thisNode )return idx;
        }
        
        return -1;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="Sibling">
    protected TreeTableNodeBasic prevSibling;
    protected long prevSiblingPSSCN;
    
    @Override
    public TreeTableNodeBasic getPreviousSibling() {
        TreeTableNodeBasic parent = this.parent;
        if( parent==null )return null;
        if( parent.childrenSCN != prevSiblingPSSCN ){
            TreeTableNodeBasic sib = getSibling(-1);
            prevSibling = sib;
            prevSiblingPSSCN = parent.childrenSCN;
            return sib;
        }
        return prevSibling;
        //return getSibling(-1);
    }
    
    @Override
    public TreeTableNodeBasic getSibling(int offset) {
        final TreeTableNodeBasic self = this;
        final int foffset = offset;
        
        return (TreeTableNodeBasic) syncrun(new Func0() {
            @Override
            public Object apply() {
                Object o1 = getSibling(self, foffset);
                //Object o2 = getSiblingDef(self, foffset);
                //if( !Objects.equals(o1, o2) ){
                //    logWarning("getSibling() has diff",foffset);
                //}
                return (TreeTableNodeBasic)o1;
            }
        }, "getSibling", offset);
    }
    
    /**
     * Возвращает соседний узел
     * @param thisNode Узел относительно которого происходит смещение
     * @param offset Смещение от узла
     * @return Узел или null в случаи достижения края
     */
    /*private static Object getSiblingDef(TreeTableNodeBasic thisNode, int offset){
        if (thisNode== null) {
            throw new IllegalArgumentException("thisNode==null");
        }
        
        if( offset==0 )return thisNode;
        
        TreeTableNodeBasic parent = thisNode.getParent();
        if( parent==null )return null;
        if( !(parent instanceof TreeNode) )return null;
        
        TreeTableNodeBasic parentTN = (TreeTableNodeBasic)parent;
        Object[] siblings = parentTN.getChildren();
        if( siblings==null )return null;
        
        int selfIdx = -1;
        int idx = -1;
        for( idx=0; idx<siblings.length; idx++ ){
            Object sibling = siblings[idx];
            if( sibling!=null && sibling==thisNode ){
                selfIdx = idx;
            }
        }
        
        if( selfIdx<0 )return null;
        
        int targetIdx = selfIdx + offset;
        if( targetIdx<0 )return null;
        if( targetIdx>=siblings.length )return null;
        return siblings[targetIdx];
    }*/

    /**
     * Возвращает соседний узел
     * @param thisNode Узел относительно которого происходит смещение
     * @param offset Смещение от узла
     * @return Узел или null в случаи достижения края
     */
    public static Object getSibling(TreeTableNodeBasic thisNode, int offset){
        if (thisNode== null) {
            throw new IllegalArgumentException("thisNode==null");
        }
        
        if( offset==0 )return thisNode;
        
        TreeTableNodeBasic parent = thisNode.getParent();
        if( parent==null )return null;
        
        List<TreeTableNodeBasic> sibs = parent.getChildrenList();
            
        if( sibs==null || sibs.isEmpty() )return null;
        
        int sibIdx = -1;
        int thisIdx = -1;
        for( TreeTableNodeBasic sib : sibs ){
            sibIdx++;
            if( sib==null )continue;
            if( sib==thisNode ){
                thisIdx = sibIdx;
                break;
            }
        }
        if( thisIdx<0 )return null;
        
        int targetIdx = thisIdx + offset;
        if( targetIdx<0 )return null;
        if( targetIdx<sibs.size() ){
            TreeTableNodeBasic target = sibs.get(targetIdx);
            return target;
        }
        return null;
    }
    //</editor-fold>
    
    /*
    @Override
    public int getRootOffset() {
        //return super.getRootOffset();
        Integer ro = (Integer) syncrun(new Func0() {
            @Override
            public Object apply() {
                return TreeTableNodeBasic.getRootOffsetOf( TreeTableNodeBasic.this, null );
            }
        }, "getRootOffset()");
        return ro;
    }
    */
    //</editor-fold>
}
